package fr.inra.agrosyst.services.pz0import.security;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: UserRoleImporter.java 5081 2015-09-02 10:40:56Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/security/UserRoleImporter.java $
 * %%
 * Copyright (C) 2013 - 2015 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Strings;
import com.google.common.collect.Maps;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.DomainTopiaDao;
import fr.inra.agrosyst.api.entities.GrowingPlan;
import fr.inra.agrosyst.api.entities.GrowingPlanTopiaDao;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.GrowingSystemTopiaDao;
import fr.inra.agrosyst.api.entities.Network;
import fr.inra.agrosyst.api.entities.NetworkTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefLocation;
import fr.inra.agrosyst.api.entities.referential.RefLocationTopiaDao;
import fr.inra.agrosyst.api.entities.security.AgrosystUser;
import fr.inra.agrosyst.api.entities.security.AgrosystUserTopiaDao;
import fr.inra.agrosyst.api.entities.security.RoleType;
import fr.inra.agrosyst.api.entities.security.UserRole;
import fr.inra.agrosyst.api.entities.security.UserRoleTopiaDao;
import fr.inra.agrosyst.api.services.plot.PlotService;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.domains.DomainAndDependencies;
import fr.inra.agrosyst.api.services.pz0.security.UsersRolesAndDependencies;
import fr.inra.agrosyst.api.services.security.UserRoleDto;
import fr.inra.agrosyst.api.services.security.UserRoleEntityDto;
import fr.inra.agrosyst.api.services.users.UserDto;
import fr.inra.agrosyst.api.services.users.Users;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.services.pz0import.AbstractCSVImporter;
import fr.inra.agrosyst.services.security.AuthorizationServiceImpl;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;

import java.io.InputStream;
import java.util.List;
import java.util.Map;


public class UserRoleImporter extends AbstractCSVImporter {

    protected PlotService plotService;

    protected UserRoleTopiaDao userRoleDao;
    protected AgrosystUserTopiaDao agrosystUserDao;
    protected NetworkTopiaDao networkDao;
    protected DomainTopiaDao domainDao;
    protected GrowingPlanTopiaDao growingPlanDao;
    protected GrowingSystemTopiaDao growingSystemDao;
    protected RefLocationTopiaDao locationDao;


    private static final Log log = LogFactory.getLog(UserRoleImporter.class);

    @Override
    public ImportResults importFromStream(InputStream is, Map<String, EntityAndDependencies> entityAndDependenciesByCsvId) {
        ImportResults importResults = new ImportResults(UserRole.class);

        UsersRolesAndDependencies usersRolesAndDependencies = new UsersRolesAndDependencies();
        importResults.addEntity("usersRolesImport", usersRolesAndDependencies);

        UserRoleImportModel model = new UserRoleImportModel();
        // récupère le DTO
        Import<UserRoleImportDto> importer = Import.newImport(model, is);

        // match the first csv line number with data (not header).
        long line = FIRST_LINE_NUMBER;

        for (UserRoleImportDto dto : importer) {
            boolean error;

            UserRoleDto userRoleDto;

            if (StringUtils.isNotBlank(dto.getId())) {
                Pair<Boolean, UserRoleDto> result = getUserRoleDto(importResults,  dto, line);
                if (result == null) {
                    importResults.addInfoLine(line, String.format("/!\\ RÔLE UTILISATEUR AVEC IDENTIFIANT'%s' IGNORÉ, le rôle existe déjà !", dto.getId()));
                    continue;
                }
                error = result.getLeft();
                userRoleDto = result.getValue();

                String userId = dto.getAgrosystUserId();
                if (StringUtils.isNotBlank(userId)) {
                    AgrosystUser agrosystUser = agrosystUserDao.forTopiaIdEquals(userId).findUniqueOrNull();
                    if (agrosystUser != null) {
                        UserDto userDto = Users.TO_USER_DTO.apply(agrosystUser);
                        userRoleDto.setUser(userDto);
                        usersRolesAndDependencies.addUserRole(userId, userRoleDto);
                    } else {
                        error = true;
                        importResults.addErrorLine(line, String.format("ÉCHEC D'AJOUT DU RÔLE UTILISATEUR dont l'identifiant est %s, l'utilisateur dont l'identifiant est '%s' n'existe pas", dto.getId(), userId));
                    }
                } else {
                    error = true;
                    importResults.addErrorLine(line, String.format("ÉCHEC D'AJOUT DU RÔLE UTILISATEUR dont l'identifiant est %s, la colonne 'agrosystuser' n'est pas renseignée", dto.getId()));
                }

                pz0IdToObject.put(UserRoleDto.class, dto.getId(), userRoleDto);
                if (!error) {
                    importResults.increaseAddedRecords();
                    importResults.addInfoLine(line, ", " + String.format("AJOUT DU ROLE UTILISATEUR, %s à l'utilisateur %s %s", userRoleDto.getType(), userRoleDto.getUser().getLastName(), userRoleDto.getUser().getFirstName()));
                } else {
                    importResults.increaseIgnoredRecords();
                    importResults.addErrorLine(line, String.format("ÉCHEC D'AJOUT DU RÔLE UTILISATEUR dont l'identifiant est '%s'", dto.getId()));
                }
            } else {
                importResults.addErrorLine(line, "ÉCHEC D'AJOUT DU RÔLE UTILISATEUR, Identifiant requis");
            }


            line++;
        }

        return importResults;
    }

    protected RoleType getRoleType(ImportResults importResults, UserRoleImportDto dto, long line) {
        RoleType roleType = null;
        if (StringUtils.isNotBlank(dto.getTypeName())) {
            try {
                roleType = RoleType.valueOf(dto.getTypeName());
            } catch (IllegalArgumentException e) {
                importResults.addErrorLine(line, String.format("ECHEC D'AJOUT DE TYPE DE ROLE, Le type de role avec comme nom %s n'existe pas.", dto.getTypeName()));
            }
        }
        return roleType;
    }

    protected Pair<Boolean, UserRoleDto> getUserRoleDto (ImportResults importResults,  UserRoleImportDto dto, long line) {
        Boolean error = false;
        UserRoleDto userRoleDto = new UserRoleDto();

        Pair<Boolean, UserRoleDto> result = null;

        Map<RoleType, Map<String, UserRole>> userRoleByType = getUserRoleByTypeAndIds(dto);

        String userRoleTopiaId = userRoleDao.forTopiaIdEquals(dto.getId()).exists() ? dto.getId() : null;
        userRoleDto.setTopiaId(userRoleTopiaId);

        RoleType roleType = getRoleType(importResults, dto, line);
        userRoleDto.setType(roleType);

        String identifier = null;
        String label = null;
        Integer campaign = null;

        boolean alreadyExistingRule = false;
        if (roleType != null) {
            if (!RoleType.IS_DATA_PROCESSOR.equals(roleType) && !RoleType.ADMIN.equals(roleType)) {
                UserRoleEntityDto entityDto = new UserRoleEntityDto();
                userRoleDto.setEntity(entityDto);
                switch (roleType) {
                    case NETWORK_RESPONSIBLE:
                        identifier = dto.getNetworkId();
                        if (StringUtils.isNotBlank(identifier)) {
                            if (userRoleByType.get(RoleType.NETWORK_RESPONSIBLE) != null && userRoleByType.get(RoleType.NETWORK_RESPONSIBLE).get(identifier) != null) {
                                alreadyExistingRule = true;
                            } else {
                                Network network = networkDao.forTopiaIdEquals(identifier).findUniqueOrNull();
                                if (network != null) {
                                    label = network.getName();
                                } else {
                                    error = true;
                                    importResults.addErrorLine(line, String.format("ÉCHEC D'AJOUT DU ROLE UTILISATEUR, l'identifiant réseau %s n'est pas reconnu, aucune entité retournée", identifier));
                                }
                            }
                        } else {
                            error = true;
                            importResults.addErrorLine(line, "ÉCHEC D'AJOUT DU ROLE UTILISATEUR, le type de rôle 'Responsable réseau' nécessite un identifiant de réseau, la colonne 'networkid' n'est pas renseignée");
                        }
                        break;
                    case DOMAIN_RESPONSIBLE:
                        identifier = dto.getDomainCode();
                        if (StringUtils.isNotBlank(identifier)) {
                            if (userRoleByType.get(RoleType.DOMAIN_RESPONSIBLE) != null && userRoleByType.get(RoleType.DOMAIN_RESPONSIBLE).get(identifier) != null) {
                                alreadyExistingRule = true;
                            } else {
                                List<DomainAndDependencies> domainsAndDependencies = (List<DomainAndDependencies>) pz0CodeToObject.get(Domain.class, identifier);
                                if (CollectionUtils.isNotEmpty(domainsAndDependencies)) {
                                    DomainAndDependencies domainAndDependencies = domainsAndDependencies.get(0);
                                    if (domainAndDependencies != null) {
                                        Domain domain = domainAndDependencies.getDomain();
                                        identifier = domain.getCode();
                                        String refLocationId = domainAndDependencies.getLocationId();
                                        RefLocation location = null;
                                        if (StringUtils.isNotBlank(refLocationId)) {
                                            location = locationDao.forTopiaIdEquals(refLocationId).findUniqueOrNull();
                                        }
                                        String postCode = location != null ? location.getCodePostal() : "";
                                        label = String.format("%s (%s)", domain.getName(), postCode);
                                    } else {
                                        error = true;
                                        importResults.addErrorLine(line, String.format("ÉCHEC D'AJOUT DU ROLE UTILISATEUR, aucun domain retrouvé avec l'identifiant %s", identifier));
                                    }
                                }
                            }

                        } else {
                            error = true;
                            importResults.addErrorLine(line, "ÉCHEC D'AJOUT DU ROLE UTILISATEUR, le type de rôle 'Responsables du domaine' nécessite un code de domaine, la colonne 'domaincode' n'est pas renseignée");
                        }

                        break;
                    case GROWING_PLAN_RESPONSIBLE:
                        identifier = dto.getGrowingPlanCode();
                        if (StringUtils.isNotBlank(identifier)) {
                            if (userRoleByType.get(RoleType.GROWING_PLAN_RESPONSIBLE) != null && userRoleByType.get(RoleType.GROWING_PLAN_RESPONSIBLE).get(identifier) != null) {
                                alreadyExistingRule = true;
                            } else {
                                List<GrowingPlan> growingPlans = (List<GrowingPlan>) pz0CodeToObject.get(GrowingPlan.class, identifier);
                                if (CollectionUtils.isNotEmpty(growingPlans)) {
                                    GrowingPlan growingPlan = growingPlans.get(0);
                                    identifier = growingPlan.getCode();
                                    label = AuthorizationServiceImpl.GET_GROWING_PLAN_ENTITY_LABEL.apply(growingPlan);
                                } else {
                                    error = true;
                                    importResults.addErrorLine(line, String.format("ÉCHEC D'AJOUT DU ROLE UTILISATEUR, aucun dispositif retrouvé avec l'identifiant %s", identifier));
                                }
                            }
                        } else {
                            error = true;
                            importResults.addErrorLine(line, "ÉCHEC D'AJOUT DU ROLE UTILISATEUR, le type de rôle 'Responsable dispositif' nécessite un code de dispositif, la colonne 'growingplancode' n'est pas renseignée");
                        }

                        break;
                    case GS_DATA_PROCESSOR:
                        GrowingSystem growingSystem = null;
                        if (StringUtils.isNotBlank(dto.getGrowingSystemId()) ||StringUtils.isNotBlank(dto.getGrowingSystemCode())) {
                            if (!Strings.isNullOrEmpty(dto.getGrowingSystemId())) {
                                growingSystem = (GrowingSystem) pz0IdToObject.get(GrowingSystem.class, dto.getGrowingSystemId());

                                if (growingSystem != null) {
                                    campaign = growingSystem.getGrowingPlan().getDomain().getCampaign();
                                } else {
                                    error = true;
                                    importResults.addErrorLine(line, String.format("ÉCHEC D'AJOUT DU ROLE UTILISATEUR, le système de culture dont l'identifiant est %s n'existe pas", dto.getGrowingSystemId()));
                                }
                            } else {
                                List<GrowingSystem> growingSystems = (List<GrowingSystem>) pz0CodeToObject.get(GrowingSystem.class, dto.getGrowingSystemCode());

                                if (CollectionUtils.isNotEmpty(growingSystems)) {
                                    growingSystem = growingSystems.get(0);
                                    if (growingSystem == null) {
                                        error = true;
                                        importResults.addErrorLine(line, String.format("ÉCHEC D'AJOUT DU ROLE UTILISATEUR, le système de culture dont le code est %s n'existe pas", dto.getGrowingSystemCode()));
                                    }
                                }
                            }
                            if (growingSystem != null) {
                                identifier = growingSystem.getCode();
                                if (userRoleByType.get(RoleType.GS_DATA_PROCESSOR) != null && userRoleByType.get(RoleType.GS_DATA_PROCESSOR).get(identifier) != null) {
                                    alreadyExistingRule = true;
                                } else {
                                    label = AuthorizationServiceImpl.GET_GROWING_SYSTEM_ENTITY_LABEL.apply(growingSystem);
                                }
                            }
                        } else {
                            error = true;
                            importResults.addErrorLine(line, "ÉCHEC D'AJOUT DU ROLE UTILISATEUR, le type de rôle 'Exploitant SdC' nécessite un identifiant de système de culture de renseigné or les colonnes 'growingsystemcode' ou 'growingsystemid' ne sont pas renseignées");
                        }

                        break;
                    default:
                        error = true;
                        identifier = null;
                        label = null;
                        break;
                }
                entityDto.setIdentifier(identifier);
                entityDto.setLabel(label);
                entityDto.setCampaign(campaign);
            }
        } else {
            error = true;
            importResults.addErrorLine(line, "ÉCHEC D'AJOUT DU ROLE UTILISATEUR, le type de rôle" );
        }

        if (!alreadyExistingRule) {
            result = Pair.of(error, userRoleDto);
        }
        return result;
    }

    protected Map<RoleType, Map<String, UserRole>> getUserRoleByTypeAndIds(UserRoleImportDto dto) {
        Map<RoleType, Map<String, UserRole>> userRoleByType = Maps.newHashMap();
        String userId = dto.getAgrosystUserId();
        if (StringUtils.isNotBlank(userId)) {
            AgrosystUser agrosystUser = agrosystUserDao.forTopiaIdEquals(userId).findUniqueOrNull();
            if (agrosystUser != null) {
                List<UserRole> roles = userRoleDao.findAllForUserId(userId);
                for (UserRole role : roles) {
                    RoleType type = role.getType();
                    String identifier = null;
                    switch (type) {
                        case ADMIN:
                        case IS_DATA_PROCESSOR:
                            break;
                        case NETWORK_RESPONSIBLE:
                            identifier = role.getNetworkId();
                            break;
                        case DOMAIN_RESPONSIBLE:
                            identifier = role.getDomainCode();
                            break;
                        case GROWING_PLAN_RESPONSIBLE:
                            identifier = role.getGrowingPlanCode();
                            break;
                        case GS_DATA_PROCESSOR:
                            identifier = role.getGrowingSystemId();
                            break;
                    }

                    if (identifier != null) {
                        Map<String, UserRole> userRolesForCurrentType = userRoleByType.get(type);
                        if (userRolesForCurrentType == null) {
                            userRolesForCurrentType = Maps.newHashMap();
                            userRoleByType.put(type, userRolesForCurrentType);
                        }
                        userRolesForCurrentType.put(identifier, role);
                    }
                }
            }
        }
        return userRoleByType;
    }

    @Override
    public void init(ServiceContext serviceContext) {
        super.init(serviceContext);

        plotService = getServiceFactory().newService(PlotService.class);

        userRoleDao = getPersistenceContext().getUserRoleDao();
        agrosystUserDao = getPersistenceContext().getAgrosystUserDao();
        networkDao = getPersistenceContext().getNetworkDao();
        domainDao = getPersistenceContext().getDomainDao();
        growingPlanDao = getPersistenceContext().getGrowingPlanDao();
        growingSystemDao = getPersistenceContext().getGrowingSystemDao();
        locationDao = getPersistenceContext().getRefLocationDao();
    }


}
