package fr.inra.agrosyst.services.pz0import.practicedSystem.practicedIntervention;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: PracticedInterventionImporter.java 5073 2015-08-27 16:10:55Z eancelet $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/practicedSystem/practicedIntervention/PracticedInterventionImporter.java $
 * %%
 * Copyright (C) 2013 - 2015 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.MaterielTransportUnit;
import fr.inra.agrosyst.api.entities.MaterielWorkRateUnit;
import fr.inra.agrosyst.api.entities.practiced.PracticedIntervention;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystem;
import fr.inra.agrosyst.api.services.domain.CroppingPlanEntryDto;
import fr.inra.agrosyst.api.services.domain.CroppingPlanSpeciesDto;
import fr.inra.agrosyst.api.services.itk.SpeciesStadeDto;
import fr.inra.agrosyst.api.services.practiced.PracticedCropCycleConnectionDto;
import fr.inra.agrosyst.api.services.practiced.PracticedCropCyclePhaseDto;
import fr.inra.agrosyst.api.services.practiced.PracticedInterventionDto;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.domains.DomainAndDependencies;
import fr.inra.agrosyst.api.services.pz0.practicedSystem.PracticedSystemAndDependencies;
import fr.inra.agrosyst.api.services.pz0.practicedSystem.Pz0PracticedCropCycleConnection;
import fr.inra.agrosyst.api.services.pz0.practicedSystem.Pz0PracticedCropCycleNode;
import fr.inra.agrosyst.api.services.pz0.practicedSystem.Pz0PracticedCropCyclePhase;
import fr.inra.agrosyst.api.services.pz0.practicedSystem.Pz0PracticedIntervention;
import fr.inra.agrosyst.api.services.pz0.practicedSystem.Pz0PracticedPerennialCropCycle;
import fr.inra.agrosyst.api.services.pz0.practicedSystem.Pz0PracticedSeasonalCropCycle;
import fr.inra.agrosyst.services.pz0import.AbstractCSVImporter;
import fr.inra.agrosyst.services.pz0import.practicedSystem.PracticedSystemImporter;
import fr.inra.agrosyst.services.pz0import.practicedSystem.practicedCropCycleNode.PracticedCropCycleNodeImporter;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.InputStream;
import java.util.Collection;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Created by davidcosse on 05/02/15.
 */
public class PracticedInterventionImporter extends AbstractCSVImporter {

    private static final Log log = LogFactory.getLog(PracticedInterventionImporter.class);
    protected static final Pattern DATE_PATTERN = Pattern.compile("^(0?[1-9]|[12][0-9]|3[01])/(0?[1-9]|1[012])");

    @Override
    public ImportResults importFromStream(InputStream is, Map<String, EntityAndDependencies> entitiesByCsvId) {
        ImportResults importResults = new ImportResults(PracticedIntervention.class);

        PracticedInterventionImportModel model = new PracticedInterventionImportModel();
        // récupère le DTO
        Import<PracticedInterventionImportDto> importer = Import.newImport(model, is);

        // match the first csv line number with data (not header).
        long line = FIRST_LINE_NUMBER;
        for (PracticedInterventionImportDto dto : importer) {
            boolean error;

            Pz0PracticedIntervention pz0Intervention = new Pz0PracticedIntervention(dto.getId(), dto.getPracticedCropCycleConnectionId(), dto.getPracticedCropCyclePhaseId());
            PracticedInterventionDto interventionDto = pz0Intervention.getPracticedInterventionDto();
            Binder<PracticedInterventionImportDto, PracticedInterventionDto> binder = BinderFactory.newBinder(PracticedInterventionImportDto.class, PracticedInterventionDto.class);
            binder.copy(dto, interventionDto);

            // valid required related objets
            error = validPhaseOrConnectionFields(importResults, line, dto, false);
            error = validPhase(importResults, line, dto, error);
            error = validConnection(importResults, line, dto, error);
            error = addInterventionToPhaseOrConnection(importResults, line, dto, pz0Intervention, error);

            // valid required field
            error = validName(importResults, line, interventionDto, error);
            error = validRank(importResults, line, dto ,interventionDto, error);
            error = validStatingPeriod(importResults, line, interventionDto, error);
            error = validEndingPeriod(importResults, line, interventionDto, error);

            // valid required Enums
            error = validType(importResults, line, dto, interventionDto, error);

            // valid optional Enums
            error = validTransitVolumeUnit(importResults, line, dto, interventionDto, error);
            error = validWorkRateUnit(importResults, line, dto, interventionDto, error);

            // other validation
            error = validIntermediateCropStatus(importResults, line, dto, pz0Intervention, error);

            pz0IdToObject.put(Pz0PracticedIntervention.class, dto.getId(), pz0Intervention);

            createSpeciesStadesToIntervention(pz0Intervention);
            if (!error) {
                importResults.addInfoLine(line, ", " + "INTERVENTION DU SYNTHÉTISÉ VALIDÉE, csvid: " + dto.getId());
                importResults.increaseAddedRecords();
            } else {
                importResults.increaseIgnoredRecords();
                importResults.addErrorLine(line, "INTERVENTION DU SYNTHÉTISÉ IGNORÉE csvid:" + dto.getId());
            }
            line++;
        }
        return importResults;
    }

    protected void createSpeciesStadesToIntervention(Pz0PracticedIntervention pz0Intervention) {
        String practicedSystemId = pz0Intervention.getPracticedSystemId();
        if (StringUtils.isNotBlank(practicedSystemId) && StringUtils.isNotBlank(pz0Intervention.getCropCode())) {
            PracticedInterventionDto practicedInterventionDto = pz0Intervention.getPracticedInterventionDto();
            PracticedSystemAndDependencies practicedSystemAndDependencies = (PracticedSystemAndDependencies) pz0IdToObject.get(PracticedSystem.class, practicedSystemId);
            if (practicedSystemAndDependencies != null) {

                PracticedSystem practicedSystem = practicedSystemAndDependencies.getPracticedSystem();
                if (practicedSystem != null) {
                    DomainAndDependencies domainAndDependencies = PracticedSystemImporter.getDomainAndDependenciesFromPracticedSystem(practicedSystem);

                    if (domainAndDependencies != null) {
                        Collection<CroppingPlanEntryDto> croppingPlanEntryDtos = domainAndDependencies.getCroppingPlanEntryDtosByPZ0Code().values();

                        if (CollectionUtils.isNotEmpty(croppingPlanEntryDtos)) {
                            if (StringUtils.isNotBlank(pz0Intervention.getCropCode())) {
                                String pz0CropCode = domainAndDependencies.getCroppingPlanEntriesAgrosystCodeToCsvCode().get(pz0Intervention.getCropCode());
                                if (StringUtils.isNotBlank(pz0CropCode)) {
                                    CroppingPlanEntryDto cpe = domainAndDependencies.getCroppingPlanEntryDtosByPZ0Code().get(pz0CropCode);
                                    if (cpe != null) {
                                        Collection<CroppingPlanSpeciesDto> interventionSpecies = cpe.getSpecies();
                                        if (CollectionUtils.isNotEmpty(interventionSpecies)) {
                                            for (CroppingPlanSpeciesDto interventionSpecy : interventionSpecies) {
                                                SpeciesStadeDto speciesStade = new SpeciesStadeDto();
                                                String agrosystCropCode = interventionSpecy.getCode();
                                                speciesStade.setSpeciesCode(agrosystCropCode);
                                                practicedInterventionDto.addSpeciesStadeDto(speciesStade);
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    protected boolean validIntermediateCropStatus(ImportResults importResults, long line, PracticedInterventionImportDto from, Pz0PracticedIntervention pz0Intervention, boolean error){
        if (from.isIntermediateCrop()) {
            if (StringUtils.isBlank(from.getPracticedCropCyclePhaseId())) {
                if (StringUtils.isNotBlank(from.getPracticedCropCycleConnectionId())) {
                    Pz0PracticedCropCycleConnection  pz0Connection = (Pz0PracticedCropCycleConnection)pz0IdToObject.get(Pz0PracticedCropCycleConnection.class, from.getPracticedCropCycleConnectionId());
                    if(pz0Connection != null) {
                        PracticedCropCycleConnectionDto practicedCropCycleConnectionDto = pz0Connection.getPracticedCropCycleConnectionDto();
                        if (StringUtils.isBlank(practicedCropCycleConnectionDto.getIntermediateCroppingPlanEntryCode())) {
                            importResults.addErrorLine(line, "INTERVENTION IGNORÉE!, l'intervention est affectée à une culture intermédiaire or aucune culture intermédiaire n'est présente.");
                            error = true;
                        }
                    }
                }
            } else {
                importResults.addErrorLine(line, "INTERVENTION IGNORÉE!, une intervention portant sur une culture intermédiaire ne peut-être affectée à une phase.");
                error = true;
            }

        } else if (StringUtils.isBlank(pz0Intervention.getCropCode())) {
            importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, aucune culture avec pour code '%s' ne fait partie des cultures principales du domaine.", pz0Intervention.getCropCode()));
            error = true;
        }
        return error;
    }

    protected boolean validPhaseOrConnectionFields(ImportResults importResults, long line, PracticedInterventionImportDto from, boolean error) {
        String practicedCropCyclePhaseId = from.getPracticedCropCyclePhaseId();
        String practicedCropCycleConnectionId = from.getPracticedCropCycleConnectionId();
        if(StringUtils.isBlank(practicedCropCyclePhaseId) && StringUtils.isBlank(practicedCropCycleConnectionId)){
            importResults.addErrorLine(line, "INTERVENTION IGNORÉE!, une des colonnes suivante doit-être renseigné: 'practicedcropcyclephase' ou 'practicedcropcycleconnection' mais aucune ne l'est.");
            error = true;
        }
        return error;
    }

    protected boolean validPhase(ImportResults importResults, long line, PracticedInterventionImportDto from, boolean error) {
        String phaseId = from.getPracticedCropCyclePhaseId();
        if (!error && StringUtils.isNotBlank(phaseId)) {
            Pz0PracticedCropCyclePhase pz0Phase = (Pz0PracticedCropCyclePhase)pz0IdToObject.get(Pz0PracticedCropCyclePhase.class, phaseId);
            if(pz0Phase == null) {
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, la phase avec comme id %s n'esiste pas.", phaseId));
                error = true;
            }

        }
        return error;
    }

    protected boolean validConnection(ImportResults importResults, long line, PracticedInterventionImportDto from, boolean error) {
        String connectionId = from.getPracticedCropCycleConnectionId();
        if (!error && StringUtils.isNotBlank(connectionId)) {
            Pz0PracticedCropCycleConnection  pz0Connection = (Pz0PracticedCropCycleConnection)pz0IdToObject.get(Pz0PracticedCropCycleConnection.class, connectionId);
            if(pz0Connection == null) {
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, la connexion avec comme id '%s' n'esiste pas.", connectionId));
                error = true;
            }
        }
        return error;
    }

    protected boolean addInterventionToPhaseOrConnection(ImportResults importResults, long line, PracticedInterventionImportDto from, Pz0PracticedIntervention pz0Intervention, boolean error) {
        String connectionId = from.getPracticedCropCycleConnectionId();
        String phaseId = from.getPracticedCropCyclePhaseId();
        if (StringUtils.isNotBlank(connectionId)) {
            Pz0PracticedCropCycleConnection  pz0Connection = (Pz0PracticedCropCycleConnection)pz0IdToObject.get(Pz0PracticedCropCycleConnection.class, connectionId);
            if (pz0Connection != null) {
                pz0Connection.addPracticedPz0Intervention(pz0Intervention);
                PracticedCropCycleConnectionDto connectionDto = pz0Connection.getPracticedCropCycleConnectionDto();
                String cropCode = getSeasonalCropCode(pz0Intervention, connectionDto);
                pz0Intervention.setCropCode(cropCode);

                String practicedSystemId = getPracticedSystemIdFromPz0Connexion(pz0Connection);
                pz0Intervention.setPracticedSystemId(practicedSystemId);

            } else {
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, la connexion avec comme id %s n'esiste pas.", connectionId));
                error = true;
            }
            pz0IdToObject.put(PracticedInterventionDto.class, PracticedCropCycleConnectionDto.class, from.getId(), connectionId);
        } else if (StringUtils.isNotBlank(phaseId)){
            Pz0PracticedCropCyclePhase  pz0Phase = (Pz0PracticedCropCyclePhase)pz0IdToObject.get(Pz0PracticedCropCyclePhase.class, phaseId);
            if (pz0Phase != null) {
                Pz0PracticedPerennialCropCycle pz0PerennialCycle = (Pz0PracticedPerennialCropCycle) pz0IdToObject.get(Pz0PracticedPerennialCropCycle.class, pz0Phase.getPracticedPerennialCropCycleId());
                if (pz0PerennialCycle != null) {
                    String agrosystCropCode = pz0CodeToAgrosystCode.get(CroppingPlanEntryDto.class, pz0PerennialCycle.getPz0CropCode());
                    if (StringUtils.isEmpty(agrosystCropCode)) {
                        error = true;
                        importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, il n'est pas possible d'attribuer un code de culture à l'intervention %s.", from.getId()));
                    } else {
                        pz0Intervention.setCropCode(agrosystCropCode);
                    }
                }                
                pz0Phase.addPracticedPz0Intervention(pz0Intervention);
                String practicedSystemId = getPracticedSystemPz0IdFromPz0Phase(pz0Phase);
                pz0Intervention.setPracticedSystemId(practicedSystemId);

            } else {
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, la phase avec comme id %s n'esiste pas.", phaseId));
                error = true;
            }
            pz0IdToObject.put(PracticedInterventionDto.class, PracticedCropCyclePhaseDto.class, from.getId(), phaseId);
        }
        return error;
    }

    protected String getPracticedSystemIdFromPz0Connexion(Pz0PracticedCropCycleConnection pz0Connection) {
        String practicedSystemId = null;
        String nodoPz0Id = pz0Connection.getPz0NodeId();
        if (StringUtils.isNoneBlank(nodoPz0Id)) {
            Pz0PracticedCropCycleNode pz0None = (Pz0PracticedCropCycleNode) pz0IdToObject.get(Pz0PracticedCropCycleNode.class,  nodoPz0Id);
            if (pz0None != null && StringUtils.isNotBlank(pz0None.getPracticedSeasonalCropCycleId())) {
                String cycleId = pz0None.getPracticedSeasonalCropCycleId();
                Pz0PracticedSeasonalCropCycle pz0Cycle = (Pz0PracticedSeasonalCropCycle) pz0IdToObject.get(Pz0PracticedSeasonalCropCycle.class, cycleId);
                if (pz0Cycle != null) {
                    practicedSystemId = pz0Cycle.getPracticedSystemId();
                }
            }
        }

        return practicedSystemId;
    }

    protected String getPracticedSystemPz0IdFromPz0Phase(Pz0PracticedCropCyclePhase pz0Phase) {
        String practicedSystemId = null;
        String cycleId = pz0Phase.getPracticedPerennialCropCycleId();
        if (StringUtils.isNotBlank(cycleId)) {
            Pz0PracticedPerennialCropCycle cycle = (Pz0PracticedPerennialCropCycle) pz0IdToObject.get(Pz0PracticedPerennialCropCycle.class, cycleId);
            practicedSystemId = cycle != null ? cycle.getPracticedSystemId() : null;
        }
        return practicedSystemId;
    }

    protected String getSeasonalCropCode(Pz0PracticedIntervention pz0Intervention, PracticedCropCycleConnectionDto connectionDto) {
        String cropCode;
        if (pz0Intervention.getPracticedInterventionDto().isIntermediateCrop()) {
            cropCode = connectionDto.getIntermediateCroppingPlanEntryCode();
        } else {
            String csvNodeId = StringUtils.remove(connectionDto.getTargetId(), PracticedCropCycleNodeImporter.NEW_NODE);
            Pz0PracticedCropCycleNode node = (Pz0PracticedCropCycleNode) pz0IdToObject.get(Pz0PracticedCropCycleNode.class, csvNodeId);
            cropCode = node.getNodeDto().getCroppingPlanEntryCode();
        }
        return cropCode;
    }

    protected boolean validType(ImportResults importResults, long line, PracticedInterventionImportDto from, PracticedInterventionDto to, boolean error) {
        String typeName = from.getTypeName();
        if(StringUtils.isNotBlank(typeName)){
            try {
                AgrosystInterventionType type = AgrosystInterventionType.valueOf(typeName);
                to.setType(type);
            } catch (IllegalArgumentException e) {
                error = true;
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, le type d'intervention %s n'est pas reconnu.", typeName));
            }
        } else {
            error = true;
            importResults.addErrorLine(line, "INTERVENTION IGNORÉE!, la colonne 'type' n'est pas renseigné.");

        }
        return error;
    }

    protected boolean validRank(ImportResults importResults, long line, PracticedInterventionImportDto from, PracticedInterventionDto to, boolean error) {
        Integer rank = from.getRankNumber();
        if (rank != null) {
            to.setRank(rank);
        } else {
            error = true;
            importResults.addErrorLine(line, "INTERVENTION IGNORÉE!, la colonne 'rank' n'est pas renseigné.");
        }
        return error;
    }

    protected boolean validName(ImportResults importResults, long line, PracticedInterventionDto to, boolean error) {
        String name = to.getName();
        if (StringUtils.isBlank(name)) {
            error = true;
            importResults.addErrorLine(line, "INTERVENTION IGNORÉE!, la colonne 'rank' n'est pas renseigné.");
        }
        return error;
    }

    protected boolean validStatingPeriod(ImportResults importResults, long line, PracticedInterventionDto to, boolean error) {
        String startingPeriodDate = to.getStartingPeriodDate();
        if (StringUtils.isNotBlank(startingPeriodDate)) {
            Matcher matcher = DATE_PATTERN.matcher(startingPeriodDate);
            if (!matcher.matches()) {
                error = true;
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, la date d'intervention ou date de début %s n'est pas reconnue.", startingPeriodDate));
            }
        } else {
            error = true;
            importResults.addErrorLine(line, "INTERVENTION IGNORÉE!, la colonne 'startingperioddate' n'est pas renseigné.");
        }
        return error;
    }

    protected boolean validEndingPeriod(ImportResults importResults, long line, PracticedInterventionDto to, boolean error) {
        String endingPeriodDate = to.getEndingPeriodDate();
        if (StringUtils.isNotBlank(endingPeriodDate)) {
            Matcher matcher = DATE_PATTERN.matcher(endingPeriodDate);
            if (!matcher.matches()) {
                error = true;
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, la date de fin d'intervention %s n'est pas reconnue.", endingPeriodDate));
            }
        } else if (to.getStartingPeriodDate() != null) {
            to.setEndingPeriodDate(to.getStartingPeriodDate());
        } else {
            error = true;
            importResults.addErrorLine(line, "INTERVENTION IGNORÉE!, la colonne 'endingperioddate' n'est pas renseigné.");
        }
        return error;
    }

    protected boolean validTransitVolumeUnit(ImportResults importResults, long line, PracticedInterventionImportDto from, PracticedInterventionDto to, boolean error) {
        String transitVolumeName = from.getTransitVolumeUnitName();
        if (StringUtils.isNotBlank(transitVolumeName)) {
            try {
                MaterielTransportUnit type = MaterielTransportUnit.valueOf(transitVolumeName);
                to.setTransitVolumeUnit(type);
            } catch (IllegalArgumentException e) {
                error = true;
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, l'unité de volume par voyage %s n'est pas reconnu.", transitVolumeName));
            }
        }
        return error;
    }

    protected boolean validWorkRateUnit(ImportResults importResults, long line, PracticedInterventionImportDto from, PracticedInterventionDto to, boolean error) {
        String workRateUnitName = from.getWorkRateUnitName();

        if (StringUtils.isNotBlank(workRateUnitName)) {
            try {
                MaterielWorkRateUnit type = MaterielWorkRateUnit.valueOf(workRateUnitName);
                to.setWorkRateUnit(type);
            } catch (IllegalArgumentException e) {
                error = true;
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, l'unité de temps de travail %s n'est pas reconnue.", workRateUnitName));
            }
        } else {
            if (from.getWorkRate() != null) {
                error = true;
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, aucune unité de temps de travail %s n'est pas renseignée alors que le temps de travail '%f' est renseigné.", workRateUnitName, from.getWorkRate()));
            }
        }
        return error;
    }


}
