package fr.inra.agrosyst.services.pz0import.growingSystem;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: GrowingSystemImporter.java 5107 2015-10-21 13:07:49Z eancelet $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/growingSystem/GrowingSystemImporter.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.collect.Lists;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.DomainTopiaDao;
import fr.inra.agrosyst.api.entities.GrowingPlan;
import fr.inra.agrosyst.api.entities.GrowingPlanTopiaDao;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.Sector;
import fr.inra.agrosyst.api.entities.managementmode.CategoryStrategy;
import fr.inra.agrosyst.api.entities.referential.RefTypeAgriculture;
import fr.inra.agrosyst.api.entities.referential.RefTypeAgricultureTopiaDao;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemService;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.domains.DomainAndDependencies;
import fr.inra.agrosyst.api.services.pz0.growingSystem.GrowingSystemAndDependencies;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.services.pz0import.AbstractCSVImporter;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.keyvalue.MultiKey;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;
import java.io.InputStream;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
*
* @author eancelet
* 
* TODO le cas échéant :
* * * * ajout des characteritics_growingsystem
* 
* 
*/


public class GrowingSystemImporter extends AbstractCSVImporter {

    private static final Log log = LogFactory.getLog(GrowingSystemImporter.class);

    protected GrowingSystemService growingSystemService;
    protected DomainTopiaDao domainDao;
    protected GrowingPlanTopiaDao growingPlanDao;
    protected RefTypeAgricultureTopiaDao refTypeAgricultureDao;
    
    @Override
    public ImportResults importFromStream(InputStream is, Map<String, EntityAndDependencies> growingPlansByCsvIds) {
        ImportResults importResults = new ImportResults(GrowingSystem.class);

        GrowingSystemImportModel model = new GrowingSystemImportModel();

        // récupère le DTO
        Import<GrowingSystemImportDto> importer = Import.newImport(model, is);

        // match the csv line number.
        long line = FIRST_LINE_NUMBER;
        for (GrowingSystemImportDto dto : importer) {
            boolean error;
            GrowingSystem growingSystem = growingSystemService.newGrowingSystem();

            Binder<GrowingSystemImportDto, GrowingSystem> growingSystemBinder = BinderFactory.newBinder(GrowingSystemImportDto.class, GrowingSystem.class);
            growingSystemBinder.copyExcluding(dto, growingSystem, GrowingSystem.PROPERTY_TOPIA_ID, GrowingSystem.PROPERTY_CODE);

            // valid required fields
            error = validGrowingPlanAffiliation(importResults, line, dto, growingSystem, false);
            error = validGrowingSystemName(importResults, line, growingSystem, error);
            error = validGrowingSystemUpdateDate(importResults, line, growingSystem, error);
            error = validGrowingSystemSector(importResults, dto, line, growingSystem, error);
            error = validStartingDate(importResults, dto, line, growingSystem.getStartingDate(), error);

            error = addGrowingSystemCategoryStrategy(importResults, dto, line, growingSystem, error);

            // Type Agriculture
            String refTypeAgricultureId = getTypeAgricultureId(importResults, line, dto, growingSystem);
            
            if (!error) {
                error = validDephyNumberAffiliation(importResults, line, dto, growingSystem, error);
            }
            error = excludeDupplicates(importResults, line, dto.getId(), error);
            error = validGrowingSystemCode(importResults, line, dto, growingSystem, error);

            pz0IdToObject.put(GrowingSystem.class, dto.getId(), growingSystem);

            List<GrowingSystem> growingSystems = (List<GrowingSystem>) pz0CodeToObject.get(GrowingSystem.class, dto.getCode());
            if (growingSystems == null) {
                growingSystems = Lists.newArrayList();
                pz0CodeToObject.put(GrowingSystem.class, dto.getCode(), growingSystems);
            }
            growingSystems.add(growingSystem);

            if (!error) {
                importResults.addInfoLine(line, ", " + "SDC VALIDÉ, csvid: " + dto.getId());
                importResults.increaseAddedRecords();
            } else {
                importResults.increaseIgnoredRecords();
                importResults.addErrorLine(line, "SDC IGNORÉ csvid:" + dto.getId());
            }
            GrowingSystemAndDependencies growingSystemAndDependencies = new GrowingSystemAndDependencies(growingSystem, refTypeAgricultureId, dto.getGrowingPlanId());
            importResults.addEntity(dto.getId(), growingSystemAndDependencies);

            line++;
        }


        return importResults;
    }

    protected boolean excludeDupplicates(ImportResults importResults, long line, String csvId, boolean error) {
        if (pz0IdToObject.get(GrowingSystem.class, csvId) != null) {
            error = true;
            importResults.addErrorLine(line, String.format("SdC IGNORÉ ! Le topiaId %s est déjà présent dans le fichier !", csvId));
        }
        return error;
    }

    protected boolean validStartingDate(ImportResults importResults, GrowingSystemImportDto dto, long line, Date startingDate, boolean error) {
        if (startingDate == null) {
            error = true;
            importResults.addErrorLine(line, String.format("SDC IGNORÉ!, le SdC de topiaId %s n'a pas de date de début",dto.getId()));
        }
        return error;
    }

    protected String getTypeAgricultureId(ImportResults importResults, long line, GrowingSystemImportDto dto, GrowingSystem growingSystem) {
        String refTypeAgricultureId = null;
        if (StringUtils.isNotBlank(dto.getTypeAgricultureId())) {
            RefTypeAgriculture refTypeAgriculture = refTypeAgricultureDao.forTopiaIdEquals(dto.getTypeAgricultureId()).findUniqueOrNull();
            if (refTypeAgriculture==null) {
                importResults.addInfoLine(line, ", " + String.format("ATTENTION ! le Type d'agriculture de topiaId %s n'existe pas en base de données pour le SdC %s.", dto.getTypeAgricultureId(), dto.getName()));
            }
            else {
                refTypeAgricultureId = refTypeAgriculture.getTopiaId();
                growingSystem.setTypeAgriculture(refTypeAgriculture);
            }
        }
        return refTypeAgricultureId;
    }

    protected boolean validDephyNumberAffiliation(ImportResults importResults, long line, GrowingSystemImportDto dto, GrowingSystem growingSystem, boolean error) {
        String dephyNumber = dto.getDephyNumber();
        if (StringUtils.isNotBlank(dephyNumber)) {
            Integer campaign = growingSystem.getGrowingPlan().getDomain().getCampaign();
            LinkedHashMap<Integer, List<String>> relatedGrowingSystems = growingSystemService.getGSCodeGPCodeDomainCodeByCampaignForGrowingSystemDephyId(dephyNumber);
            if (relatedGrowingSystems != null && !relatedGrowingSystems.isEmpty()) {
                if (!relatedGrowingSystems.containsKey(campaign)) {
                    // growing system with same Dephy Number is found !

                    // propagate persisted Growing System code to PZ0 imported growing system
                    List<String> codes = relatedGrowingSystems.values().iterator().next();
                    if (CollectionUtils.isNotEmpty(codes)) {
                        MultiKey<Object> gsKey = new MultiKey<Object>(GrowingSystem.class, dto.getCode());
                        if (AbstractCSVImporter.pz0CodeToAgrosystCode.containsKey(gsKey)) {
                            AbstractCSVImporter.pz0CodeToAgrosystCode.put(GrowingSystem.class, dto.getCode(), codes.get(0));
                        }
                        growingSystem.setCode(codes.get(0));

                        // propagate persisted Growing Plan code to PZ0 imported growing plan
                        String growingPlanCsvCode = AbstractCSVImporter.agrosystCodeToPz0Code.get(GrowingPlan.class, growingSystem.getGrowingPlan().getCode());
                        MultiKey<Object> gpKey = new MultiKey<Object>(GrowingPlan.class, growingPlanCsvCode);
                        if (AbstractCSVImporter.pz0CodeToAgrosystCode.containsKey(gpKey)) {
                            AbstractCSVImporter.pz0CodeToAgrosystCode.put(GrowingPlan.class, growingPlanCsvCode, codes.get(1));
                            List<GrowingPlan> growingPlans = (List<GrowingPlan>) pz0CodeToObject.get(GrowingPlan.class, growingPlanCsvCode);
                            if (CollectionUtils.isNotEmpty(growingPlans)) {
                                for (GrowingPlan growingPlan : growingPlans) {
                                    growingPlan.setCode(codes.get(1));
                                }
                            }
                        }

                        // propagate persisted domain code to PZ0 imported domain
                        String domainCsvCode = AbstractCSVImporter.agrosystCodeToPz0Code.get(Domain.class, growingSystem.getGrowingPlan().getDomain().getCode());
                        MultiKey<Object> dKey = new MultiKey<Object>(Domain.class, domainCsvCode);
                        if (AbstractCSVImporter.pz0CodeToAgrosystCode.containsKey(dKey)) {
                            AbstractCSVImporter.pz0CodeToAgrosystCode.put(Domain.class, domainCsvCode, codes.get(2));
                            List<DomainAndDependencies> domainsAndDependencies = (List<DomainAndDependencies>) pz0CodeToObject.get(Domain.class, domainCsvCode);
                            if (CollectionUtils.isNotEmpty(domainsAndDependencies)) {
                                for (DomainAndDependencies domainAndDependencies : domainsAndDependencies) {
                                    Domain agrosystDomains = domainDao.forProperties(Domain.PROPERTY_CODE, codes.get(2), Domain.PROPERTY_CAMPAIGN, domainAndDependencies.getDomain().getCampaign()).findAnyOrNull();                                               
                                    if (agrosystDomains != null) {
                                        error = true;
                                        importResults.addErrorLine(line, String.format("SDC IGNORÉ!, un domaine affilié au domaine %s - %s auquel est relié le SdC %s existe déjà pour la campagne %s",codes.get(2), growingSystem.getGrowingPlan().getDomain().getName(), dto.getDephyNumber(), domainAndDependencies.getDomain().getCampaign()));
                                    }
                                    domainAndDependencies.getDomain().setCode(codes.get(2));
                                }
                            }
                        }
                    }
                } else {
                    error = true;
                    importResults.addErrorLine(line, String.format("SDC IGNORÉ!, un system de culture dont le numéro dephy est %s existe déjà pour la campagne %s",dto.getDephyNumber(), campaign));
                }
            } else {
                importResults.addInfoLine(line, ", " + String.format("ATTENTION !, le système de culture avec comme id %s n'a pas retouvé d'affiliation avec un système de culture ayant comme numéro DEPHY: %s", dto.getId(), dto.getDephyNumber()));
            }
        }
        return error;
    }

    protected boolean validGrowingPlanAffiliation(ImportResults importResults, long line, GrowingSystemImportDto dto, GrowingSystem growingSystem, boolean error) {
        try {
            if (StringUtils.isNotBlank(dto.getGrowingPlanId())) {
                GrowingPlan growingPlan = (GrowingPlan)pz0IdToObject.get(GrowingPlan.class, dto.getGrowingPlanId());
                if (growingPlan != null) {
                    growingSystem.setGrowingPlan(growingPlan);
                } else {
                    importResults.addErrorLine(line, String.format("SDC IGNORÉ!, aucun dispositif n'est retrouvée avec l'identifiant suivant: %s",dto.getGrowingPlanId()));
                    error = true;
                }
            } else {
                importResults.addErrorLine(line, String.format("SDC IGNORÉ!, la colonne 'growingplan' n'est pas renseignée pour le SDC %s", dto.getName()));
                error = true;
            }
        } catch (ClassCastException e) {
            importResults.addErrorLine(line, String.format("SDC IGNORÉ!, dispositif non retrouvé pour le SDC %s", dto.getName()));
            error = true;
        }
        return error;

    }

    protected boolean validGrowingSystemCode(ImportResults importResults, long line, GrowingSystemImportDto dto, GrowingSystem growingSystem, boolean error) {
        String code = growingSystem.getCode();
        if (StringUtils.isNotBlank(code)) {
            if (StringUtils.isNotBlank(dto.getCode())) {
                pz0CodeToAgrosystCode.put(GrowingSystem.class, dto.getCode(), growingSystem.getCode());
            }
        } else if (StringUtils.isNotBlank(dto.getCode())) {
            growingSystem.setCode(csvCodeToAgrosystCode(GrowingSystem.class,  dto.getCode()));
        } else {
            importResults.addErrorLine(line, "SDC IGNORÉ!, le code du système de culture n'a pu être retrouvé et la colonne 'code' n'est pas renseignée.");
            error = true;
        }
        return error;
    }

    protected boolean validGrowingSystemName(ImportResults importResults, long line, GrowingSystem growingSystem, boolean error) {
        if (StringUtils.isBlank(growingSystem.getName())) {
            importResults.addErrorLine(line, "SDC IGNORÉ!, Le nom du SDC n'est pas renseignés.");
            error = true;
        }
        return error;
    }

    protected boolean validGrowingSystemSector(ImportResults importResults, GrowingSystemImportDto dto, long line, GrowingSystem growingSystem, boolean error) {
        Sector sector;
        String value = dto.getSectorImporterValue();
        if (StringUtils.isNotBlank(value)) {
            try {
                sector = Sector.valueOf(value);
                growingSystem.setSector(sector);
            } catch (IllegalArgumentException ex) {
                error = true;
                importResults.addErrorLine(line, String.format("SDC IGNORÉ! Secteur non valide: %s", value));
            }
        } else {
            importResults.addErrorLine(line, "SDC IGNORÉ!, la colonne 'sector' n'est pas renseignés.");
            error = true;
        }
        return error;

    }

    protected boolean validGrowingSystemUpdateDate(ImportResults importResults, long line, GrowingSystem growingSystem, boolean error) {
        if (growingSystem.getUpdateDate() == null) {
            importResults.addErrorLine(line, "SDC IGNORÉ!, la date de mise à jour n'est pas renseignés.");
            error = true;
        }
        return error;
    }

    protected boolean addGrowingSystemCategoryStrategy(ImportResults importResults, GrowingSystemImportDto dto, long line, GrowingSystem growingSystem, boolean error) {
        CategoryStrategy categoryStrategy = null;
        String value = dto.getCategoryStrategyImporterValue();
        if (StringUtils.isNotBlank(value)) {
            try {
                categoryStrategy = CategoryStrategy.valueOf(value);
            } catch (IllegalArgumentException ex) {
                error = true;
                importResults.addErrorLine(line, String.format("SDC IGNORÉ! Catégorie de stratégie : (growingSystem.categoryStrategy), non supportée: %s", value));
            }
        }
        growingSystem.setCategoryStrategy(categoryStrategy);
        return error;

    }

    @Override
    public void init(ServiceContext serviceContext) {
        super.init(serviceContext);
        refTypeAgricultureDao = getPersistenceContext().getRefTypeAgricultureDao();
        growingPlanDao = getPersistenceContext().getGrowingPlanDao();
        domainDao = getPersistenceContext().getDomainDao();
        growingSystemService = getServiceFactory().newService(GrowingSystemService.class);
    }

}
