package fr.inra.agrosyst.services.pz0import.growingPlan;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: GrowingPlanImporter.java 5041 2015-07-15 13:37:42Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/growingPlan/GrowingPlanImporter.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.collect.Lists;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.DomainTopiaDao;
import fr.inra.agrosyst.api.entities.GrowingPlan;
import fr.inra.agrosyst.api.entities.TypeDEPHY;
import fr.inra.agrosyst.api.services.growingplan.GrowingPlanService;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.growingPlan.GrowingPlanAndDependencies;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.services.pz0import.AbstractCSVImporter;
import fr.inra.agrosyst.services.referential.csv.AbstractAgrosystModel;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.InputStream;
import java.text.ParseException;
import java.util.List;
import java.util.Map;

/**
 *
 * @author schellen
 */
public class GrowingPlanImporter extends AbstractCSVImporter {

    private static final Log log = LogFactory.getLog(GrowingPlanImporter.class);

    private GrowingPlanService growingPlanService;
    protected DomainTopiaDao domainDao;

    
    @Override
    public ImportResults importFromStream(InputStream is, Map<String, EntityAndDependencies> domainsByCsvIds) {

        ImportResults importResults = new ImportResults(GrowingPlan.class);

        GrowingPlanImportModel model = new GrowingPlanImportModel();
        Import<GrowingPlanImportDto> importer = Import.newImport(model, is);

        // match the csv line number.
        long line = FIRST_LINE_NUMBER;
        for (GrowingPlanImportDto dto : importer) {
            boolean error;
            GrowingPlan growingPlan = growingPlanService.newGrowingPlan();

            Binder<GrowingPlanImportDto, GrowingPlan> growingPlanBinder = BinderFactory.newBinder(GrowingPlanImportDto.class, GrowingPlan.class);
            growingPlanBinder.copyExcluding(dto, growingPlan,
                    GrowingPlan.PROPERTY_TOPIA_ID);

            growingPlan.setCode(csvCodeToAgrosystCode(GrowingPlan.class, dto.getCode()));

            // required datas
            error = validGrowingPlanCode(importResults, line, growingPlan, false);
            error = validDomainAffiliation(importResults, line, dto, growingPlan, domainsByCsvIds, error);
            error = validGrowingPlanName(importResults, line, growingPlan, error);
            error = validType(importResults, line, dto, growingPlan, error);

            GrowingPlanAndDependencies growingPlanAndDependencies = new GrowingPlanAndDependencies(growingPlan, dto.getDomainId());
            importResults.addEntity(dto.getId(), growingPlanAndDependencies);
            if (!error) {
                importResults.addInfoLine(line, ", " + "DISPOSITIF VALIDÉ, csvid: " + dto.getId());
                importResults.increaseAddedRecords();

            } else {
                importResults.increaseIgnoredRecords();
                importResults.addErrorLine(line, "DISPOSITIF IGNORÉ csvid:" + dto.getId());
            }
            pz0IdToObject.put(GrowingPlan.class, dto.getId(), growingPlan);

            List<GrowingPlan> growingPlans = (List<GrowingPlan>) pz0CodeToObject.get(GrowingPlan.class, dto.getCode());
            if (growingPlans == null) {
                growingPlans = Lists.newArrayList();
                pz0CodeToObject.put(GrowingPlan.class, dto.getCode(), growingPlans);
            }
            growingPlans.add(growingPlan);

            pz0IdToRelatedObjectId.put(GrowingPlan.class, GrowingPlanAndDependencies.class, growingPlan, growingPlanAndDependencies);
            line++;
        }
        return importResults;
    }

    protected boolean validGrowingPlanCode(ImportResults importResults, long line, GrowingPlan growingPlan, boolean error) {
        if (StringUtils.isBlank(growingPlan.getCode())) {
            importResults.addErrorLine(line, "DISPOSITIF IGNORÉ!, Le code du domaine n'est pas valide.");
            error = true;
        }
        return error;
    }

    protected boolean validDomainAffiliation(ImportResults importResults, long line, GrowingPlanImportDto dto, GrowingPlan growingPlan, Map<String, EntityAndDependencies> domainsByCsvIds, boolean error) {
        try {
            if (StringUtils.isNotBlank(dto.getDomainId())) {
                EntityAndDependencies domainAndDependencies =  domainsByCsvIds.get(dto.getDomainId());
                if (domainAndDependencies != null) {
                    growingPlan.setDomain((Domain) domainAndDependencies.getEntity());
                } else {
                    importResults.addErrorLine(line, String.format("DISPOSITIF IGNORÉ!, aucun domain n'est retrouvée avec l'identifiant suivant: %s",dto.getDomainId()));
                    error = true;
                }
            } else {
                importResults.addErrorLine(line, String.format("DISPOSITIF IGNORÉ!, la colonne 'domaine' n'est pas renseignée pour le dispositif\"%s\"!", dto.getName()));
                error = true;
            }
        } catch (ClassCastException e) {
            importResults.addErrorLine(line, String.format("DISPOSITIF IGNORÉ!, domain non retrouvé pour le dispositif\"%s\"!", dto.getName()));
            error = true;
        }
        return error;

    }

    protected boolean validGrowingPlanName(ImportResults importResults, long line, GrowingPlan growingPlan, boolean error) {
        if (StringUtils.isBlank(growingPlan.getName())) {
            importResults.addErrorLine(line, "DISPOSITIF IGNORÉ!, Le nom du dispositif n'est pas renseignés.");
            error = true;
        }
        return error;
    }

    protected boolean validType(ImportResults importResults, long line, GrowingPlanImportDto dto, GrowingPlan growingPlan, boolean error) {
        String value = dto.getTypeDEPHYImportValues();
        if (StringUtils.isNotBlank(value)) {
            try {
                TypeDEPHY typeDEPHY = AbstractAgrosystModel.TYPE_DEPHY_PARSER.parse(value);
                typeDEPHY = typeDEPHY == null ? TypeDEPHY.valueOf(value) : typeDEPHY;
                growingPlan.setType(typeDEPHY);
            } catch (ParseException pe) {
                importResults.addErrorLine(line, "DISPOSITIF IGNORÉ!, Le type de dispositif n'est pas correct.");
                error = true;
            } catch (IllegalArgumentException ie) {
                importResults.addErrorLine(line, "DISPOSITIF IGNORÉ!, Le type de dispositif n'est pas correct.");
                error = true;
            }
        } else {
            importResults.addErrorLine(line, String.format("DISPOSITIF IGNORÉ!, La colonne 'type' n'est pas renseigné pour le dispositif %s.", growingPlan.getName()));
            error = true;
        }
        return error;
    }


    @Override
    public void init(ServiceContext serviceContext) {
        super.init(serviceContext);
        domainDao = getPersistenceContext().getDomainDao();
        growingPlanService = getServiceFactory().newService(GrowingPlanService.class);
    }

}
