package fr.inra.agrosyst.services.pz0import.domain;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: CroppingPanEntrySpeciesImporter.java 5066 2015-08-26 09:22:26Z eancelet $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/domain/CroppingPanEntrySpeciesImporter.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import fr.inra.agrosyst.api.entities.CroppingPlanSpecies;
import fr.inra.agrosyst.api.entities.referential.RefEspece;
import fr.inra.agrosyst.api.entities.referential.RefEspeceTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefVariete;
import fr.inra.agrosyst.api.entities.referential.RefVarieteGevesTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefVarietePlantGrapeTopiaDao;
import fr.inra.agrosyst.api.services.domain.CroppingPlanEntryDto;
import fr.inra.agrosyst.api.services.domain.CroppingPlanSpeciesDto;
import fr.inra.agrosyst.api.services.domain.DomainService;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.domains.DomainAndDependencies;
import fr.inra.agrosyst.api.services.referential.ReferentialService;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.services.pz0import.AbstractCSVImporter;
import fr.inra.agrosyst.services.pz0import.domain.dto.CroppingPlanSpeciesImportDto;
import fr.inra.agrosyst.services.pz0import.domain.model.CroppingPlanSpeciesImportModel;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.InputStream;
import java.util.Map;

/**
 * Created by davidcosse on 05/12/14.
 */
public class CroppingPanEntrySpeciesImporter extends AbstractCSVImporter {

    private static final Log log = LogFactory.getLog(CroppingPanEntrySpeciesImporter.class);
    protected DomainService domainService;
    protected ReferentialService referentialService;
    protected RefVarieteGevesTopiaDao refVarieteGevesesDao;
    protected RefVarietePlantGrapeTopiaDao refVarietePlantGrapeDao;
    protected RefEspeceTopiaDao refEspecesDao;

    @Override
    public ImportResults importFromStream(InputStream is,  Map<String, EntityAndDependencies> entitiesByCsvIds) {

        ImportResults importResults = new ImportResults(CroppingPlanSpecies.class);

        CroppingPlanSpeciesImportModel model = new CroppingPlanSpeciesImportModel();
        Import<CroppingPlanSpeciesImportDto> importer = Import.newImport(model, is);

        // match the csv line number.
        Long line = FIRST_LINE_NUMBER;
        for (CroppingPlanSpeciesImportDto dto : importer) {
            boolean error;
            // we use CroppingPlanEntryDto because the service used to save the domain required a collection of this type object.
            CroppingPlanSpeciesDto croppingPlanSpeciesDto = new CroppingPlanSpeciesDto();

            Binder<CroppingPlanSpeciesImportDto, CroppingPlanSpeciesDto> croppingPlanEntryDtoBinder = BinderFactory.newBinder(CroppingPlanSpeciesImportDto.class, CroppingPlanSpeciesDto.class);
            croppingPlanEntryDtoBinder.copy(dto, croppingPlanSpeciesDto);

            error = validCroppingPlanEntryCode(importResults, line, dto, false);

            error = validCroppingPlanEntryAffiliation(importResults, line, dto, croppingPlanSpeciesDto, error);

            error = validRefSpecies(importResults, line, dto, croppingPlanSpeciesDto, error);

            error = validVariety(importResults, line, dto, croppingPlanSpeciesDto, error);

            error = excludeDuplicate(importResults, line, dto.getId(), error);
            
            // the species must be related to an existing Cropping Plan, if not import fail
            stock(importResults, line, dto, error, croppingPlanSpeciesDto);

            line++;
        }
        return  importResults;
    }

    protected boolean excludeDuplicate(ImportResults importResults, Long line, String speciesCsvId, boolean error) {
        if (pz0IdToObject.get(CroppingPlanSpeciesImportDto.class, speciesCsvId) != null) {
            error = true;
            importResults.addErrorLine(line, String.format("ESPECE IGNORÉE, il existe déjà une croppingPlanSpecies avec comme topiaId: %s", speciesCsvId));
        }
        return error;
    }

    protected boolean validCroppingPlanEntryCode(ImportResults importResults, Long line, CroppingPlanSpeciesImportDto dto, boolean error) {
        if (StringUtils.isBlank(dto.getCode())) {
            importResults.addErrorLine(line, "ESPECE IGNORÉE, le colonne 'code' n'est pas renseignée");
            error = true;
        }
        return error;
    }

    protected boolean validCroppingPlanEntryAffiliation(ImportResults importResults, Long line, CroppingPlanSpeciesImportDto dto, CroppingPlanSpeciesDto croppingPlanSpeciesDto, boolean error) {
        if (StringUtils.isNotBlank(dto.getCroppingPlanEntryId())) {
            CroppingPlanEntryDto croppingPlanEntryDto = (CroppingPlanEntryDto) pz0IdToObject.get(CroppingPlanEntryDto.class, dto.getCroppingPlanEntryId());
            if (croppingPlanEntryDto != null) {
                croppingPlanEntryDto.addSpecies(croppingPlanSpeciesDto);
                DomainAndDependencies domainAndDependencies = (DomainAndDependencies) pz0IdToObject.get(CroppingPlanEntryDto.class, DomainAndDependencies.class, dto.getCroppingPlanEntryId());
                // domain and dependencies can be null if crop import, that's the reason why species is not added to crop from it.
                if (domainAndDependencies != null) {
                    croppingPlanSpeciesDto.setCode(csvCodeToAgrosystCode(CroppingPlanEntryDto.class, dto.getCode()));
                    domainAndDependencies.addSpecies(dto.getId(), dto.getCode(), croppingPlanSpeciesDto);
                }
            } else {
                importResults.addErrorLine(line, String.format("ESPECE IGNORÉE, la culture avec comme identifiant %s n'a pas été retrouvée pour l'espèce csvid: %s", dto.getCroppingPlanEntryId(), dto.getId()));
                error = true;
            }
        } else {
            importResults.addErrorLine(line, String.format("ESPECE IGNORÉE, la colonne 'croppingplanentry' n'est pas renseignée pour la culture csvid: %s", dto.getId()));
            error = true;
        }
        return error;
    }

    protected void stock(ImportResults importResults, Long line, CroppingPlanSpeciesImportDto dto, boolean error, CroppingPlanSpeciesDto croppingPlanSpeciesDto) {
        // the CroppingPlanSpeciesImportDto is push instead of the CroppingPlanSpeciesDto because it carry up more data that the second one
        // and they will probably be use forward into the import process.
        dto.setCroppingPlanSpeciesDto(croppingPlanSpeciesDto);
        pz0IdToObject.put(CroppingPlanSpeciesImportDto.class, dto.getId(), dto);
        if(!error) {
            importResults.addInfoLine(line, ", " + "ESPECE VALIDÉE, csvid " + dto.getId());
            importResults.increaseAddedRecords();
        } else {
            importResults.addErrorLine(line, "ESPECE IGNORÉE, csvid: " + dto.getId());
            importResults.increaseIgnoredRecords();
        }
    }

    protected boolean validRefSpecies(ImportResults importResults, Long line, CroppingPlanSpeciesImportDto from, CroppingPlanSpeciesDto croppingPlanSpeciesDto, boolean error) {
        String speciesId = croppingPlanSpeciesDto.getSpeciesId();
        if (StringUtils.isNotBlank(speciesId)) {
            RefEspece species = refEspecesDao.forTopiaIdEquals(croppingPlanSpeciesDto.getSpeciesId()).findUniqueOrNull();
            from.setSpecies(species);
            if (species == null) {
                error = true;
                importResults.addErrorLine(line, "L'expèce n'a pas été retrouvée, id: " + speciesId);
            }
        } else {
            error = true;
            importResults.addErrorLine(line, "La colonne species n'est pas rensignée");
        }
        return error;
    }

    protected boolean validVariety(ImportResults importResults, Long line, CroppingPlanSpeciesImportDto from,CroppingPlanSpeciesDto croppingPlanSpeciesDto, boolean error) {
        String varietyId = croppingPlanSpeciesDto.getVarietyId();
        if (StringUtils.isNotBlank(varietyId)) {
            RefVariete variety = getVarietyFromId(varietyId);
            if (variety == null) {
                error = true;
                importResults.addErrorLine(line, "La variété de l'expèce n'a pas été retrouvée, id: " + varietyId);
            } else if (from.getSpecies() != null){
                if (!referentialService.validVarietesFromSpeciesId(variety, from.getSpecies().getTopiaId())) {
                    error = true;
                    String qualifiant = from.getSpecies().getLibelle_qualifiant_AEE() != null ? from.getSpecies().getLibelle_qualifiant_AEE() : "";
                    String speciesName = from.getSpecies().getLibelle_espece_botanique() != null ? from.getSpecies().getLibelle_espece_botanique() : "";
                    importResults.addErrorLine(line, String.format("La variété %s n'est pas valide pour l'espèce %s', id: %s", variety.getLabel(), speciesName + "," + qualifiant, varietyId));
                }
            }
        }
        return error;
    }

    protected RefVariete getVarietyFromId(String id) {
        RefVariete variety = refVarieteGevesesDao.forTopiaIdEquals(id).findUniqueOrNull();
        if (variety == null) {
            variety = refVarietePlantGrapeDao.forTopiaIdEquals(id).findUniqueOrNull();
        }
        return variety;
    }

    @Override
    public void init(ServiceContext serviceContext) {
        super.init(serviceContext);
        domainService = getServiceFactory().newService(DomainService.class);
        referentialService = getServiceFactory().newService(ReferentialService.class);
        refVarieteGevesesDao = getPersistenceContext().getRefVarieteGevesDao();
        refVarietePlantGrapeDao = getPersistenceContext().getRefVarietePlantGrapeDao();
        refEspecesDao = getPersistenceContext().getRefEspeceDao();
    }

}
