package fr.inra.agrosyst.services.pz0import.action;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: ActionImporter.java 5092 2015-10-09 08:17:34Z eancelet $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/action/ActionImporter.java $
 * %%
 * Copyright (C) 2013 - 2015 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.ToolsCoupling;
import fr.inra.agrosyst.api.entities.action.AbstractAction;
import fr.inra.agrosyst.api.entities.action.BiologicalControlAction;
import fr.inra.agrosyst.api.entities.action.BiologicalControlActionImpl;
import fr.inra.agrosyst.api.entities.action.CarriageAction;
import fr.inra.agrosyst.api.entities.action.CarriageActionImpl;
import fr.inra.agrosyst.api.entities.action.HarvestingAction;
import fr.inra.agrosyst.api.entities.action.HarvestingActionImpl;
import fr.inra.agrosyst.api.entities.action.IrrigationAction;
import fr.inra.agrosyst.api.entities.action.IrrigationActionImpl;
import fr.inra.agrosyst.api.entities.action.MaintenancePruningVinesActionImpl;
import fr.inra.agrosyst.api.entities.action.MineralFertilizersSpreadingAction;
import fr.inra.agrosyst.api.entities.action.MineralFertilizersSpreadingActionImpl;
import fr.inra.agrosyst.api.entities.action.OrganicFertilizersSpreadingAction;
import fr.inra.agrosyst.api.entities.action.OrganicFertilizersSpreadingActionImpl;
import fr.inra.agrosyst.api.entities.action.OtherActionImpl;
import fr.inra.agrosyst.api.entities.action.PesticidesSpreadingAction;
import fr.inra.agrosyst.api.entities.action.PesticidesSpreadingActionImpl;
import fr.inra.agrosyst.api.entities.action.SeedType;
import fr.inra.agrosyst.api.entities.action.SeedingAction;
import fr.inra.agrosyst.api.entities.action.SeedingActionImpl;
import fr.inra.agrosyst.api.entities.action.TillageAction;
import fr.inra.agrosyst.api.entities.action.TillageActionImpl;
import fr.inra.agrosyst.api.entities.action.YealdUnit;
import fr.inra.agrosyst.api.entities.referential.RefInterventionAgrosystTravailEDI;
import fr.inra.agrosyst.api.entities.referential.RefInterventionAgrosystTravailEDITopiaDao;
import fr.inra.agrosyst.api.services.action.ActionService;
import fr.inra.agrosyst.api.services.effective.EffectiveInterventionDto;
import fr.inra.agrosyst.api.services.practiced.PracticedInterventionDto;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.effective.Pz0EffectiveIntervention;
import fr.inra.agrosyst.api.services.pz0.practicedSystem.Pz0PracticedIntervention;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.services.pz0import.AbstractCSVImporter;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

/**
 * Created by davidcosse on 09/02/15.
 */
public class ActionImporter extends AbstractCSVImporter {

    private static final Log log = LogFactory.getLog(ActionImporter.class);

    protected ActionService actionService;

    protected RefInterventionAgrosystTravailEDITopiaDao refInterventionAgrosystTravailEDIDao;

    protected final Map<String, List<AgrosystInterventionType>> interventionAgrosystTypes = Maps.newHashMap();

    @Override
    public ImportResults importFromStream(InputStream is, Map<String, EntityAndDependencies> entitiesByCsvId) {
        ImportResults importResults = new ImportResults(AbstractAction.class);

        ActionModel model = new ActionModel();
        // récupère le DTO
        Import<ActionImportDto> importer = Import.newImport(model, is);

        // match the first csv line number with data (not header).
        long line = FIRST_LINE_NUMBER;
        for (ActionImportDto dto : importer) {
            boolean error;

            // valid required fields
            error = validIntervention(importResults, line, dto, false);

            Pz0PracticedIntervention pz0PracticedIntervention = null;
            Pz0EffectiveIntervention pz0EffectiveIntervention = null;

            if (!error) {
                pz0PracticedIntervention = getPracticedIntervention(dto);
                pz0EffectiveIntervention = getPz0EffectiveIntervention(dto);
            }

            AbstractAction abstractAction = createAction(dto);
            if (abstractAction != null) {
                error = validToolsCouplingCode(importResults, line, dto, abstractAction, pz0PracticedIntervention, pz0EffectiveIntervention, error);

                error = validMainAction(importResults, line, dto, abstractAction, error);

                error = validRequiredActionData(importResults, line, dto, abstractAction, error);
                
                error = excludeDuplicate(importResults, line, dto.getId(), error);
            } else {
                error = true;
                importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION IGNORÉE!, Le discriminator %s n'est pas valide.", dto.getTopiaDiscriminator()));
            }

            String interventionId = pz0PracticedIntervention != null ? dto.getPracticedInterventionId() : dto.getEffectiveInterventionId();
            error = validInterventionUniqueActionType(importResults, line, interventionId, pz0PracticedIntervention, pz0EffectiveIntervention, abstractAction, error);

            pz0IdToObject.put(AbstractAction.class, dto.getId(), abstractAction);
            if (pz0PracticedIntervention != null) {
                pz0PracticedIntervention.getPracticedInterventionDto().addAction(abstractAction);
                pz0IdToObject.put(AbstractAction.class, Pz0PracticedIntervention.class, dto.getId(), pz0PracticedIntervention);
            } else if (pz0EffectiveIntervention != null){
                pz0EffectiveIntervention.getEffectiveInterventionDto().addAction(abstractAction);
                pz0IdToObject.put(AbstractAction.class, Pz0EffectiveIntervention.class, dto.getId(), pz0EffectiveIntervention);
            }

            if (!error) {
                String actionType = StringUtils.remove(dto.getTopiaDiscriminator(), "fr.inra.agrosyst.api.entities.action.");
                importResults.addInfoLine(line, ", " + String.format("AJOUT DE L'ACTION '%s':'%s' VALIDÉE", actionType, dto.getId()));
                importResults.increaseAddedRecords();
            } else {
                importResults.increaseIgnoredRecords();
                importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION '%s' IGNORÉE", dto.getId()));
            }
            line++;
        }
        return importResults;
    }

    private boolean excludeDuplicate(ImportResults importResults, long lineNumber, String csvId, boolean error) {
        if ((pz0IdToObject.get(AbstractAction.class, Pz0PracticedIntervention.class, csvId) != null) || (pz0IdToObject.get(AbstractAction.class, Pz0EffectiveIntervention.class, csvId) != null)) {
            importResults.addErrorLine(lineNumber, String.format("AJOUT DE L'ACTION IGNORÉE!, l'action de topiaId '%s' est déjà présente dans le fichier", csvId));
            error = true;
        }
        return error;
    }

    protected boolean validInterventionUniqueActionType(ImportResults importResults, long lineNumber, String interventionId, Pz0PracticedIntervention pz0PracticedIntervention, Pz0EffectiveIntervention pz0EffectiveIntervention, AbstractAction action, boolean error) {
        if (!error) {
            if (StringUtils.isNotBlank(interventionId)) {
                List<AgrosystInterventionType> usedActionTypes = interventionAgrosystTypes.get(interventionId);
                if (usedActionTypes == null) {
                    usedActionTypes = getInteventionAgrosystInterventionType(pz0PracticedIntervention, pz0EffectiveIntervention);
                    if (usedActionTypes != null) {
                        interventionAgrosystTypes.put(interventionId, usedActionTypes);
                    } else {
                        importResults.addErrorLine(lineNumber, "AJOUT DE L'ACTION IGNORÉE!, aucune intervention présente pour l'action");
                        error = true;
                    }
                }

                AgrosystInterventionType interventionType = action.getMainAction().getIntervention_agrosyst();
                if (interventionType != AgrosystInterventionType.ENTRETIEN_TAILLE_VIGNE_ET_VERGER && usedActionTypes != null && usedActionTypes.contains(interventionType)) {
                    importResults.addErrorLine(lineNumber, String.format("AJOUT DE L'ACTION IGNORÉE!, le type d'action '%s' ne peut être retrouvé plus d'une fois par intervention", interventionType.name()));
                    error = true;
                } else if (usedActionTypes != null){
                    usedActionTypes.add(interventionType);
                }
            }

        }

        return error;
    }

    protected List<AgrosystInterventionType> getInteventionAgrosystInterventionType(Pz0PracticedIntervention pz0PracticedIntervention, Pz0EffectiveIntervention pz0EffectiveIntervention) {
        Collection<AbstractAction> actions;
        if (pz0PracticedIntervention != null) {
            actions = pz0PracticedIntervention.getPracticedInterventionDto().getActions();
        } else if (pz0EffectiveIntervention != null) {
            actions = pz0EffectiveIntervention.getEffectiveInterventionDto().getActions();
        } else {
            return null;
        }

        List<AgrosystInterventionType> usedActionTypes = Lists.newArrayList();
        Set<AgrosystInterventionType> actionTypes = Sets.newHashSet();
        if (actions != null) {
            for (AbstractAction action : actions) {
                actionTypes.add(action.getMainAction().getIntervention_agrosyst());
            }
            usedActionTypes = new ArrayList<AgrosystInterventionType>(actionTypes);
        }
        return usedActionTypes;
    }

    protected AbstractAction createAction(ActionImportDto dto) {
        AbstractAction action = null;
        String discriminator = dto.getTopiaDiscriminator();
        String actionType = StringUtils.remove(discriminator, "fr.inra.agrosyst.api.entities.action.");

        if (actionType.equals("MineralFertilizersSpreadingActionImpl")) {
            action = new MineralFertilizersSpreadingActionImpl();
        } else if(actionType.equals("PesticidesSpreadingActionImpl")) {
            action = new PesticidesSpreadingActionImpl();
        } else if(actionType.equals("OtherActionImpl")) {
            action = new OtherActionImpl();
        } else if(actionType.equals("MaintenancePruningVinesActionImpl")) {
            action = new MaintenancePruningVinesActionImpl();
        } else if(actionType.equals("OrganicFertilizersSpreadingActionImpl")) {
            action = new OrganicFertilizersSpreadingActionImpl();
        } else if(actionType.equals("IrrigationActionImpl")) {
            action = new IrrigationActionImpl();
        } else if(actionType.equals("BiologicalControlActionImpl")) {
            action = new BiologicalControlActionImpl();
        } else if(actionType.equals("HarvestingActionImpl")) {
            action = new HarvestingActionImpl();
        } else if(actionType.equals("SeedingActionImpl")) {
            action = new SeedingActionImpl();
        } else if(actionType.equals("CarriageActionImpl")) {
            action = new CarriageActionImpl();
        } else if(actionType.equals("TillageActionImpl")) {
            action = new TillageActionImpl();
        }
        if (action != null) {
            action.setTopiaId(ActionService.NEW_ACTION_PREFIX + UUID.randomUUID());
        }
        return action;
    }

    protected boolean validIntervention(ImportResults importResults, long line, ActionImportDto from, boolean error) {
        String practicedInterventionId = from.getPracticedInterventionId();
        String effectiveInterventionId = from.getEffectiveInterventionId();

        if (StringUtils.isNotBlank(practicedInterventionId)) {
            Pz0PracticedIntervention pz0Intervention = (Pz0PracticedIntervention)pz0IdToObject.get(Pz0PracticedIntervention.class, practicedInterventionId);
            if (pz0Intervention == null) {
                error = true;
                importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION IGNORÉE!, L'intervention synthétisé %s n'a pas été retrouvé.", practicedInterventionId));
            }
        } else if (StringUtils.isNotBlank(effectiveInterventionId)) {
            Pz0EffectiveIntervention importedIntervention = (Pz0EffectiveIntervention)pz0IdToObject.get(Pz0EffectiveIntervention.class, effectiveInterventionId);
            if (importedIntervention == null) {
                error = true;
                importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION IGNORÉE!, L'intervention synthétisé %s n'a pas été retrouvé.", effectiveInterventionId));
            }
        } else {
            error = true;
            importResults.addErrorLine(line, "AJOUT DE L'ACTION IGNORÉE!, La colonne 'practicedintervention' ou 'effectiveintervention' doit être renseignée.");
        }

        return error;
    }

    protected Pz0EffectiveIntervention getPz0EffectiveIntervention(ActionImportDto from) {
        String effectiveInterventionId = from.getEffectiveInterventionId();
        Pz0EffectiveIntervention pz0Intervention = (Pz0EffectiveIntervention)pz0IdToObject.get(Pz0EffectiveIntervention.class, effectiveInterventionId);
        return pz0Intervention;
    }

    protected Pz0PracticedIntervention getPracticedIntervention(ActionImportDto from) {
        String practicedInterventionId = from.getPracticedInterventionId();
        Pz0PracticedIntervention pz0Intervention = (Pz0PracticedIntervention)pz0IdToObject.get(Pz0PracticedIntervention.class, practicedInterventionId);
        return pz0Intervention;
    }


    protected boolean validToolsCouplingCode(ImportResults importResults,
                                             long line,
                                             ActionImportDto from,
                                             AbstractAction to,
                                             Pz0PracticedIntervention pz0PracticedIntervention,
                                             Pz0EffectiveIntervention pz0EffectiveIntervention,
                                             boolean error) {

        String toolsCouplingCode = from.getToolsCouplingCode();
        if (StringUtils.isNotBlank(toolsCouplingCode)){
            String agrosystTCCode = pz0CodeToAgrosystCode.get(ToolsCoupling.class, toolsCouplingCode);
            to.setToolsCouplingCode(agrosystTCCode);
            if (pz0PracticedIntervention != null) {
                PracticedInterventionDto practicedInterventionDto = pz0PracticedIntervention.getPracticedInterventionDto();
                if (practicedInterventionDto.getToolsCouplingCodes() == null || !practicedInterventionDto.getToolsCouplingCodes().contains(agrosystTCCode)) {
                    error = true;
                    importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION IGNORÉE!, La combinaison d'outils %s n'a pas été retrouvée sur l'intervention: %s.", toolsCouplingCode, from.getPracticedInterventionId()));
                }
            } else if (pz0EffectiveIntervention != null) {
                EffectiveInterventionDto effectiveInterventionDto = pz0EffectiveIntervention.getEffectiveInterventionDto();
                if (effectiveInterventionDto.getToolsCouplingCodes() == null || !effectiveInterventionDto.getToolsCouplingCodes().contains(agrosystTCCode)) {
                    error = true;
                    importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION IGNORÉE!, La combinaison d'outils %s n'a pas été retrouvée sur l'intervention: %s.", toolsCouplingCode, from.getEffectiveInterventionId()));
                }
            }
        }

        return error;
    }

    protected boolean validMainAction(ImportResults importResults,
                                      long line,
                                      ActionImportDto from,
                                      AbstractAction abstractAction,
                                      boolean error) {

        String mainActionId = from.getMainActionId();
        if (StringUtils.isNotBlank(mainActionId)) {
            RefInterventionAgrosystTravailEDI interventionAgrosystTravailEDI = refInterventionAgrosystTravailEDIDao.forTopiaIdEquals(mainActionId).findUniqueOrNull();
            if(interventionAgrosystTravailEDI != null) {
                AgrosystInterventionType actionType = interventionAgrosystTravailEDI.getIntervention_agrosyst();
                switch (actionType) {
                    case APPLICATION_DE_PRODUITS_FERTILISANTS_MINERAUX:
                        if(!(abstractAction instanceof MineralFertilizersSpreadingActionImpl)) {
                            error = true;
                            importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION IGNORÉE!, l'action '%s' principale ne correspond pas avec le type d'action '%s'", abstractAction.getClass().getSimpleName(), "MineralFertilizersSpreadingAction"));
                        }
                        break;
                    case APPLICATION_DE_PRODUITS_PHYTOSANITAIRES:
                        if(!(abstractAction instanceof PesticidesSpreadingActionImpl)) {
                            error = true;
                            importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION IGNORÉE!, l'action '%s' principale ne correspond pas avec le type d'action '%s'", abstractAction.getClass().getSimpleName(), "PesticidesSpreadingAction"));
                        }
                        break;
                    case AUTRE:
                        if(!(abstractAction instanceof OtherActionImpl)) {
                            error = true;
                            importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION IGNORÉE!, l'action '%s' principale ne correspond pas avec le type d'action '%s'", abstractAction.getClass().getSimpleName(), "OtherAction"));
                        }
                        break;
                    case ENTRETIEN_TAILLE_VIGNE_ET_VERGER:
                        if(!(abstractAction instanceof MaintenancePruningVinesActionImpl)) {
                            error = true;
                            importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION IGNORÉE!, l'action '%s' principale ne correspond pas avec le type d'action '%s'", abstractAction.getClass().getSimpleName(), "MaintenancePruningVinesAction"));
                        }
                        break;
                    case EPANDAGES_ORGANIQUES:
                        if(!(abstractAction instanceof OrganicFertilizersSpreadingActionImpl)) {
                            error = true;
                            importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION IGNORÉE!, l'action '%s' principale ne correspond pas avec le type d'action '%s'", abstractAction.getClass().getSimpleName(), "OrganicFertilizersSpreadingAction"));
                        }
                        break;
                    case IRRIGATION:
                        if(!(abstractAction instanceof IrrigationActionImpl)) {
                            error = true;
                            importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION IGNORÉE!, l'action '%s' principale ne correspond pas avec le type d'action '%s'", abstractAction.getClass().getSimpleName(), "IrrigationAction"));
                        }
                        break;
                    case LUTTE_BIOLOGIQUE:
                        if(!(abstractAction instanceof BiologicalControlActionImpl)) {
                            error = true;
                            importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION IGNORÉE!, l'action '%s' principale ne correspond pas avec le type d'action '%s'", abstractAction.getClass().getSimpleName(), "BiologicalControlAction"));
                        }
                        break;
                    case RECOLTE:
                        if(!(abstractAction instanceof HarvestingActionImpl)) {
                            error = true;
                            importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION IGNORÉE!, l'action '%s' principale ne correspond pas avec le type d'action '%s'", abstractAction.getClass().getSimpleName(), "HarvestingAction"));
                        }
                        break;
                    case SEMIS:
                        if(!(abstractAction instanceof SeedingActionImpl)) {
                            error = true;
                            importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION IGNORÉE!, l'action '%s' principale ne correspond pas avec le type d'action '%s'", abstractAction.getClass().getSimpleName(), "SeedingAction"));
                        }
                        break;
                    case TRANSPORT:
                        if(!(abstractAction instanceof CarriageActionImpl)) {
                            error = true;
                            importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION IGNORÉE!, l'action '%s' principale ne correspond pas avec le type d'action '%s'", abstractAction.getClass().getSimpleName(), "CarriageAction"));
                        }
                        break;
                    case TRAVAIL_DU_SOL:
                        if(!(abstractAction instanceof TillageActionImpl)) {
                            error = true;
                            importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION IGNORÉE!, l'action '%s' principale ne correspond pas avec le type d'action '%s'", abstractAction.getClass().getSimpleName(), "TillageAction"));
                        }
                        break;
                }
                abstractAction.setMainAction(interventionAgrosystTravailEDI);
            } else {
                error = true;
                importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION IGNORÉE!, La l'action principale avec comme identifiant %s n'a pas été trouvée.", mainActionId));
            }

        } else {
            error = true;
            importResults.addErrorLine(line, "AJOUT DE L'ACTION IGNORÉE!, La colonne 'mainaction' doit être renseignée.");
        }
        return error;
    }

    protected boolean validRequiredActionData(ImportResults importResults,
                                              long line,
                                              ActionImportDto from,
                                              AbstractAction to,
                                              boolean error) {
        if (to instanceof TillageAction) {
            // nothing to validate
            ((TillageAction) to).setTillageDepth(from.getTillageDepth());
            ((TillageAction) to).setOtherSettingTool(from.getOtherSettingTool());
        } else if (to instanceof MineralFertilizersSpreadingAction) {
            if (from.getBurial() == null) {
                error = true;
                importResults.addErrorLine(line, "AJOUT DE L'ACTION IGNORÉE!, la colonne 'burial' n'est pas renseignée pour l'action application de produits fertilisants minéreaux.");
            } else {
                ((MineralFertilizersSpreadingAction) to).setBurial(from.getBurial());
            }
            if (from.getLocalizedSpreading() == null) {
                error = true;
                importResults.addErrorLine(line, "AJOUT DE L'ACTION IGNORÉE!, la colonne 'localizedspreading' n'est pas renseignée pour l'action application de produits fertilisants minéreaux.");
            } else {
                ((MineralFertilizersSpreadingAction) to).setLocalizedSpreading(from.getLocalizedSpreading());
            }
        } else if (to instanceof OrganicFertilizersSpreadingAction) {
            if (from.getLandFilledWaste() == null) {
                error = true;
                importResults.addErrorLine(line, "AJOUT DE L'ACTION IGNORÉE!, la colonne 'landfilledwaste' n'est pas renseignée pour l'action épandage organique.");
            } else {
                ((OrganicFertilizersSpreadingAction) to).setLandfilledWaste(from.getLandFilledWaste());
            }
        } else if (to instanceof BiologicalControlAction) {
            ((BiologicalControlAction) to).setBoiledQuantityPerTrip(from.getBoiledQuantityPerTrip());
            ((BiologicalControlAction) to).setTripFrequency(from.getTripFrequency());
            if (from.getBoiledQuantity() != null) {
                ((BiologicalControlAction) to).setBoiledQuantity(from.getBoiledQuantity());
            }
            if (from.getProportionOfTreatedSurface() == null) {
                error = true;
                importResults.addErrorLine(line, "AJOUT DE L'ACTION IGNORÉE!, la colonne 'proportionoftreatedsurface' n'est pas renseignée pour l'action de lutte biologique.");
            } else {
                ((BiologicalControlAction) to).setProportionOfTreatedSurface(from.getProportionOfTreatedSurface());
            }
        } else if (to instanceof PesticidesSpreadingAction) {
            ((PesticidesSpreadingAction) to).setBoiledQuantity(from.getBoiledQuantity());
            ((PesticidesSpreadingAction) to).setBoiledQuantityPerTrip(from.getBoiledQuantityPerTrip());
            ((PesticidesSpreadingAction) to).setTripFrequency(from.getTripFrequency());
            if (from.getAntiDriftNozzle() == null) {
                error = true;
                importResults.addErrorLine(line, "AJOUT DE L'ACTION IGNORÉE!, la colonne 'antidriftnozzle' n'est pas renseignée pour l'action d'application de produits phytosanitaires.");
            } else {
                ((PesticidesSpreadingAction) to).setAntiDriftNozzle(from.getAntiDriftNozzle());
            }
            if (from.getProportionOfTreatedSurface() == null) {
                error = true;
                importResults.addErrorLine(line, "AJOUT DE L'ACTION IGNORÉE!, la colonne 'proportionoftreatedsurface' n'est pas renseignée pour l'action d'application de produits phytosanitaires.");
            } else {
                ((PesticidesSpreadingAction) to).setProportionOfTreatedSurface(from.getProportionOfTreatedSurface());
            }
        } else if (to instanceof SeedingAction) {
            ((SeedingAction) to).setYealdTarget(from.getYealdTarget());
            try {
                if (StringUtils.isNotBlank(from.getYealdUnitName())) {
                    ((SeedingAction) to).setYealdUnit(YealdUnit.valueOf(from.getYealdUnitName()));
                }
            } catch (IllegalArgumentException e) {
                error = true;
                importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION IGNORÉE!, unité de rendement non reconnue: %s pour l'action de type semis.", from.getYealdUnitName()));
            }
            try {
                if (StringUtils.isNotBlank(from.getSeedTypeName())) {
                    ((SeedingAction) to).setSeedType(SeedType.valueOf(from.getSeedTypeName()));
                }
            } catch (IllegalArgumentException e) {
                error = true;
                importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION IGNORÉE!, le type de semis n'est pas reconnue: %s pour l'action de type semis.", from.getSeedTypeName()));
            }
        }  else if (to instanceof CarriageAction) {
            ((CarriageAction) to).setLoadCapacity(from.getLoadCapacity());
            ((CarriageAction) to).setTripFrequency(from.getTripFrequency());
        } else if (to instanceof HarvestingAction){
            ((HarvestingAction) to).setMoisturePercent(from.getMoisturePercent());

            if (from.getExportedWaste() == null) {
                error = true;
                importResults.addErrorLine(line, "AJOUT DE L'ACTION IGNORÉE!, la colonne 'exportedwaste' n'est pas renseignée pour l'action de récolte.");
            } else {
                ((HarvestingAction) to).setExportedWaste(from.getExportedWaste());
            }
        }  else if (to instanceof IrrigationAction){
            ((IrrigationAction) to).setWaterQuantityMax(from.getWaterQuantityMin());
            ((IrrigationAction) to).setWaterQuantityMax(from.getWaterQuantityMax());
            ((IrrigationAction) to).setWaterQuantityMedian(from.getWaterQuantityMedian());
            ((IrrigationAction) to).setAzoteQuantity(from.getAzoteQuantity());
            if (from.getWaterQuantityAverage() == null) {
                error = true;
                importResults.addErrorLine(line, "AJOUT DE L'ACTION IGNORÉE!, la colonne 'waterquantityaverage' n'est pas renseignée pour l'action d'irrigation.");
            } else {
                ((IrrigationAction) to).setWaterQuantityAverage(from.getWaterQuantityAverage());
            }
        }
        return error;
    }

    public void init(ServiceContext serviceContext) {
        super.init(serviceContext);
        actionService = getServiceFactory().newService(ActionService.class);
        refInterventionAgrosystTravailEDIDao = getPersistenceContext().getRefInterventionAgrosystTravailEDIDao();
    }

}
