package fr.inra.agrosyst.services.pz0import;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: AbstractCSVImporter.java 5041 2015-07-15 13:37:42Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/AbstractCSVImporter.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Predicates;
import com.google.common.collect.Collections2;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import fr.inra.agrosyst.api.entities.AgrosystTopiaPersistenceContext;
import fr.inra.agrosyst.api.entities.SolWaterPh;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.services.ServiceFactory;
import fr.inra.agrosyst.api.services.itk.SpeciesStadeDto;
import fr.inra.agrosyst.api.services.pz0.domains.DomainAndDependencies;
import fr.inra.agrosyst.api.services.pz0.effective.EffectiveCropCycleAndDependencies;
import fr.inra.agrosyst.api.services.pz0.plot.PlotAndDependencies;
import fr.inra.agrosyst.services.DefaultServiceFactory;
import fr.inra.agrosyst.services.ServiceContext;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.map.MultiKeyMap;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

/**
 *
 * @author Antoine Schellenberger
 */
public abstract class AbstractCSVImporter implements ICsvImporter {

    private static final String DD_MM_YYYY= "dd/MM/yyyy";
    private static final String YYYY_MM_DD_HH_MM_SS_SSS = "yyyy-MM-dd HH:mm:ss.SSS";
    protected ServiceContext serviceContext;
    protected ServiceFactory serviceFactory;

    // map that allow to keep relation between entities throw there code as describe is the Pz0 csv files
    protected static final MultiKeyMap<Object, String> pz0CodeToAgrosystCode = new MultiKeyMap<Object, String>();
    protected static final MultiKeyMap<Object, String> agrosystCodeToPz0Code = new MultiKeyMap<Object, String>();

    // map that keep relation between object and there csvid
    protected static final MultiKeyMap<Object, Object> pz0IdToObject = new MultiKeyMap<Object, Object>();
    // map that keep relation between object and there code
    protected static final MultiKeyMap<Object, Object> pz0CodeToObject = new MultiKeyMap<Object, Object>();
    // use to be able from a PZ0 id to find related objects ex: GrowingPlan, DomainAndDependencies.class -> DomainAndDependencies
    protected static final MultiKeyMap<Object, Object> pz0IdToRelatedObjectId = new MultiKeyMap<Object, Object>();
    // map permettant de stocker les csvCode des combinaisons d'outils pour un domaine (code) et une campagne
    protected static final MultiKeyMap<Object, Set<String>>  domainCodeCampaignTCCode = new MultiKeyMap<Object, Set<String>> ();
    // map permettant de stocker les csvCode des cultures pour un domaine (code) et une campagne
    protected static final MultiKeyMap<Object, Set<String>>  domainCodeCampaignCropCode = new MultiKeyMap<Object, Set<String>> ();
    // Effective crop cycle for a zone.
    protected static final Map<String, EffectiveCropCycleAndDependencies> effectiveCropCycleAndDependenciesByZone = Maps.newHashMap();

    // line 1 is file header
    protected final Long FIRST_LINE_NUMBER = 2L;

    public static Map<String, EffectiveCropCycleAndDependencies> getEffectiveCropCycleAndDependenciesByZone() {
        return effectiveCropCycleAndDependenciesByZone;
    }

    public static PlotAndDependencies getPlotAndDependenciesForZone(String zoneId) {
        PlotAndDependencies plotAndDependencies = (PlotAndDependencies) pz0IdToRelatedObjectId.get(Zone.class, PlotAndDependencies.class, zoneId);
        return plotAndDependencies;
    }


    protected AgrosystTopiaPersistenceContext getPersistenceContext() {
        return serviceContext.getPersistenceContext();
    }
    
    protected ServiceFactory getServiceFactory() {
        if (serviceFactory == null) {
            serviceFactory = new DefaultServiceFactory(serviceContext);
        }
        return serviceFactory;
    }

    @Override
    public void init(ServiceContext serviceContext) {
        this.serviceContext = serviceContext;
    }


    /**
     * Convert csv code to Agrosyst code format.
     * Maintain a map of object's class, csv code to Agrosyst Code allowing to keep affiliation.
     * @param _class the class of the targeted code
     * @param csvCode the csv code
     * @return the Agrosyst code
     */
    protected String csvCodeToAgrosystCode(Class _class, String csvCode) {
        String agrosystCode = null;
        if (StringUtils.isNotBlank(csvCode)) {
            agrosystCode = pz0CodeToAgrosystCode.get(_class, csvCode);
            if (StringUtils.isBlank(agrosystCode)) {
                agrosystCode = UUID.randomUUID().toString();
                pz0CodeToAgrosystCode.put(_class, csvCode, agrosystCode);
                agrosystCodeToPz0Code.put(_class, agrosystCode, csvCode);
            }
        }

        return agrosystCode;
    }

    /**
     * Try to get SolWaterPh value from String value.
     * May throw NumberFormatException or IllegalArgumentException exception
     * @param value String value for SolWaterPh, value may be the name of SolWaterPh's Enum value ou a numeric value.
     *              Value must not be null
     * @return SolWaterPh
     */
    protected SolWaterPh getSolWaterPh(String value) {
        SolWaterPh solWaterPh = null;

        value = StringUtils.replaceChars(value, ',', '.');
        if ( NumberUtils.isNumber(value) ) {
            double dVal = Double.parseDouble(value);
            if (dVal < 5.5) {
                solWaterPh = SolWaterPh.TRES_ACIDE;
            } else if (dVal < 6.5) {
                solWaterPh = SolWaterPh.ACIDE;
            } else if (dVal < 7.5) {
                solWaterPh = SolWaterPh.NEUTRE;
            } else if (dVal != 0.0){
                solWaterPh = SolWaterPh.BASIQUE;
            }
        } else {
            solWaterPh = SolWaterPh.valueOf(value);
        }
        return solWaterPh;
    }


    protected EffectiveCropCycleAndDependencies getEffectiveCropCycleAndDependencies(String zoneId) {
        EffectiveCropCycleAndDependencies effectiveCropCycleAndDependencies = effectiveCropCycleAndDependenciesByZone.get(zoneId);
        if (effectiveCropCycleAndDependencies == null) {
            effectiveCropCycleAndDependencies = new EffectiveCropCycleAndDependencies(zoneId, getDomainAndDependenciesForZone(zoneId));
            effectiveCropCycleAndDependenciesByZone.put(zoneId, effectiveCropCycleAndDependencies);
        }
        return effectiveCropCycleAndDependencies;
    }

    /**
     * Add DomainDependencies to cycle, will be useful to valid crop and species later on
     * @param zoneId zone Id
     */
    protected DomainAndDependencies getDomainAndDependenciesForZone(String zoneId) {
        PlotAndDependencies plotAndDependencies = (PlotAndDependencies) pz0IdToRelatedObjectId.get(Zone.class, PlotAndDependencies.class, zoneId);
        DomainAndDependencies domainAndDependencies = plotAndDependencies != null ? plotAndDependencies.getDomainAndDependencies() : null;
        return domainAndDependencies;
    }

    protected String getDatePatternForInput(String input) {
        String pattern;
        if (input.matches("([0-9]{2})/([0-9]{2})/([0-9]{4})")) {
            pattern = DD_MM_YYYY;
        } else {
            pattern = YYYY_MM_DD_HH_MM_SS_SSS;
        }
        return pattern;
    }

    protected Set<String> getSpeciesStadeCodes(List<SpeciesStadeDto> speciesStadeDtos) {
        Set<String> speciesCodes = Sets.newHashSet();

        if (CollectionUtils.isNotEmpty(speciesStadeDtos)) {
            for (SpeciesStadeDto speciesStadeDto : speciesStadeDtos) {
                String speciesCode = speciesStadeDto.getSpeciesCode();
                speciesCodes.add(speciesCode);
            }
            Collections2.filter(speciesCodes, Predicates.notNull());
        }
        return speciesCodes;
    }

}
