package fr.inra.agrosyst.services.pz0import.practicedSystem.practicedPerennialCropCycle;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: PracticedPerennialCropCycleImporter.java 5069 2015-08-26 12:07:27Z eancelet $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.2/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/practicedSystem/practicedPerennialCropCycle/PracticedPerennialCropCycleImporter.java $
 * %%
 * Copyright (C) 2013 - 2015 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.OrchardFrutalForm;
import fr.inra.agrosyst.api.entities.PollinatorSpreadMode;
import fr.inra.agrosyst.api.entities.VineFrutalForm;
import fr.inra.agrosyst.api.entities.WeedType;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCycle;
import fr.inra.agrosyst.api.entities.practiced.PracticedPerennialCropCycle;
import fr.inra.agrosyst.api.entities.practiced.PracticedPerennialCropCycleImpl;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystem;
import fr.inra.agrosyst.api.entities.referential.RefOrientationEDI;
import fr.inra.agrosyst.api.entities.referential.RefOrientationEDITopiaDao;
import fr.inra.agrosyst.api.services.domain.CroppingPlanEntryDto;
import fr.inra.agrosyst.api.services.practiced.PracticedPerennialCropCycleDto;
import fr.inra.agrosyst.api.services.practiced.PracticedSystemService;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.domains.DomainAndDependencies;
import fr.inra.agrosyst.api.services.pz0.practicedSystem.PracticedSystemAndDependencies;
import fr.inra.agrosyst.api.services.pz0.practicedSystem.Pz0PracticedPerennialCropCycle;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.services.pz0import.AbstractCSVImporter;
import fr.inra.agrosyst.services.pz0import.practicedSystem.PracticedSystemImporter;
import fr.inra.agrosyst.services.pz0import.practicedSystem.practicedCropCycle.PracticedCropCycleImportDto;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.InputStream;
import java.util.List;
import java.util.Map;

/**
 * Created by davidcosse on 25/03/15.
 */
public class PracticedPerennialCropCycleImporter extends AbstractCSVImporter {

    private static final Log log = LogFactory.getLog(PracticedPerennialCropCycleImporter.class);

    protected PracticedSystemService practicedSystemService;

    protected RefOrientationEDITopiaDao orientationEDIDao;

    protected static final Integer LOWER_VALID_YEAR = 1700;
    protected static final Integer MAX_VALID_YEAR = 2100;

    @Override
    public ImportResults importFromStream(InputStream is, Map<String, EntityAndDependencies> entitiesByCsvId) {

        ImportResults importResults = new ImportResults(PracticedPerennialCropCycleDto.class);

        PracticedPerennialCropCycleImportModel model = new PracticedPerennialCropCycleImportModel();
        // récupère le DTO
        Import<PracticedPerennialCropCycleImportDto> importer = Import.newImport(model, is);

        // match the first csv line number with data (not header).
        long line = FIRST_LINE_NUMBER;
        for (PracticedPerennialCropCycleImportDto dto : importer) {
            boolean error;

            // PracticedPerennialCropCycleDto required by the service PracticedSystem
            Pz0PracticedPerennialCropCycle pz0PracticedPerennialCropCycle = new Pz0PracticedPerennialCropCycle(dto.getId(), dto.getCroppingPlanEntryCode());
            PracticedPerennialCropCycleDto practicedPerennialCropCycleDto = pz0PracticedPerennialCropCycle.getPracticedPerennialCropCycleDto();

            // PracticedPerennialCropCycle to persist
            PracticedPerennialCropCycle practicedPerennialCropCycle = new PracticedPerennialCropCycleImpl();
            Binder<PracticedPerennialCropCycleImportDto, PracticedPerennialCropCycle> binder = BinderFactory.newBinder(PracticedPerennialCropCycleImportDto.class, PracticedPerennialCropCycle.class);
            binder.copyExcluding(dto, practicedPerennialCropCycle,
                    PracticedPerennialCropCycle.PROPERTY_TOPIA_ID,
                    PracticedPerennialCropCycle.PROPERTY_CROPPING_PLAN_ENTRY_CODE,
                    PracticedPerennialCropCycle.PROPERTY_POLLINATOR,
                    PracticedPerennialCropCycle.PROPERTY_WEED_TYPE,
                    PracticedPerennialCropCycle.PROPERTY_POLLINATOR_SPREAD_MODE,
                    PracticedPerennialCropCycle.PROPERTY_ORCHARD_FRUTAL_FORM,
                    PracticedPerennialCropCycle.PROPERTY_VINE_FRUTAL_FORM,
                    PracticedPerennialCropCycle.PROPERTY_ORIENTATION);
            // add PracticedPerennialCropCycle to the dto
            practicedPerennialCropCycleDto.setPracticedPerennialCropCycle(practicedPerennialCropCycle);

            // valid required fields
            error = validPracticedSystemAffiliation(importResults, line, dto, pz0PracticedPerennialCropCycle, false);
            error = validCrops(importResults, line, dto, practicedPerennialCropCycleDto, error);

            // required
            error = validPollinator(importResults, line, dto, practicedPerennialCropCycle, error);
            error = validSolOccupationPercent(importResults, line, dto, practicedPerennialCropCycle, error);
            error = validWeedType(importResults, line, dto, practicedPerennialCropCycle, error);

            // enums
            error = validPollinatorSpreadMode(importResults, line, dto, practicedPerennialCropCycle, error);
            error = validOrchardFrutalForm(importResults, line, dto, practicedPerennialCropCycle, error);
            error = validVineFrutalForm(importResults, line, dto, practicedPerennialCropCycle, error);

            // valid related objects
            error = validOrientation(importResults, line, dto, practicedPerennialCropCycle, error);

            // valid specific import control
            error = validPlantingDensity(importResults, line, dto, practicedPerennialCropCycle, error);
            error = validPlantingYear(importResults, line, dto, practicedPerennialCropCycle, error);
            
            error = excludeDuplicate(importResults, line, dto.getId(), error);

            // need to be able to find it out PracticedPerennialCropCycleDto to add phase ont it
            pz0IdToObject.put(Pz0PracticedPerennialCropCycle.class, dto.getId(), pz0PracticedPerennialCropCycle);

            if (!error) {
                importResults.addInfoLine(line, ", " + "CYCLE DE CULTURE DU SYNTHÉTISÉ VALIDÉ, csvid: " + dto.getId());
                importResults.increaseAddedRecords();
            } else {
                importResults.increaseIgnoredRecords();
                importResults.addErrorLine(line, "CYCLE DE CULTURE DU SYNTHÉTISÉ IGNORÉ csvid:" + dto.getId());
            }

            line++;
        }
        return importResults;
    }

    protected boolean excludeDuplicate(ImportResults importResults, long line, String csvId, boolean error) {
        if (pz0IdToObject.get(Pz0PracticedPerennialCropCycle.class, csvId) != null) {
            error = true;
            importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU SYNTHÉTISÉ IGNORÉ!, le topiaId %s est déjà présent dans le fichier.", csvId));
        }
        return error;
    }

    protected boolean validCrops(ImportResults importResults, long line, PracticedPerennialCropCycleImportDto dto, PracticedPerennialCropCycleDto practicedPerennialCropCycleDto, boolean error) {
        String cropCode = dto.getCroppingPlanEntryCode();
        if (StringUtils.isNotBlank(cropCode)) {
            // try to find agrosyst crop code from PZ0 crop code
            String agrosystCropCode = pz0CodeToAgrosystCode.get(CroppingPlanEntryDto.class, cropCode);
            if (StringUtils.isNotBlank(agrosystCropCode)) {
                // matching crop found
                // check crop belong to practiced system
                PracticedCropCycleImportDto cropCycleDto = (PracticedCropCycleImportDto) pz0IdToObject.get(PracticedCropCycle.class, dto.getId());
                error = checkCropExistOnImportedDomain(importResults, line, error, cropCode, cropCycleDto);
                // set code to cycle
                practicedPerennialCropCycleDto.getPracticedPerennialCropCycle().setCroppingPlanEntryCode(agrosystCropCode);

            } else {
                // no crop imported with this code, look for one on persisted ones
                PracticedCropCycleImportDto cropCycleDto = (PracticedCropCycleImportDto) pz0IdToObject.get(PracticedCropCycle.class, dto.getId());
                if (cropCycleDto != null) {
                    PracticedSystemAndDependencies practicedSystemAndDependencies = (PracticedSystemAndDependencies) pz0IdToObject.get(PracticedSystem.class, cropCycleDto.getPracticedSystemId());
                    error = checkCropExistOnPersistedDomain(importResults, line, error, cropCode, practicedSystemAndDependencies);
                }  else {
                    error = true;
                    importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU SYNTHÉTISÉ IGNORÉ!, la culture au code:%s n'a pas été retrouvée', le système synthétisé %s n'a pas été retrouvé.", cropCode, dto.getId()));
                }
                // set code to cycle
                practicedPerennialCropCycleDto.getPracticedPerennialCropCycle().setCroppingPlanEntryCode(cropCode);
            }
        } else {
            error = true;
            importResults.addErrorLine(line, "CYCLE DE CULTURE PÉRENNE DU SYNTHÉTISÉ IGNORÉ!, la colone 'croppingplanentrycode' n'est pas renseignée");
        }
        return  error;
    }

    protected boolean checkCropExistOnPersistedDomain(ImportResults importResults, long line, boolean error, String cropCode, PracticedSystemAndDependencies practicedSystemAndDependencies) {
        if (practicedSystemAndDependencies != null) {
            try {
                String domainCode = practicedSystemAndDependencies.getEntity().getGrowingSystem().getGrowingPlan().getDomain().getCode();
                String campaigns = practicedSystemAndDependencies.getEntity().getCampaigns();
                if (StringUtils.isNotBlank(campaigns)) {
                    List<String> practicedSystemCroppingPlanEntriesCodes  = practicedSystemService.getCropCodesFromDomainCodeForCampaigns(domainCode, campaigns);
                    if (practicedSystemCroppingPlanEntriesCodes == null || !practicedSystemCroppingPlanEntriesCodes.contains(cropCode)) {
                        error = true;
                        importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU SYNTHÉTISÉ IGNORÉ!, la culture au code:'%s' n'a pas été retrouvée à partir du domain dont le code est '%s' et les campagnes: %s", cropCode, domainCode, campaigns));
                    }
                } else {
                    error = true;
                    importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU SYNTHÉTISÉ IGNORÉ!, la culture au code:'%s' n'a pas été retrouvée aucune campagne n'est indiquée permettant de retrouver le domaine sur le system synthétisé '%S", cropCode, practicedSystemAndDependencies.getCsvId()));
                }

            } catch (NullPointerException e) {
                error = true;
                importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU SYNTHÉTISÉ IGNORÉ!, la culture au code:'%s' n'a pas été retrouvée impossible de remonter jusqu'au domaine", cropCode));
            }

        } else {
            error = true;
            importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU SYNTHÉTISÉ IGNORÉ!, la culture au code:'%s' n'a pas été retrouvée, aucun système synthétisé trouvé", cropCode));
        }
        return error;
    }

    protected boolean checkCropExistOnImportedDomain(ImportResults importResults, long line, boolean error, String cropCode, PracticedCropCycleImportDto cropCycleDto) {
        if (cropCycleDto != null) {
            PracticedSystemAndDependencies practicedSystemAndDependencies = (PracticedSystemAndDependencies) pz0IdToObject.get(PracticedSystem.class, cropCycleDto.getPracticedSystemId());
            try {
                if (practicedSystemAndDependencies != null) {
                    DomainAndDependencies domainAndDependencies = PracticedSystemImporter.getDomainAndDependenciesFromPracticedSystem(practicedSystemAndDependencies.getEntity());
                    if (domainAndDependencies != null) {
                        if (!domainAndDependencies.getCroppingPlanEntryDtosByPZ0Code().containsKey(cropCode)) {
                            error = true;
                            Domain domain = (Domain) domainAndDependencies.getEntity();
                            importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU SYNTHÉTISÉ IGNORÉ!, la culture au code:'%s' n'a pas été retrouvée' à partir du domain %s et pour la campagne %d pour le système synthétisé %s", cropCode, domain.getName(), domain.getCampaign(), practicedSystemAndDependencies.getCsvId()));
                        }
                    } else {
                        error = true;
                        importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU SYNTHÉTISÉ IGNORÉ!, la culture au code:'%s' n'a pas été retrouvée' le domain n'a pu être retouvé pour le système synthétisé '%s'", cropCode, practicedSystemAndDependencies.getCsvId()));
                    }

                } else {
                    error = true;
                    importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU SYNTHÉTISÉ IGNORÉ!, le système synthétisé:'%s' n'a pas été retrouvée'", cropCycleDto.getPracticedSystemId()));
                }
            } catch (NullPointerException e) {
                error = true;
                importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU SYNTHÉTISÉ IGNORÉ!, la culture au code:'%s' n'a pas été retrouvée' impossible de remonter jusqu'au domaine", cropCode));
            }
        } else {
            error = true;
            importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU SYNTHÉTISÉ IGNORÉ!, la culture au code:'%s' n'a pas été retrouvée', aucun système synthétisé n'est retrouvée", cropCode));
        }
        return error;
    }

    protected boolean validPracticedSystemAffiliation(ImportResults importResults, long line, PracticedPerennialCropCycleImportDto dto, Pz0PracticedPerennialCropCycle pz0cycle, boolean error) {
        String target = dto.getId();
        if (StringUtils.isNotBlank(target)) {
            PracticedCropCycleImportDto cropCycleDto = (PracticedCropCycleImportDto) pz0IdToObject.get(PracticedCropCycle.class, target);
            if (cropCycleDto != null) {
                PracticedSystemAndDependencies practicedSystemAndDependencies = (PracticedSystemAndDependencies) pz0IdToObject.get(PracticedSystem.class, cropCycleDto.getPracticedSystemId());

                if (practicedSystemAndDependencies != null) {
                    practicedSystemAndDependencies.addPracticedPerennialCropCycleDto(pz0cycle);
                    pz0cycle.setPracticedSystemId(cropCycleDto.getPracticedSystemId());
                } else {
                    importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU SYNTHÉTISÉ IGNORÉ!, aucun système synthétisé n'est retrouvée avec l'identifiant suivant: %s", target));
                    error = true;
                }
            } else {
                importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU SYNTHÉTISÉ IGNORÉ!, aucun système synthétisé n'est retrouvée avec l'identifiant suivant: %s", target));
                error = true;
            }

        } else {
            importResults.addErrorLine(line, "CYCLE DE CULTURE PÉRENNE DU SYNTHÉTISÉ IGNORÉ!, la colonne 'topiaId' n'est pas renseignée");
            error = true;
        }
        return error;
    }

    protected boolean validPollinator(ImportResults importResults, long line, PracticedPerennialCropCycleImportDto dto, PracticedPerennialCropCycle practicedPerennialCropCycle, boolean error) {
        Boolean isPollinator = dto.isPollinator();
        if (isPollinator == null) {
            importResults.addErrorLine(line, "CYCLE DE CULTURE PÉRENNE DU SYNTHÉTISÉ IGNORÉ!, la colonne 'pollinator' n'est pas renseignée pour le cycle d'id: " + dto.getId());
            error = true;
        } else {
            practicedPerennialCropCycle.setPollinator(isPollinator);
        }
        return error;
    }

    protected boolean validSolOccupationPercent(ImportResults importResults, long line, PracticedPerennialCropCycleImportDto dto, PracticedPerennialCropCycle practicedPerennialCropCycle, boolean error) {
        Double solOccupationPercent = dto.getSolOccupationPercent();
        if (solOccupationPercent == null) {
            importResults.addErrorLine(line, "CYCLE DE CULTURE PÉRENNE DU SYNTHÉTISÉ IGNORÉ!, la colonne 'growingsystemportion' n'est pas renseignée pour le cycle d'id: " + dto.getId());
            error = true;
        } else {
            practicedPerennialCropCycle.setSolOccupationPercent(solOccupationPercent);
        }
        return error;
    }

    protected boolean validWeedType(ImportResults importResults, long line, PracticedPerennialCropCycleImportDto dto, PracticedPerennialCropCycle practicedPerennialCropCycle, boolean error) {
        String weedTypeSt = dto.getWeedType();
        if (StringUtils.isNotBlank(weedTypeSt)) {
            WeedType weedType = null;
            try {
                weedType = WeedType.valueOf(weedTypeSt);
            } catch (IllegalArgumentException e) {
                importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU SYNTHÉTISÉ IGNORÉ!, type d'enherbement '%s' est non valide pour le cycle d'id: %s", weedTypeSt, dto.getId()));
                error = true;
            }
            practicedPerennialCropCycle.setWeedType(weedType);
        } else {
            importResults.addErrorLine(line, "CYCLE DE CULTURE PÉRENNE DU SYNTHÉTISÉ IGNORÉ!, la colonne 'weedType' n'est pas renseignée pour le cycle d'id: " + dto.getId());
            error = true;
        }
        return error;
    }

    protected boolean validPollinatorSpreadMode(ImportResults importResults, long line, PracticedPerennialCropCycleImportDto dto, PracticedPerennialCropCycle practicedPerennialCropCycle, boolean error) {
        String pollinatorSpreadSt = dto.getPollinatorSpreadMode();
        if (StringUtils.isNotBlank(pollinatorSpreadSt)) {
            PollinatorSpreadMode pollinatorSpreadMode = null;
            try {
                pollinatorSpreadMode = PollinatorSpreadMode.valueOf(pollinatorSpreadSt);
            } catch (IllegalArgumentException e) {
                importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU SYNTHÉTISÉ IGNORÉ!, mode de répartition des pollinisateurs '%s' non valide pour le cycle d'id: %s", pollinatorSpreadSt, dto.getId()));
                error = true;
            }
            practicedPerennialCropCycle.setPollinatorSpreadMode(pollinatorSpreadMode);
        }
        return error;
    }

    protected boolean validOrchardFrutalForm(ImportResults importResults, long line, PracticedPerennialCropCycleImportDto dto, PracticedPerennialCropCycle practicedPerennialCropCycle, boolean error) {
        String orchardFrutalFormSt = dto.getOrchardFrutalForm();
        if (StringUtils.isNotBlank(orchardFrutalFormSt)) {
            OrchardFrutalForm orchardFrutalForm = null;
            try {
                orchardFrutalForm = OrchardFrutalForm.valueOf(orchardFrutalFormSt);
            } catch (IllegalArgumentException e) {
                importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU SYNTHÉTISÉ IGNORÉ!, forme fruitière vergers '%s' non valide pour le cycle d'id: %s", orchardFrutalFormSt, dto.getId()));
                error = true;
            }
            practicedPerennialCropCycle.setOrchardFrutalForm(orchardFrutalForm);
        }
        return error;
    }

    protected boolean validVineFrutalForm(ImportResults importResults, long line, PracticedPerennialCropCycleImportDto dto, PracticedPerennialCropCycle practicedPerennialCropCycle, boolean error) {
        String vineFrutalFormSt = dto.getVineFrutalForm();
        if (StringUtils.isNotBlank(vineFrutalFormSt)) {
            VineFrutalForm frutalForm = null;
            try {
                frutalForm = VineFrutalForm.valueOf(vineFrutalFormSt);
            } catch (IllegalArgumentException e) {
                importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU SYNTHÉTISÉ IGNORÉ!, Forme fruitière vigne '%s' non valide pour le cycle d'id: %s", vineFrutalFormSt, dto.getId()));
                error = true;
            }
            practicedPerennialCropCycle.setVineFrutalForm(frutalForm);
        }
        return error;
    }

    protected boolean validOrientation(ImportResults importResults, long line, PracticedPerennialCropCycleImportDto dto, PracticedPerennialCropCycle practicedPerennialCropCycle, boolean error) {
        String orientationId = dto.getOrientation();
        if (StringUtils.isNotBlank(orientationId)) {
            RefOrientationEDI orientationEDI = orientationEDIDao.forTopiaIdEquals(orientationId).findUniqueOrNull();
            if (orientationEDI != null) {
                practicedPerennialCropCycle.setOrientation(orientationEDI);
            } else {
                error = true;
                importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU SYNTHÉTISÉ IGNORÉ!, l'orientation des rangs avec comme identifiant %s n'a pas été retrouvée pour le cycle d'id: %s", orientationId, dto.getId()));
            }
        }
        return error;
    }

    protected boolean validPlantingDensity(ImportResults importResults, long line, PracticedPerennialCropCycleImportDto dto, PracticedPerennialCropCycle practicedPerennialCropCycle, boolean error) {
        // A renseigner si différent de : 10000 / ((interFurrow / 100) * (spacing / 100)
        
        // eancelet 2015-08-25 : actually, plantingInterFurrow + plantingSpacing + plantingDensity can be null 
        /*
        if ((practicedPerennialCropCycle.getPlantingInterFurrow() == null || practicedPerennialCropCycle.getPlantingSpacing() == null) && practicedPerennialCropCycle.getPlantingDensity() == null) {
            importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU SYNTHÉTISÉ IGNORÉ!, la densité de plantation doit être renseigné " +
                    "dans le cas ou celle-ci ne peut pas être calculée à partir de " +
                    "l'inter-rang de plantation et l'espacement de plantation sur le rang pour le cycle d'id: %s", dto.getId()));
            error = true;
        }
        */

        if (practicedPerennialCropCycle.getPlantingDensity() != null && practicedPerennialCropCycle.getPlantingInterFurrow() != null && practicedPerennialCropCycle.getPlantingSpacing() != null) {
            double plantingDensity = 10000 / ((practicedPerennialCropCycle.getPlantingInterFurrow().doubleValue() / 100) * (practicedPerennialCropCycle.getPlantingSpacing().doubleValue() / 100));
            if ((practicedPerennialCropCycle.getPlantingDensity().compareTo(plantingDensity) != 0)) {
                importResults.addErrorLine(line, String.format(
                        "CYCLE DE CULTURE PÉRENNE DU SYNTHÉTISÉ IGNORÉ!, conflit sur la densité de plantation pour le cycle d'id: %s, la densité de plantation est renseignée " +
                                "alors que l'inter-rang de plantation et l'espacement de plantation sur le rang le sont aussi. Valeur importée: %f, valeur calculée: %f", dto.getId(), practicedPerennialCropCycle.getPlantingDensity(), plantingDensity));
                error = true;
            }
        } else if (practicedPerennialCropCycle.getPlantingInterFurrow() != null && practicedPerennialCropCycle.getPlantingSpacing() != null) {
            double plantingDensity = 10000 / ((practicedPerennialCropCycle.getPlantingInterFurrow().doubleValue() / 100) * (practicedPerennialCropCycle.getPlantingSpacing().doubleValue() / 100));
            practicedPerennialCropCycle.setPlantingDensity(plantingDensity);
        }
        return error;
    }

    protected boolean validPlantingYear(ImportResults importResults, long line, PracticedPerennialCropCycleImportDto dto, PracticedPerennialCropCycle practicedPerennialCropCycle, boolean error) {
        Integer plantingYear = dto.getPlantingYear();
        if (plantingYear != null) {
            if (plantingYear > LOWER_VALID_YEAR && plantingYear < MAX_VALID_YEAR) {
                practicedPerennialCropCycle.setPlantingYear(plantingYear);
            } else {
                error = true;
                importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU SYNTHÉTISÉ IGNORÉ!, année de plantation %s non valide pour le cycle d'id: %s", dto.getPlantingYear(), dto.getId()));
            }
        }
        return error;
    }

    @Override
    public void init(ServiceContext serviceContext) {
        super.init(serviceContext);
        orientationEDIDao = getPersistenceContext().getRefOrientationEDIDao();
        practicedSystemService = getServiceFactory().newService(PracticedSystemService.class);
    }

}
