package fr.inra.agrosyst.services.pz0import.practicedSystem.practicedCropCycleSpecies;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: PracticedCropCycleSpeciesImporter.java 5057 2015-08-12 13:29:21Z eancelet $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.2/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/practicedSystem/practicedCropCycleSpecies/PracticedCropCycleSpeciesImporter.java $
 * %%
 * Copyright (C) 2013 - 2015 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.CroppingPlanSpecies;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCycleSpecies;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCycleSpeciesImpl;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystem;
import fr.inra.agrosyst.api.entities.referential.RefClonePlantGrape;
import fr.inra.agrosyst.api.entities.referential.RefClonePlantGrapeTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefVariete;
import fr.inra.agrosyst.api.entities.referential.RefVarieteTopiaDao;
import fr.inra.agrosyst.api.services.domain.CroppingPlanEntryDto;
import fr.inra.agrosyst.api.services.domain.CroppingPlanSpeciesDto;
import fr.inra.agrosyst.api.services.domain.DomainService;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.domains.DomainAndDependencies;
import fr.inra.agrosyst.api.services.pz0.practicedSystem.PracticedSystemAndDependencies;
import fr.inra.agrosyst.api.services.pz0.practicedSystem.Pz0PracticedPerennialCropCycle;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.services.common.CommonService;
import fr.inra.agrosyst.services.pz0import.AbstractCSVImporter;
import fr.inra.agrosyst.services.pz0import.practicedSystem.PracticedSystemImporter;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.nuiton.csv.Import;

import java.io.InputStream;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Created by davidcosse on 10/06/15.
 */
public class PracticedCropCycleSpeciesImporter extends AbstractCSVImporter {

    // services
    protected DomainService domainService;

    // dao
    protected RefClonePlantGrapeTopiaDao refClonePlantGrapesDao;
    protected RefVarieteTopiaDao refVarieteDao;


    @Override
    public ImportResults importFromStream(InputStream is, Map<String, EntityAndDependencies> entitiesByCsvId) {
        ImportResults importResults = new ImportResults(PracticedCropCycleSpecies.class);

        PracticedCropCycleSpeciesImportModel model = new PracticedCropCycleSpeciesImportModel();
        // récupère le DTO
        Import<PracticedCropCycleSpeciesImportDto> importer = Import.newImport(model, is);

        // match the first csv line number with data (not header).
        long line = FIRST_LINE_NUMBER;
        for (PracticedCropCycleSpeciesImportDto dto : importer) {
            boolean error;

            PracticedCropCycleSpecies practicedCropCycleSpecies = new PracticedCropCycleSpeciesImpl();
            practicedCropCycleSpecies.setPlantsCertified(dto.getIsPlantsCertified());

            // valid required fields
            error = validPracticedPerennialCropCycleAffiliation(importResults, line, dto, false);
            error = validCroppingPlanSpeciesCode(importResults, line, dto, error);

            // valid optional
            error = validGraftCloneId(importResults, line, practicedCropCycleSpecies, dto, error);
            error = validGraftSupportId(importResults, line, practicedCropCycleSpecies, dto, error);
            error = validOverGraftDate(importResults, line, practicedCropCycleSpecies, dto, error);

            if (!error) {
                practicedCropCycleSpecies.setCroppingPlanSpeciesCode(dto.croppingPlanSpeciesCode);
                practicedCropCycleSpecies.setCycle(dto.getPz0PracticedPerennialCropCycle().getPracticedPerennialCropCycleDto().getPracticedPerennialCropCycle());
                dto.getPz0PracticedPerennialCropCycle().getPracticedPerennialCropCycleDto().addCroppingPlanSpeciesIds(dto.croppingPlanSpeciesCode);
                dto.getPz0PracticedPerennialCropCycle().getPracticedPerennialCropCycleDto().getPracticedPerennialCropCycle().addPracticedCropCycleSpecies(practicedCropCycleSpecies);
                importResults.increaseAddedRecords();
            } else {
                importResults.increaseIgnoredRecords();
                importResults.addErrorLine(line, "ESPÈCE DU CYCLE DE CULTURE DU SYNTHÉTISÉ IGNORÉE csvid:" + dto.getId());
            }

            line++;
        }
        return importResults;
    }

    protected boolean validPracticedPerennialCropCycleAffiliation(ImportResults importResults, long line, PracticedCropCycleSpeciesImportDto dto, boolean error) {
        String practicedPerennialCropCycleId = dto.getPracticedPerennialCycleId();
        if (StringUtils.isNotBlank(practicedPerennialCropCycleId)) {
            Pz0PracticedPerennialCropCycle pz0PracticedPerennialCropCycle = (Pz0PracticedPerennialCropCycle) pz0IdToObject.get(Pz0PracticedPerennialCropCycle.class, practicedPerennialCropCycleId);
            if (pz0PracticedPerennialCropCycle == null) {
                importResults.addErrorLine(line, String.format("ESPÈCE DU CYCLE DE CULTURE DU SYNTHÉTISÉ IGNORÉE!, aucun cycle de culture perenne n'est retrouvé avec l'identifiant suivant: '%s'", practicedPerennialCropCycleId));
                error = true;
            } else {
                dto.setPz0PracticedPerennialCropCycle(pz0PracticedPerennialCropCycle);
                String cropCode = pz0PracticedPerennialCropCycle.getPz0CropCode();
                dto.setCroppingPlanEntryCode(cropCode);

                String practicedSystemId = pz0PracticedPerennialCropCycle.getPracticedSystemId();
                if (StringUtils.isNotBlank(practicedSystemId)) {
                    PracticedSystemAndDependencies practicedSystemAndDependencies = (PracticedSystemAndDependencies) pz0IdToObject.get(PracticedSystem.class, practicedSystemId);
                    if (practicedSystemAndDependencies != null) {
                        PracticedSystem practicedSystem = practicedSystemAndDependencies.getPracticedSystem();
                        if (practicedSystem != null) {
                            String campaigns = practicedSystem.getCampaigns();
                            dto.setPracticedSystemCampaigns(campaigns);
                            dto.setDomainAndDependencies(PracticedSystemImporter.getDomainAndDependenciesFromPracticedSystem(practicedSystem));
                        } else {
                            importResults.addErrorLine(line, String.format("ESPÈCE DU CYCLE DE CULTURE DU SYNTHÉTISÉ IGNORÉE!, l'espèce au code '%s' n'a pu être retrouvé car le système synthétisé avec comme identifiant '%s' n'a pu être lui-même retrouvé", practicedPerennialCropCycleId, practicedSystemId));
                            error = true;
                        }
                    } else {
                        importResults.addErrorLine(line, String.format("ESPÈCE DU CYCLE DE CULTURE DU SYNTHÉTISÉ IGNORÉE!, l'espèce au code '%s' n'a pu être retrouvé car le système synthétisé avec comme identifiant '%s' n'a pu être lui-même retrouvé", practicedPerennialCropCycleId, practicedSystemId));
                        error = true;
                    }
                } else {
                    importResults.addErrorLine(line, String.format("ESPÈCE DU CYCLE DE CULTURE DU SYNTHÉTISÉ IGNORÉE!, le système synthétisé avec comme identifiant '%s' n'existe pas ", practicedSystemId));
                    error = true;
                }
            }
        } else {
            importResults.addErrorLine(line, "ESPÈCE DU CYCLE DE CULTURE DU SYNTHÉTISÉ IGNORÉE!, la colonne 'cycle' n'est pas renseignée !");
            error = true;
        }
        return error;
    }

    protected boolean validCroppingPlanSpeciesCode(ImportResults importResults, long line, PracticedCropCycleSpeciesImportDto dto, boolean error) {
        String target = dto.getCroppingPlanSpeciesCode();

        if (StringUtils.isNotBlank(target)) {

            boolean isValidSpecies = validSpeciesFromPz0ImportedValues(dto);
            isValidSpecies = validSpeciesFromExistingValues(importResults, line,dto, isValidSpecies);

            if (!isValidSpecies) {
                String pz0DomainId = dto.getDomainAndDependencies() == null ? "?" : dto.getDomainAndDependencies().getCsvId();
                importResults.addErrorLine(line, String.format("ESPÈCE DU CYCLE DE CULTURE DU SYNTHÉTISÉ IGNORÉE!, l'espèce au code '%s' n'a pas été retrouvée sur la culture au code '%s' du domaine '%s' pour les campagnes '%s' !", dto.getCroppingPlanSpeciesCode(), dto.getCroppingPlanEntryCode(), pz0DomainId, dto.getPracticedSystemCampaigns()));
                error = true;
            }
        } else {
            error = true;
            importResults.addErrorLine(line, "ESPÈCE DU CYCLE DE CULTURE DU SYNTHÉTISÉ IGNORÉE!, la colonne 'croppingplanspeciescode' n'est pas renseignée !");
        }

        return error;
    }

    protected boolean validSpeciesFromExistingValues(ImportResults importResults, long line, PracticedCropCycleSpeciesImportDto dto, boolean isValidSpecies) {
        if (!isValidSpecies) {

            if (StringUtils.isNotBlank(dto.getPracticedSystemCampaigns())) {
                Set<Integer> intCampaigns = CommonService.GET_CAMPAIGNS_SET.apply(dto.getPracticedSystemCampaigns());
                try {
                    if (StringUtils.isNotBlank(dto.getCroppingPlanEntryCode())) {
                        Pair<CroppingPlanEntry, Map<String, CroppingPlanSpecies>> cropAndSpeciesByCodes = domainService.getEntryAndSpeciesFromCode(dto.getCroppingPlanEntryCode(), intCampaigns);
                        isValidSpecies = cropAndSpeciesByCodes.getValue().containsKey(dto.getCroppingPlanSpeciesCode());
                    } else {
                        isValidSpecies = false;
                    }
                } catch (Exception e) {
                    isValidSpecies = false;
                }
            } else {
                importResults.addErrorLine(line, String.format("ESPÈCE DU CYCLE DE CULTURE DU SYNTHÉTISÉ IGNORÉE!, aucune campagne n'est définie sur le système synthétisé '%s', les espèces ne peuvent être retrouvées", dto.getPracticedPerennialCycleId()));
                isValidSpecies = false;
            }
        }
        return isValidSpecies;
    }

    protected boolean validSpeciesFromPz0ImportedValues(PracticedCropCycleSpeciesImportDto dto) {
        DomainAndDependencies domainAndDependencies = dto.getDomainAndDependencies();
        String domainsCSVCode = domainAndDependencies.getCsvCode();
        List<DomainAndDependencies> domainsAndDependencies = (List<DomainAndDependencies>) pz0CodeToObject.get(Domain.class, domainsCSVCode);

        boolean isValidSpecies = false;

        for (DomainAndDependencies domainsAndDependency : domainsAndDependencies) {
            if (domainsAndDependency != null && !isValidSpecies) {
                Map<String, CroppingPlanSpeciesDto> speciesDtosByCodes = domainAndDependencies.getCroppingPlanSpeciesByCsvCode();
                CroppingPlanSpeciesDto croppingPlanSpeciesDto = speciesDtosByCodes.get(dto.getCroppingPlanSpeciesCode());

                if (croppingPlanSpeciesDto != null) {
                    Map<String, CroppingPlanEntryDto> croppingPlanEntryDtosByPZ0Code = domainAndDependencies.getCroppingPlanEntryDtosByPZ0Code();
                    if (croppingPlanEntryDtosByPZ0Code != null) {
                        CroppingPlanEntryDto croppingPlanEntryDto = croppingPlanEntryDtosByPZ0Code.get(dto.getCroppingPlanEntryCode());
                        if (croppingPlanEntryDto != null) {
                            isValidSpecies = croppingPlanEntryDto.getSpecies().contains(croppingPlanSpeciesDto);
                        }
                    }
                }
            }
        }

        return isValidSpecies;
    }

    protected boolean validGraftCloneId(ImportResults importResults, long line, PracticedCropCycleSpecies practicedCropCycleSpecies, PracticedCropCycleSpeciesImportDto dto, boolean error) {
        String target = dto.getGraftCloneId();
        if (StringUtils.isNotBlank(target)) {
            RefClonePlantGrape graftClone = refClonePlantGrapesDao.forTopiaIdEquals(target).findUniqueOrNull();
            practicedCropCycleSpecies.setGraftClone(graftClone);
            if (graftClone == null) {
                error = true;
                importResults.addErrorLine(line, String.format("ESPÈCE DU CYCLE DE CULTURE DU SYNTHÉTISÉ IGNORÉE!, le clone de la greffe avec comme identifiant '%s' n'existe pas !", target));
            }
        }
        return error;
    }

    protected boolean validGraftSupportId(ImportResults importResults, long line, PracticedCropCycleSpecies practicedCropCycleSpecies, PracticedCropCycleSpeciesImportDto dto, boolean error) {
        String target = dto.getGraftSupportId();
        if (StringUtils.isNotBlank(target)) {
            RefVariete graftSupport = refVarieteDao.forTopiaIdEquals(target).findUniqueOrNull();
            practicedCropCycleSpecies.setGraftSupport(graftSupport);
            if (graftSupport == null) {
                error = true;
                importResults.addErrorLine(line, String.format("ESPÈCE DU CYCLE DE CULTURE DU SYNTHÉTISÉ IGNORÉE!, le porte-greffe (graftsupport) avec comme identifiant '%s' n'existe pas !", target));
            }
        }
        return error;
    }

    protected boolean validOverGraftDate(ImportResults importResults, long line, PracticedCropCycleSpecies practicedCropCycleSpecies, PracticedCropCycleSpeciesImportDto dto, boolean error) {
        String overGraftDate = dto.getOverGraftDate();
        if (StringUtils.isNotBlank(overGraftDate)) {
            String datePattern = getDatePatternForInput(overGraftDate);
            SimpleDateFormat format = new SimpleDateFormat(datePattern);
            try {
                Date date = format.parse(overGraftDate);
                practicedCropCycleSpecies.setOverGraftDate(date);
            } catch (ParseException e) {
                error = true;
                importResults.addErrorLine(line, String.format("ESPÈCE DU CYCLE DE CULTURE DU SYNTHÉTISÉ IGNORÉE!, la date de sur-greffage %s n'est pas valide.", overGraftDate));
            }

        }
        return error;
    }

    @Override
    public void init(ServiceContext serviceContext) {
        super.init(serviceContext);
        refClonePlantGrapesDao = serviceContext.getPersistenceContext().getRefClonePlantGrapeDao();
        refVarieteDao = serviceContext.getPersistenceContext().getRefVarieteDao();

        domainService = getServiceFactory().newService(DomainService.class);
    }
}
