package fr.inra.agrosyst.services.pz0import.plot;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: PlotImporter.java 5036 2015-07-10 21:45:54Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.2/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/plot/PlotImporter.java $
 * %%
 * Copyright (C) 2013 - 2015 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Strings;
import fr.inra.agrosyst.api.entities.BufferStrip;
import fr.inra.agrosyst.api.entities.DomainTopiaDao;
import fr.inra.agrosyst.api.entities.FrostProtectionType;
import fr.inra.agrosyst.api.entities.Ground;
import fr.inra.agrosyst.api.entities.GroundTopiaDao;
import fr.inra.agrosyst.api.entities.GrowingSystemTopiaDao;
import fr.inra.agrosyst.api.entities.HosesPositionning;
import fr.inra.agrosyst.api.entities.IrrigationSystemType;
import fr.inra.agrosyst.api.entities.MaxSlope;
import fr.inra.agrosyst.api.entities.Plot;
import fr.inra.agrosyst.api.entities.PlotTopiaDao;
import fr.inra.agrosyst.api.entities.PompEngineType;
import fr.inra.agrosyst.api.entities.SolWaterPh;
import fr.inra.agrosyst.api.entities.WaterFlowDistance;
import fr.inra.agrosyst.api.entities.referential.RefLocation;
import fr.inra.agrosyst.api.entities.referential.RefLocationTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefSolProfondeurIndigo;
import fr.inra.agrosyst.api.entities.referential.RefSolProfondeurIndigoTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefSolTextureGeppa;
import fr.inra.agrosyst.api.entities.referential.RefSolTextureGeppaTopiaDao;
import fr.inra.agrosyst.api.services.plot.PlotService;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.domains.DomainAndDependencies;
import fr.inra.agrosyst.api.services.pz0.growingSystem.GrowingSystemAndDependencies;
import fr.inra.agrosyst.api.services.pz0.plot.PlotAndDependencies;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.services.pz0import.AbstractCSVImporter;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.InputStream;
import java.util.Map;

/**
 * Created by davidcosse on 29/01/15.
 */
public class PlotImporter extends AbstractCSVImporter {

    private static final Log log = LogFactory.getLog(PlotImporter.class);

    protected PlotService plotService;

    protected DomainTopiaDao domainDao;
    protected GrowingSystemTopiaDao growingSystemDao;
    protected RefLocationTopiaDao locationDao;
    protected RefSolTextureGeppaTopiaDao refSolTextureGeppaDao;
    protected RefSolProfondeurIndigoTopiaDao refSolProfondeurIndigoDao;
    protected GroundTopiaDao groundDao;
    protected PlotTopiaDao plotTopiaDao;

    @Override
    public ImportResults importFromStream(InputStream is, Map<String, EntityAndDependencies> entityAndDependenciesByCsvId) {
        ImportResults importResults = new ImportResults(Plot.class);

        PlotImportModel model = new PlotImportModel();
        // récupère le DTO
        Import<PlotImportDto> importer = Import.newImport(model, is);

        // match the first csv line number with data (not header).
        long line = FIRST_LINE_NUMBER;
        for (PlotImportDto dto : importer) {
            boolean error = false;
            Plot plot = plotTopiaDao.forTopiaIdEquals(dto.getId()).findUniqueOrNull();

            if (plot != null) {
                error = true;
                importResults.addErrorLine(line, String.format("PARCELLE IGNORÉE!, une parcelle avec l'identifiant %s existe déjà", dto.getId()));
            }

            if (!error) {
                Binder<PlotImportDto, Plot> practicedPlotBinder = BinderFactory.newBinder(PlotImportDto.class, Plot.class);
                plot = plotService.getPlot(null);
                practicedPlotBinder.copyExcluding(dto, plot,
                        Plot.PROPERTY_TOPIA_ID,
                        Plot.PROPERTY_LOCATION,
                        Plot.PROPERTY_SURFACE_TEXTURE,
                        Plot.PROPERTY_SUB_SOIL_TEXTURE,
                        Plot.PROPERTY_SOL_DEPTH,
                        Plot.PROPERTY_WATER_FLOW_DISTANCE,
                        Plot.PROPERTY_IRRIGATION_SYSTEM_TYPE,
                        Plot.PROPERTY_POMP_ENGINE_TYPE,
                        Plot.PROPERTY_HAIL_PROTECTION,
                        Plot.PROPERTY_FROST_PROTECTION_TYPE,
                        Plot.PROPERTY_SOL_WATER_PH,
                        Plot.PROPERTY_MAX_SLOPE,
                        Plot.PROPERTY_BUFFER_STRIP);

                PlotAndDependencies plotAndDependencies = new PlotAndDependencies(plot);

                // valid required fields
                error = validPlotName(importResults, line, plot, false);
                error = validDomainAffiliation(importResults, line, dto, plotAndDependencies, entityAndDependenciesByCsvId, error);
                error = validWaterFlowDistance(importResults, line, dto, plot, error);
                error = validBufferStrip(importResults, line, dto, plot, error);
                error = validMaxSlope(importResults, line, dto, plot, error);
                error = validCode(importResults, line, dto, plot, error);
                error = validValidated(importResults, line, dto, plot, error);
                error = validDeletedZones(importResults, line, dto, plot, error);

                // valid optional entities
                error = validGrowingSystemAffiliation(importResults, line, dto, plotAndDependencies, entityAndDependenciesByCsvId, error);
                error = validLocation(importResults, line, dto, plotAndDependencies, error);
                error = validSurfaceTexture(importResults, line, dto, plotAndDependencies, error);
                error = validSubSoilTexture(importResults, line, dto, plotAndDependencies, error);
                error = validSolDepth(importResults, line, dto, plotAndDependencies, error);
                error = validGround(importResults, line, dto, plotAndDependencies, error);

                // valid optional enums
                error = validIrrigationSystemType(importResults, line, dto, plot, error);
                error = validPompEngineType(importResults, line, dto, plot, error);
                error = validHosesPositionning(importResults, line, dto, plot, error);
                error = validFrostProtectionType(importResults, line, dto, plot, error);
                error = validSolWaterPh(importResults, line, dto, plot, error);

                if (!error) {
                    importResults.addInfoLine(line, ", " + "PARCELLE VALIDÉE, csvid: " + dto.getId());
                    importResults.increaseAddedRecords();
                } else {
                    importResults.increaseIgnoredRecords();
                    importResults.addErrorLine(line, "PARCELLE IGNORÉE csvid:" + dto.getId());
                }
                importResults.addEntity(dto.getId(), plotAndDependencies);
            }

            line++;
        }
        return importResults;
    }

    protected boolean validPlotName(ImportResults results, long line, Plot plot, boolean error) {

        if (StringUtils.isBlank(plot.getName())) {
            results.addErrorLine(line, "PARCELLE IGNORÉE!, la colonne 'name' n'est pas renseignée !");
            error = true;
        }

        return error;
    }

    protected boolean validDomainAffiliation(ImportResults results, long line, PlotImportDto dto, PlotAndDependencies plotAndDependencies, Map<String, EntityAndDependencies> domainsByCsvIds, boolean error) {

        if (StringUtils.isNotBlank(dto.getDomain())) {

            DomainAndDependencies domainAndDependencies = (DomainAndDependencies) domainsByCsvIds.get(dto.getDomain());
            boolean found = domainAndDependencies != null;

            if (!found) {
                found = domainDao.forTopiaIdEquals(dto.getDomain()).exists();
            }
            if (!found) {
                results.addErrorLine(line, String.format("PARCELLE IGNORÉE!, aucun domaine n'est retrouvé avec l'identifiant suivant: %s", dto.getDomain()));
                error = true;
            } else {
                plotAndDependencies.setDomainAndDependencies(domainAndDependencies);
            }

        } else {
            results.addErrorLine(line, String.format("PARCELLE IGNORÉE!, la colonne 'domain' n'est pas renseignée pour la parcelle  \"%s\"!", dto.getName()));
            error = true;
        }

        return error;
    }

    protected boolean validLocation(ImportResults result, long line, PlotImportDto dto, PlotAndDependencies plotAndDependencies, boolean error) {
        if (StringUtils.isNotBlank(dto.getLocationId())) {
            RefLocation location = locationDao.forTopiaIdEquals(dto.getLocationId()).findUniqueOrNull();
            if (location == null) {
                result.addErrorLine(line, String.format("PARCELLE IGNORÉE! l'objet RefLocation de topiaId %s n'existe pas en base de données pour la parcelle \"%s\".", dto.getLocationId(), dto.getName()));
                error = true;
            } else {
                plotAndDependencies.setLocationId(dto.getLocationId());
            }
        }
        return error;
    }

    protected boolean validGrowingSystemAffiliation(ImportResults results, long line, PlotImportDto dto, PlotAndDependencies plotAndDependencies, Map<String, EntityAndDependencies> growingSystemsByCsvIds, boolean error) {

        if (StringUtils.isNotBlank(dto.getGrowingSystem())) {

            GrowingSystemAndDependencies growingSystemAndDependencies = (GrowingSystemAndDependencies) growingSystemsByCsvIds.get(dto.getGrowingSystem());
            boolean found = growingSystemAndDependencies != null;
            if (!found) {
                found = growingSystemDao.forTopiaIdEquals(dto.getGrowingSystem()).exists();
            }
            if (!found) {
                results.addErrorLine(line, String.format("PARCELLE IGNORÉE!, aucun système de culture n'est retrouvé avec l'identifiant suivant: %s", dto.getGrowingSystem()));
                error = true;
            } else {
                plotAndDependencies.setGrowingSystemAndDependencies(growingSystemAndDependencies);
            }

        }

        return error;
    }

    protected boolean validSurfaceTexture(ImportResults result, long line, PlotImportDto dto, PlotAndDependencies plotAndDependencies, boolean error) {
        if (StringUtils.isNotBlank(dto.getSurfaceTextureId())) {
            RefSolTextureGeppa surfaceTexture = refSolTextureGeppaDao.forTopiaIdEquals(dto.getSurfaceTextureId()).findUniqueOrNull();
            if (surfaceTexture == null) {
                result.addErrorLine(line, String.format("PARCELLE IGNORÉE! l'objet RefSolTextureGeppa référencé sur la colonne 'surfacetexture' de topiaId %s n'existe pas en base de données pour la parcelle \"%s\".", dto.getLocationId(), dto.getName()));
                error = true;
            }
            plotAndDependencies.setSelectedSurfaceTextureId(dto.getSurfaceTextureId());
        }
        return error;
    }

    protected boolean validSubSoilTexture(ImportResults result, long line, PlotImportDto dto, PlotAndDependencies plotAndDependencies, boolean error) {
        if (StringUtils.isNotBlank(dto.getSubSoilTextureId())) {
            RefSolTextureGeppa subSolTexture = refSolTextureGeppaDao.forTopiaIdEquals(dto.getSubSoilTextureId()).findUniqueOrNull();
            if (subSolTexture == null) {
                result.addErrorLine(line, String.format("PARCELLE IGNORÉE! l'objet RefSolTextureGeppa référencé sur la colonne 'subsoiltexture' de topiaId %s n'existe pas en base de données pour la parcelle \"%s\".", dto.getLocationId(), dto.getName()));
                error = true;
            }
            plotAndDependencies.setSelectedSubSoilTextureId(dto.getSubSoilTextureId());
        }
        return error;
    }

    protected boolean validSolDepth(ImportResults result, long line, PlotImportDto dto, PlotAndDependencies plotAndDependencies, boolean error) {
        if (StringUtils.isNotBlank(dto.getSolDepthId())) {
            RefSolProfondeurIndigo solDepth = refSolProfondeurIndigoDao.forTopiaIdEquals(dto.getSolDepthId()).findUniqueOrNull();
            if (solDepth == null) {
                result.addErrorLine(line, String.format("PARCELLE IGNORÉE! l'objet RefSolProfondeurIndigo référencé sur la colonne 'soldepth' de topiaId %s n'existe pas en base de données pour la parcelle \"%s\".", dto.getLocationId(), dto.getName()));
                error = true;
            }
            plotAndDependencies.setSelectedSolDepthId(dto.getSolDepthId());
        }
        return error;
    }

    protected boolean validGround(ImportResults result, long line, PlotImportDto dto, PlotAndDependencies plotAndDependencies, boolean error) {
        if (StringUtils.isNotBlank(dto.getGround())) {
            if (plotAndDependencies.getDomainAndDependencies() != null) {
                Ground ground = plotAndDependencies.getDomainAndDependencies().getGroundForCsvId(dto.getGround());
                if (ground == null) {
                    result.addErrorLine(line, String.format("PARCELLE IGNORÉE! le sol dont l'identifiant est '%s' n'existe pas sur le domaine '%s'.", dto.getGround(), plotAndDependencies.getDomainAndDependencies().getCsvId()));
                    error = true;
                } else {
                    plotAndDependencies.setSelectedSolId(dto.getGround());
                }
            } else {
                result.addErrorLine(line, String.format("PARCELLE IGNORÉE! le sol dont l'identifiant est '%s' n'a pu etre retrouvé, le domaine du sol n'a pu etre retrouvé.", dto.getGround()));
                error = true;
            }

        }
        return error;
    }


    protected boolean validWaterFlowDistance(ImportResults result, long line, PlotImportDto dto, Plot plot, boolean error) {

        String value = dto.getWaterFlowDistanceImportedValue();
        WaterFlowDistance waterFlowDistance = null;
        if (StringUtils.isNotBlank(value)) {
            try {
                waterFlowDistance = WaterFlowDistance.valueOf(value);
            } catch (IllegalArgumentException e) {
                result.addErrorLine(line, String.format("PARCELLE IGNORÉE! Distance à l'eau, non supportée : %s", value));
                error = true;
            }
        } else {
            waterFlowDistance = WaterFlowDistance.MORE_THAN_TEN;
        }
        plot.setWaterFlowDistance(waterFlowDistance);
        return error;
    }

    protected boolean validIrrigationSystemType(ImportResults result, long line, PlotImportDto dto, Plot plot, boolean error) {

        String value = dto.getIrrigationSystemTypeImportedValue();
        IrrigationSystemType irrigationSystemType = null;

        if (!Strings.isNullOrEmpty(value)) {
            try {
                irrigationSystemType = IrrigationSystemType.valueOf(value);
            } catch (IllegalArgumentException e) {
                result.addErrorLine(line, String.format("PARCELLE IGNORÉE! Système d'irrigation, non supportée : %s", value));
                error = true;
            }
        }
        plot.setIrrigationSystemType(irrigationSystemType);
        return error;
    }

    protected boolean validPompEngineType(ImportResults result, long line, PlotImportDto dto, Plot plot, boolean error) {
        String value = dto.getPompEngineTypeImportedValue();
        PompEngineType pompEngineType = null;
        if (!Strings.isNullOrEmpty(value)) {
            try {
                pompEngineType = PompEngineType.valueOf(value);
            } catch (IllegalArgumentException e) {
                result.addErrorLine(line, String.format("PARCELLE IGNORÉE! Type de moteur de pompe, non supportée : %s", value));
                error = true;
            }
        }
        plot.setPompEngineType(pompEngineType);
        return error;
    }

    protected boolean validHosesPositionning(ImportResults result, long line, PlotImportDto dto, Plot plot, boolean error) {
        String value = dto.getHosesPositionningImportedValue();
        HosesPositionning hosesPositionning = null;

        if (!Strings.isNullOrEmpty(value)) {
            try {
                hosesPositionning = HosesPositionning.valueOf(value);
            } catch (IllegalArgumentException e) {
                result.addErrorLine(line, String.format("PARCELLE IGNORÉE! Positionnement des tuyaux d'arrosage, non supportée : %s", value));
                error = true;
            }
        }
        plot.setHosesPositionning(hosesPositionning);

        return error;
    }

    protected boolean validFrostProtectionType(ImportResults result, long line, PlotImportDto dto, Plot plot, boolean error) {
        String value = dto.getFrostProtectionTypeImportedValue();
        FrostProtectionType frostProtectionType = null;

        if (!Strings.isNullOrEmpty(value)) {
            try {
                frostProtectionType = FrostProtectionType.valueOf(value);
            } catch (IllegalArgumentException e) {
                result.addErrorLine(line, String.format("PARCELLE IGNORÉE! Type de moteur de pompe, non supportée : %s", value));
                error = true;
            }
        }
        plot.setFrostProtectionType(frostProtectionType);
        return error;
    }

    protected boolean validSolWaterPh(ImportResults result, long line, PlotImportDto dto, Plot plot, boolean error) {
        String value = dto.getSolWaterPhImportedValue();
        SolWaterPh solWaterPh = null;

        if (!Strings.isNullOrEmpty(value)) {
            try {
                solWaterPh = getSolWaterPh(value);
                if (solWaterPh == null) {
                    result.addErrorLine(line, String.format("PARCELLE IGNORÉE! PH de l'eau, non supportée : %s", value));
                    error = true;
                }
            } catch (NumberFormatException e) {
                result.addErrorLine(line, String.format("PARCELLE IGNORÉE! PH de l'eau, non supportée : %s", value));
                error = true;
            } catch (IllegalArgumentException e) {
                result.addErrorLine(line, String.format("PARCELLE IGNORÉE! PH de l'eau, non supportée : %s", value));
                error = true;
            }
        }
        plot.setSolWaterPh(solWaterPh);
        return error;
    }

    protected boolean validMaxSlope(ImportResults result, long line, PlotImportDto dto, Plot plot, boolean error) {

        String value = dto.getMaxSlopeImportedValue();
        MaxSlope maxSlope = null;

        if (!Strings.isNullOrEmpty(value)) {
            try {
                maxSlope = MaxSlope.valueOf(value);
            } catch (IllegalArgumentException e) {
                result.addErrorLine(line, String.format("PARCELLE IGNORÉE! Pente maximum, non supportée : %s", value));
                error = true;
            }
        } else {
            maxSlope = MaxSlope.ZERO_TO_TWO;
        }
        plot.setMaxSlope(maxSlope);
        return error;

    }

    protected boolean validCode(ImportResults importResults, long line, PlotImportDto dto, Plot plot, boolean error) {
            if (StringUtils.isBlank(dto.getCode())) {
                importResults.addErrorLine(line, String.format("PARCELLE IGNORÉE! La colonne 'code' n'est pas renseignée pour la parcelle %s!", dto.getName()));
                error = true;
            } else {
                plot.setCode(csvCodeToAgrosystCode(Plot.class, dto.getCode()));
            }
            return error;

    }

    protected boolean validValidated(ImportResults importResults, long line, PlotImportDto dto, Plot plot, boolean error) {
        if (dto.getValidated() == null) {
            importResults.addErrorLine(line, String.format("PARCELLE IGNORÉE! La colonne 'validated' n'est pas renseignée pour la parcelle %s!", dto.getName()));
            error = true;
        } else {
            plot.setValidated(dto.getValidated());
        }
        return error;

    }

    protected boolean validDeletedZones(ImportResults importResults, long line, PlotImportDto dto, Plot plot, boolean error) {
        if (dto.getDeletedZones() == null) {
            importResults.addErrorLine(line, String.format("PARCELLE IGNORÉE! La colonne 'deletedzones' n'est pas renseignée pour la parcelle %s!", dto.getName()));
            error = true;
        } else {
            plot.setDeletedZones(dto.getDeletedZones());
        }
        return error;

    }

    protected boolean validBufferStrip(ImportResults result, long line, PlotImportDto dto, Plot plot, boolean error) {
        String value = dto.getBufferStripImportedValue();
            BufferStrip bufferStrip = null;

            if (!Strings.isNullOrEmpty(value)) {
                try {
                    bufferStrip = BufferStrip.valueOf(value);
                } catch (IllegalArgumentException e) {
                    result.addErrorLine(line, String.format("PARCELLE IGNORÉE! Bande tampon, non supportée : %s", value));
                    error = true;
                }
            } else {
                bufferStrip = BufferStrip.NONE;
            }
        plot.setBufferStrip(bufferStrip);
            return error;
    }

    @Override
    public void init(ServiceContext serviceContext) {
        super.init(serviceContext);

        plotService = getServiceFactory().newService(PlotService.class);

        domainDao = getPersistenceContext().getDomainDao();
        growingSystemDao = getPersistenceContext().getGrowingSystemDao();
        locationDao = getPersistenceContext().getRefLocationDao();
        refSolTextureGeppaDao = getPersistenceContext().getRefSolTextureGeppaDao();
        refSolProfondeurIndigoDao = getPersistenceContext().getRefSolProfondeurIndigoDao();
        groundDao = getPersistenceContext().getGroundDao();
        plotTopiaDao = getPersistenceContext().getPlotDao();
    }


}
