package fr.inra.agrosyst.services.pz0import.input;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: InputImporter.java 5118 2015-10-29 14:26:43Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.2/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/input/InputImporter.java $
 * %%
 * Copyright (C) 2013 - 2015 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.collect.Sets;

import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.action.AbstractAction;
import fr.inra.agrosyst.api.entities.action.AbstractInput;
import fr.inra.agrosyst.api.entities.action.BiologicalControlAction;
import fr.inra.agrosyst.api.entities.action.BiologicalProductInput;
import fr.inra.agrosyst.api.entities.action.BiologicalProductInputImpl;
import fr.inra.agrosyst.api.entities.action.HarvestingAction;
import fr.inra.agrosyst.api.entities.action.IrrigationAction;
import fr.inra.agrosyst.api.entities.action.MaintenancePruningVinesAction;
import fr.inra.agrosyst.api.entities.action.MineralFertilizersSpreadingAction;
import fr.inra.agrosyst.api.entities.action.MineralProductInput;
import fr.inra.agrosyst.api.entities.action.MineralProductInputImpl;
import fr.inra.agrosyst.api.entities.action.MineralProductUnit;
import fr.inra.agrosyst.api.entities.action.OrganicFertilizersSpreadingAction;
import fr.inra.agrosyst.api.entities.action.OrganicProductInput;
import fr.inra.agrosyst.api.entities.action.OrganicProductInputImpl;
import fr.inra.agrosyst.api.entities.action.OrganicProductUnit;
import fr.inra.agrosyst.api.entities.action.OtherAction;
import fr.inra.agrosyst.api.entities.action.OtherProductInput;
import fr.inra.agrosyst.api.entities.action.OtherProductInputImpl;
import fr.inra.agrosyst.api.entities.action.PesticideProductInput;
import fr.inra.agrosyst.api.entities.action.PesticideProductInputImpl;
import fr.inra.agrosyst.api.entities.action.PesticidesSpreadingAction;
import fr.inra.agrosyst.api.entities.action.PhytoProductInput;
import fr.inra.agrosyst.api.entities.action.PhytoProductUnit;
import fr.inra.agrosyst.api.entities.action.SeedingAction;
import fr.inra.agrosyst.api.entities.action.SeedingActionSpecies;
import fr.inra.agrosyst.api.entities.action.SeedingProductInput;
import fr.inra.agrosyst.api.entities.action.SeedingProductInputImpl;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystem;
import fr.inra.agrosyst.api.entities.referential.RefActaDosageSPC;
import fr.inra.agrosyst.api.entities.referential.RefActaTraitementsProduit;
import fr.inra.agrosyst.api.entities.referential.RefActaTraitementsProduitTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefFertiMinUNIFA;
import fr.inra.agrosyst.api.entities.referential.RefFertiMinUNIFATopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefFertiOrga;
import fr.inra.agrosyst.api.entities.referential.RefFertiOrgaTopiaDao;
import fr.inra.agrosyst.api.services.domain.CroppingPlanSpeciesDto;
import fr.inra.agrosyst.api.services.itk.SpeciesStadeDto;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.domains.DomainAndDependencies;
import fr.inra.agrosyst.api.services.pz0.effective.Pz0EffectiveIntervention;
import fr.inra.agrosyst.api.services.pz0.plot.PlotAndDependencies;
import fr.inra.agrosyst.api.services.pz0.practicedSystem.PracticedSystemAndDependencies;
import fr.inra.agrosyst.api.services.pz0.practicedSystem.Pz0PracticedIntervention;
import fr.inra.agrosyst.api.services.referential.ReferentialService;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.services.pz0import.AbstractCSVImporter;
import fr.inra.agrosyst.services.pz0import.input.dto.InputImportDto;
import fr.inra.agrosyst.services.pz0import.input.model.InputModel;
import fr.inra.agrosyst.services.pz0import.practicedSystem.PracticedSystemImporter;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.InputStream;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Created by davidcosse on 24/02/15.
 **/
public class InputImporter extends AbstractCSVImporter {

    private static final Log log = LogFactory.getLog(InputImporter.class);

    protected RefFertiMinUNIFATopiaDao mineralProductDao;
    protected RefActaTraitementsProduitTopiaDao phytoProductDao;
    protected RefFertiOrgaTopiaDao organicProductDao;
    protected ReferentialService referentialService;
    protected Map<AgrosystInterventionType, List<String>> actaTreatmentProductTypes;

    public void setRefFertiMinUNIFATopiaDao(ReferentialService referentialService) {
        this.referentialService = referentialService;
    }

    protected Map<AgrosystInterventionType, List<String>> getActaTreatmentProductTypes() {
        if (actaTreatmentProductTypes == null || actaTreatmentProductTypes.isEmpty()) {
            actaTreatmentProductTypes = referentialService.getAllActiveActaTreatmentProductTypes();
        }
        return actaTreatmentProductTypes;
    }


    @Override
    public ImportResults importFromStream(InputStream is, Map<String, EntityAndDependencies> entitiesByCsvId) {
        ImportResults importResults = new ImportResults(AbstractInput.class);

        InputModel model = new InputModel();
        // récupère le DTO
        Import<InputImportDto> importer = Import.newImport(model, is);

        // match the first csv line number with data (not header).
        long line = FIRST_LINE_NUMBER;
        for (InputImportDto dto : importer) {

            // valid required fields
            AbstractInput input = createInput(dto, importResults, line);
            // error messages are defined in createInput()
            Boolean error = (input == null);

            error = validActionAndSpecificInputData(importResults, input, line, dto, error);

            if (!error) {
                Pz0PracticedIntervention practicedInterventionDto = (Pz0PracticedIntervention) pz0IdToObject.get(AbstractAction.class, Pz0PracticedIntervention.class, dto.getPz0ActionId());
                Pz0EffectiveIntervention effectiveInterventionDto = (Pz0EffectiveIntervention) pz0IdToObject.get(AbstractAction.class, Pz0EffectiveIntervention.class, dto.getPz0ActionId());

                if (practicedInterventionDto != null) {
                    practicedInterventionDto.getPracticedInterventionDto().addInput(input);
                    importResults.addInfoLine(line, ", " + String.format("AJOUT DE L'INTRANT '%s' VALIDÉ", dto.getId()));
                    importResults.increaseAddedRecords();
                    importResults.addEntity(dto.getId(), new EntityAndDependencies(input));
                } else if(effectiveInterventionDto != null) {
                    effectiveInterventionDto.getEffectiveInterventionDto().addInput(input);
                    importResults.addInfoLine(line, ", " + String.format("AJOUT DE L'INTRANT '%s' VALIDÉ", dto.getId()));
                    importResults.increaseAddedRecords();
                    importResults.addEntity(dto.getId(), new EntityAndDependencies(input));
                } else {
                    importResults.increaseIgnoredRecords();
                    importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT '%s' IGNORÉ, l'intervention assosiée à l'action dont l'identifiant est '%s' n'a pas été retrouvée.", dto.getId(), dto.getPz0ActionId()));
                }
            } else {
                importResults.increaseIgnoredRecords();
                importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT '%s' IGNORÉ", dto.getId()));
            }
            line++;
        }
        return importResults;
    }

    protected Boolean validPhytoProductInput(PhytoProductInput input, InputImportDto from, Set<String> speciesCode, ImportResults importResults, long line,  Boolean error) {

        Pair<Boolean, Set<String>> resultAndSpeciesIds = getRefEspeceIds(importResults, from.getPz0ActionId(), speciesCode, line, error);
        error = resultAndSpeciesIds.getLeft();
        Set<String> speciesIds = resultAndSpeciesIds.getRight();

        RefActaDosageSPC dose = null;
        RefActaTraitementsProduit phytoProduct = null;
        PhytoProductUnit phytoProductUnit = null;

        if (StringUtils.isNotBlank(from.getPhytoProductId())) {
            phytoProduct = phytoProductDao.forTopiaIdEquals(from.getPhytoProductId()).findUniqueOrNull();

            if (phytoProduct == null) {
                error = true;
                importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT '%s' IGNORÉ!, le produit avec comme identifiant '%s' n'a pas été retrouvé.", from.getId(), from.getPhytoProductId()));
            }

        } else {
            error = true;
            importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT '%s' IGNORÉ!, la colonne 'phytoproduct' n'est pas renseignée.",from.getId()));
        }

        if (StringUtils.isNotBlank(from.getPhytoProductUnitName())) {
            try {
                phytoProductUnit = PhytoProductUnit.valueOf(from.getPhytoProductUnitName());
                PhytoProductUnit refPhytoProductUnit = null;

                if (CollectionUtils.isNotEmpty(speciesIds) && phytoProduct != null) {
                    dose = referentialService.computeActaReferenceDose(phytoProduct.getId_produit(), speciesIds);
                }

                if (dose != null) {
                    refPhytoProductUnit = dose.getDosage_spc_unite();
                }
                if (refPhytoProductUnit != null && phytoProductUnit != null && !phytoProductUnit.equals(refPhytoProductUnit)) {
                    error = true;
                    String allSpeciesIds= null;
                    for (String speciesId : speciesIds) {
                        allSpeciesIds = allSpeciesIds == null ? speciesId : allSpeciesIds + " - " +speciesId;
                    }
                    importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT '%s' IGNORÉ!, l'unité de produit phytosanitaire '%s' ne correspond pas avec celle de la dose de référence '%s' obtenue à partir du produit avec comme id_produit:'%s' et les espèces suivantes: '%s'.", from.getInputTypeName(), phytoProductUnit.name(), refPhytoProductUnit.name(), phytoProduct.getId_produit(), allSpeciesIds));
                }

            } catch (IllegalArgumentException e) {
                error = true;
                importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT '%s' IGNORÉ!, unité de produit phytosanitaire non reconnue: '%s'.",from.getInputTypeName(), from.getPhytoProductUnitName()));
            }

        }

        if (input != null) {
            input.setPhytoProduct(phytoProduct);
            input.setPhytoProductUnit(phytoProductUnit);
        }

        return error;
    }

    protected AbstractAction validAndGetAction(String actionId, String inputId, String inputTypeName, ImportResults importResults, Long line) {
        AbstractAction action = null;
        if (StringUtils.isNotBlank(actionId)) {
            action = (AbstractAction) pz0IdToObject.get(AbstractAction.class, actionId);
            if (action == null) {
                importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT '%s' IGNORÉ!, l'action avec comme identifiant '%s' n'a pas été retrouvée.", inputTypeName, actionId));
            }
        } else {
            importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT IGNORÉ!, aucune action renseignée pour l'intrant '%s' de type %s.", inputId, inputTypeName));
        }
        return action;
    }


    protected AbstractInput createInput(InputImportDto from, ImportResults importResults, long line) {
        AbstractInput input = null;
        if (StringUtils.isNotBlank(from.getInputTypeName())) {
            try {
                AgrosystInterventionType type = AgrosystInterventionType.valueOf(from.getInputTypeName());
                switch (type) {
                    case APPLICATION_DE_PRODUITS_FERTILISANTS_MINERAUX:
                        input = new MineralProductInputImpl();
                        ((MineralProductInput)input).setPhytoEffect(from.isPhytoEffect());
                        break;
                    case EPANDAGES_ORGANIQUES:
                        input = new OrganicProductInputImpl();
                        ((OrganicProductInput)input).setN(from.getN());
                        ((OrganicProductInput)input).setP2O5(from.getP2o5());
                        ((OrganicProductInput)input).setK2O(from.getK2o());
                        break;
                    case APPLICATION_DE_PRODUITS_PHYTOSANITAIRES:
                        input = new PesticideProductInputImpl();
                        ((PesticideProductInput)input).setProductType(from.getProductType());
                        break;
                    case LUTTE_BIOLOGIQUE:
                        input = new BiologicalProductInputImpl();
                        break;
                    case SEMIS:
                        input = new SeedingProductInputImpl();
                        break;
                    case AUTRE:
                        input = new OtherProductInputImpl();
                        ((OtherProductInput)input).setOtherProductQtUnit(from.getOtherProductQtUnit());
                        break;
                    default:
                        importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT IGNORÉ!, type d'intrant non reconnu '%s'.", from.getInputTypeName()));
                }
                if (input != null) {
                    // bind abstact input data to input
                    Binder<InputImportDto, AbstractInput> abstractInputBinder = BinderFactory.newBinder(InputImportDto.class, AbstractInput.class);
                    abstractInputBinder.copy(from, input);
                    input.setInputType(type);
                }
            } catch(IllegalArgumentException e){
                importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT IGNORÉ!, type d'intrant non reconnu: '%s'.", from.getInputTypeName()));
            }
        } else {
            importResults.addErrorLine(line, "AJOUT DE L'INTRANT IGNORÉ!, la colonne 'inputtype' n'est pas renseignée.");
        }
        return input;
    }

    protected Boolean validActionAndSpecificInputData(ImportResults importResults, AbstractInput input, long line, InputImportDto from, Boolean error) {

        if (input != null) {
            AgrosystInterventionType type = input.getInputType();
            String expectedObject = "?";
            String presentObject = "?";
            if (type != null) {
                AbstractAction action = validAndGetAction(from.getPz0ActionId(), from.getId(), from.getInputTypeName(), importResults, line);
                if (action == null) {
                    // message are defined in validAndGetAction()
                    error = true;
                } else {
                    String actionType = action.getMainAction() != null ? action.getMainAction().getIntervention_agrosyst().name() : "?";
                    try {
                        if(AgrosystInterventionType.APPLICATION_DE_PRODUITS_FERTILISANTS_MINERAUX.equals(type)) {
                            if (StringUtils.isNotBlank(from.getMineralProductId())) {
                                expectedObject = "MineralProductInput";
                                presentObject = input.getClass().getSimpleName();
                                MineralProductInput mineralProductInput = (MineralProductInput) input;

                                expectedObject = "MineralFertilizersSpreadingAction";
                                presentObject = action.getClass().getSimpleName();
                                mineralProductInput.setMineralFertilizersSpreadingAction((MineralFertilizersSpreadingAction) action);

                                RefFertiMinUNIFA mineralProduct = (RefFertiMinUNIFA) pz0IdToObject.get(RefFertiMinUNIFA.class, from.getMineralProductId());
                                if (mineralProduct == null) {
                                    mineralProduct = mineralProductDao.forTopiaIdEquals(from.getMineralProductId()).findUniqueOrNull();
                                }
                                if (mineralProduct != null) {
                                    mineralProductInput.setMineralProduct(mineralProduct);                                    
                                } else {
                                    error = true;
                                    importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT 'MINERAL' IGNORÉ!, le produit avec comme identifiant '%s' n'a pas été retrouvé.", from.getMineralProductId()));
                                }
                                
                                if (StringUtils.isNotBlank(from.getMineralProductUnitName())) {
                                    try {
                                        MineralProductUnit mineralProductUnit = MineralProductUnit.valueOf(from.getMineralProductUnitName());
                                        mineralProductInput.setMineralProductUnit(mineralProductUnit);
                                    } catch (IllegalArgumentException e) {
                                        error = true;
                                        importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT '%s' IGNORÉ!, unité de produit minéral non reconnue '%s'.", type, from.getMineralProductUnitName()));
                                    }
                                }
                                
                            } else {
                                error = true;
                                importResults.addErrorLine(line,"AJOUT DE L'INTRANT 'MINERAL' IGNORÉ!, la colonne 'mineralproduct' n'est pas renseignée.");
                            }


                        } else if (AgrosystInterventionType.AUTRE.equals(type)) {
                            expectedObject = "OtherProductInput";
                            presentObject = input.getClass().getSimpleName();
                            OtherProductInput otherProductInput = ((OtherProductInput)input);

                            if (StringUtils.isBlank(otherProductInput.getProductName())) {
                                error = true;
                                importResults.addErrorLine(line, "AJOUT DE L'INTRANT 'AUTRE' IGNORÉ!, la colonne 'productName' n'est pas renseignée.");
                            }

                            presentObject = action.getClass().getSimpleName();
                            if (StringUtils.isNotBlank(from.getOtherActionId())) {
                                expectedObject = "OtherAction";
                                otherProductInput.setOtherAction((OtherAction) action);
                            } else if (StringUtils.isNotBlank(from.getHarvestingActionId())) {
                                expectedObject = "HarvestingAction";
                                otherProductInput.setHarvestingAction((HarvestingAction) action);
                            } else if (StringUtils.isNotBlank(from.getMaintenancePruningVinesActionId())) {
                                expectedObject = "MaintenancePruningVinesAction";
                                otherProductInput.setMaintenancePruningVinesAction((MaintenancePruningVinesAction) action);
                            } else if (StringUtils.isNotBlank(from.getIrrigationActionId())) {
                                expectedObject = "IrrigationAction";
                                otherProductInput.setIrrigationAction((IrrigationAction) action);
                            } else if (StringUtils.isNotBlank(from.getMineralFertilizersSpreadingActionId())) {
                                expectedObject = "MineralFertilizersSpreadingAction";
                                otherProductInput.setMineralFertilizersSpreadingAction((MineralFertilizersSpreadingAction) action);
                            } else if (StringUtils.isNotBlank(from.getOrganicFertilizersSpreadingActionId())) {
                                expectedObject = "OrganicFertilizersSpreadingAction";
                                otherProductInput.setOrganicFertilizersSpreadingAction((OrganicFertilizersSpreadingAction) action);
                            } else if (StringUtils.isNotBlank(from.getPesticidesSpreadingActionId())) {
                                expectedObject = "PesticidesSpreadingAction";
                                otherProductInput.setPesticidesSpreadingAction((PesticidesSpreadingAction) action);
                            } else if (StringUtils.isNotBlank(from.getBiologicalControlActionId())) {
                                expectedObject = "BiologicalControlAction";
                                otherProductInput.setBiologicalControlAction((BiologicalControlAction) action);
                            } else if (StringUtils.isNotBlank(from.getSeedingActionId())) {
                                expectedObject = "SeedingAction";
                                otherProductInput.setSeedingAction((SeedingAction) action);
                            } else {
                                importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT 'AUTRE' IGNORÉ!, l' action avec comme identifiant '%s' est une action de type '%s', elle ne peux être associée à un intrant de type 'AUTRE'.", from.getPz0ActionId(), actionType));
                                error = true;
                            }
                        } else if (AgrosystInterventionType.APPLICATION_DE_PRODUITS_PHYTOSANITAIRES.equals(type)) {
                            if (StringUtils.isNotBlank(from.getPhytoProductId())) {
                                expectedObject = "PesticideProductInput";
                                presentObject = input.getClass().getSimpleName();
                                PesticideProductInput pesticideProductInput = (PesticideProductInput) input;

                                expectedObject = "PesticidesSpreadingAction";
                                presentObject = action.getClass().getSimpleName();
                                pesticideProductInput.setPesticidesSpreadingAction((PesticidesSpreadingAction) action);

                                Set<String> speciesCodes = getSpeciesCodesFromActionId(from.getPz0ActionId());
                                error = validPhytoProductInput(pesticideProductInput, from, speciesCodes, importResults, line, error);
                            } else {
                                error = true;
                                importResults.addErrorLine(line, "AJOUT DE L'INTRANT 'APPLICATION_DE_PRODUITS_PHYTOSANITAIRES' IGNORÉ!, la colonne 'phytoproduct' n'est pas renseignée.");
                            }

                        } else if (AgrosystInterventionType.SEMIS.equals(type)) {

                            if (StringUtils.isNotBlank(from.getSeedingActionId())) {
                                expectedObject = "SeedingProductInput";
                                presentObject = input.getClass().getSimpleName();
                                SeedingProductInput seedingProductInput = (SeedingProductInput) input;

                                expectedObject = "SeedingAction";
                                presentObject = action.getClass().getSimpleName();
                                seedingProductInput.setSeedingAction((SeedingAction) action);
                                Collection<SeedingActionSpecies> specieses = ((SeedingAction)action).getSeedingSpecies();

                                Set<String> speciesCode = Sets.newHashSet();
                                if (CollectionUtils.isNotEmpty(specieses)) {

                                    if (StringUtils.isNotBlank(seedingProductInput.getProductType())) {
                                        List<String> productTypes = actaTreatmentProductTypes.get(AgrosystInterventionType.SEMIS);
                                        Set<String> allowedProductTypes = Sets.newHashSet();
                                        for (SeedingActionSpecies species : specieses) {
                                            speciesCode.add(species.getSpeciesCode());
                                            if (species.isTreatment()) {
                                                //Fongicides
                                                allowedProductTypes.add(productTypes.get(0));
                                                //Insecticides
                                                allowedProductTypes.add(productTypes.get(2));
                                            }
                                            if (species.isBiologicalSeedInoculation()) {
                                                // "Inoculation biologique de semences ou plants"
                                                allowedProductTypes.add(productTypes.get(1));
                                            }
                                        }

                                        if (!allowedProductTypes.contains(seedingProductInput.getProductType())) {
                                            error = true;
                                            importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT 'SEMIS' IGNORÉ!, pour l'intrant avec identifiant '%s', aucune espece de l'action 'Semis' avec comme identifiant '%s' n'a de traitement sélectionné pour ce type d'intrant '%s' !", from.getId(), from.getSeedingActionId(), seedingProductInput.getProductType()));
                                        }
                                    }

                                } else {
                                    error = true;
                                    importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT 'SEMIS' IGNORÉ!, aucune espece présente sur l'action 'Semis' avec comme identifiant '%s' pour l'intrant '%s' !", from.getSeedingActionId() ,from.getId()));
                                }

                                error = validPhytoProductInput(seedingProductInput, from, speciesCode, importResults, line, error);
                            } else {
                                error = true;
                                importResults.addErrorLine(line, "AJOUT DE L'INTRANT 'SEMIS' IGNORÉ!, la colonne 'seedingaction' n'est pas renseignée.");
                            }

                        } else if (AgrosystInterventionType.LUTTE_BIOLOGIQUE.equals(type)) {
                            if (StringUtils.isNotBlank(from.getBiologicalControlActionId())) {
                                expectedObject = "BiologicalProductInput";
                                presentObject = input.getClass().getSimpleName();
                                BiologicalProductInput biologicalProductInput = (BiologicalProductInput) input;

                                expectedObject = "BiologicalControlAction";
                                presentObject = action.getClass().getSimpleName();
                                biologicalProductInput.setBiologicalControlAction((BiologicalControlAction) action);

                                Set<String> speciesCode = getSpeciesCodesFromActionId(from.getBiologicalControlActionId());
                                error = validPhytoProductInput(biologicalProductInput, from, speciesCode, importResults, line, error);
                            } else {
                                error = true;
                                importResults.addErrorLine(line, "AJOUT DE L'INTRANT 'BIOLOGIQUE' IGNORÉ!, la colonne 'biologicalcontrolaction' n'est pas renseignée.");
                            }

                        } else if (AgrosystInterventionType.EPANDAGES_ORGANIQUES.equals(type)) {
                            expectedObject = "OrganicProductInput";
                            presentObject = input.getClass().getSimpleName();
                            OrganicProductInput organicProductInput = ((OrganicProductInput) input);

                            if (StringUtils.isNotBlank(from.getOrganicFertilizersSpreadingActionId())) {
                                expectedObject = "OrganicFertilizersSpreadingAction";
                                presentObject = action.getClass().getSimpleName();
                                organicProductInput.setOrganicFertilizersSpreadingAction((OrganicFertilizersSpreadingAction) action);
                            } else {
                                error = true;
                                importResults.addErrorLine(line, "AJOUT DE L'INTRANT 'ORGANIQUE' IGNORÉ!, la colonne 'organicfertilizersspreadingaction' n'est pas renseignée.");
                            }
                            if (StringUtils.isNotBlank(from.getOrganicProductId())) {
                                RefFertiOrga product = organicProductDao.forTopiaIdEquals(from.getOrganicProductId()).findUniqueOrNull();
                                if (product != null) {
                                    organicProductInput.setOrganicProduct(product);
                                } else {
                                    error = true;
                                    importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT '%s' IGNORÉ!, le produit avec comme identifiant '%s' n'a pas été retrouvée.", type, from.getOrganicProductId()));
                                }
                            } else {
                                error = true;
                                importResults.addErrorLine(line, "AJOUT DE L'INTRANT '%s' IGNORÉ!, la colonne 'organicproduct' n'est pas renseignée.");
                            }

                            if (StringUtils.isNotBlank(from.getOrganicProductUnitName())) {
                                try {
                                    OrganicProductUnit organicProductUnit = OrganicProductUnit.valueOf(from.getOrganicProductUnitName());
                                    organicProductInput.setOrganicProductUnit(organicProductUnit);
                                } catch (IllegalArgumentException e) {
                                    error = true;
                                    importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT '%s' IGNORÉ!, unité de produit organique non reconnues '%s'.", type, from.getOrganicProductUnitName()));
                                }
                            }
                        } else {
                            error = true;
                            importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT IGNORÉ!, aucun intrant ne peut être associé à un type d'intervention '%s'.", type));
                        }
                    } catch (ClassCastException e) {
                        error = true;
                        importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT '%s' IGNORÉ!, l'objet attendu:'%s' n'est pas celui en paramètre:'%s'.", type, expectedObject, StringUtils.removeEnd(presentObject, "Impl")));
                    }
                }
            } else {
                error = true;
                importResults.addErrorLine(line, "AJOUT DE L'INTRANT IGNORÉ!, aucun type d'intrant spécifié.");
            }
        }

        return error;
    }

    private Set<String> getSpeciesCodesFromActionId(String actionId) {
        Set<String> speciesCodes = Sets.newHashSet();
        List<SpeciesStadeDto> speciesStadeDtos = getSpeciesStadeDtosFromActionId(actionId);
        if (CollectionUtils.isNotEmpty(speciesStadeDtos)) {
            for (SpeciesStadeDto speciesStadeDto : speciesStadeDtos) {
                speciesCodes.add(speciesStadeDto.getSpeciesCode());
            }
        }
        return speciesCodes;
    }

    protected List<SpeciesStadeDto> getSpeciesStadeDtosFromActionId(String actionId) {
        List<SpeciesStadeDto> speciesStadeDtos = null;
        Pz0PracticedIntervention pz0PracticedIntervention = (Pz0PracticedIntervention) pz0IdToObject.get(AbstractAction.class, Pz0PracticedIntervention.class, actionId);
        if (pz0PracticedIntervention != null && pz0PracticedIntervention.getPracticedInterventionDto() != null) {
            speciesStadeDtos = pz0PracticedIntervention.getPracticedInterventionDto().getSpeciesStadesDtos();
        } else {
            Pz0EffectiveIntervention pz0EffectiveIntervention = (Pz0EffectiveIntervention) pz0IdToObject.get(AbstractAction.class, Pz0EffectiveIntervention.class, actionId);
            if (pz0EffectiveIntervention != null && pz0EffectiveIntervention.getEffectiveInterventionDto() != null) {
                speciesStadeDtos = pz0EffectiveIntervention.getEffectiveInterventionDto().getSpeciesStadesDtos();
            }
        }
        return speciesStadeDtos;
    }

    protected Pair<Boolean, Set<String>> getRefEspeceIds(ImportResults importResults, String actionId, Set<String> croppingPlanSpeciesCodes, long line, Boolean error) {
        Set<String> refEspeceIds = Sets.newHashSet();
        if (CollectionUtils.isNotEmpty(croppingPlanSpeciesCodes)) {
            DomainAndDependencies domainAndDependencies = getDomainAndDependencies(actionId);
            if (domainAndDependencies != null) {
                Map<String, CroppingPlanSpeciesDto> croppingPlanSpeciesByCsvCode = domainAndDependencies.getCroppingPlanSpeciesByCsvCode();
                Map<String, String> croppingPlanSpeciesCsvCodesByAgrosystCodes = domainAndDependencies.getCroppingPlanSpeciesAgrosystCodeToCsvCode();
                for (String agrosystCode : croppingPlanSpeciesCodes) {
                    String csvCode = croppingPlanSpeciesCsvCodesByAgrosystCodes.get(agrosystCode);
                    CroppingPlanSpeciesDto croppingPlanSpeciesDto = croppingPlanSpeciesByCsvCode.get(csvCode);
                    if (croppingPlanSpeciesDto != null) {
                        if (StringUtils.isNotBlank(croppingPlanSpeciesDto.getSpeciesId())) {
                            refEspeceIds.add(croppingPlanSpeciesDto.getSpeciesId());
                        } else {
                            error = true;
                            importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT IGNORÉ!, l'espèce au code '%s' n'est pas associée à une espèce du référentiel (RefEspece)", csvCode));
                        }
                    } else {
                        error = true;
                        importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT IGNORÉ!, l'espèce au code '%s' n'a pu être retrouvée sur le domaine %s", csvCode, domainAndDependencies.getCsvId()));
                    }
                }
            } else {
                error = true;
                importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT IGNORÉ!, le domaine associé à l'action dont l'identifiant est '%s' n'a pu être retrouvé", actionId));
            }
        }
        Pair<Boolean, Set<String>> result = Pair.of(error, refEspeceIds);
        return result;
    }

    protected DomainAndDependencies getDomainAndDependencies(String actionId) {
        DomainAndDependencies domainAndDependencies = null;

        Pz0PracticedIntervention pz0PracticedIntervention = (Pz0PracticedIntervention) pz0IdToObject.get(AbstractAction.class, Pz0PracticedIntervention.class, actionId);
        if (pz0PracticedIntervention != null) {
            String practicedSystemId = pz0PracticedIntervention.getPracticedSystemId();
            if (StringUtils.isNoneBlank(practicedSystemId)) {
                PracticedSystemAndDependencies practicedSystemAndDependencies = (PracticedSystemAndDependencies) pz0IdToObject.get(PracticedSystem.class, practicedSystemId);
                if (practicedSystemAndDependencies != null) {
                    domainAndDependencies = PracticedSystemImporter.getDomainAndDependenciesFromPracticedSystem(practicedSystemAndDependencies.getPracticedSystem());
                }
            }
        } else {
            Pz0EffectiveIntervention pz0EffectiveIntervention = (Pz0EffectiveIntervention) pz0IdToObject.get(AbstractAction.class, Pz0EffectiveIntervention.class, actionId);
            if (pz0EffectiveIntervention != null && StringUtils.isNotBlank(pz0EffectiveIntervention.getZoneId())) {
                PlotAndDependencies plotAndDependencies = (PlotAndDependencies) pz0IdToRelatedObjectId.get(Zone.class, PlotAndDependencies.class, pz0EffectiveIntervention.getZoneId());
                if (plotAndDependencies != null) {
                    domainAndDependencies = plotAndDependencies.getDomainAndDependencies();
                }
            }
        }
        return domainAndDependencies;
    }

    public void init(ServiceContext serviceContext) {
        super.init(serviceContext);
        mineralProductDao = getPersistenceContext().getRefFertiMinUNIFADao();
        phytoProductDao = getPersistenceContext().getRefActaTraitementsProduitDao();
        organicProductDao = getPersistenceContext().getRefFertiOrgaDao();
        referentialService = getServiceFactory().newService(ReferentialService.class);
    }

}
