package fr.inra.agrosyst.services.pz0import.effective.effectiveCropCycleSpecies;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: EffectiveCropCycleSpeciesImporter.java 5074 2015-09-01 08:13:11Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.2/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/effective/effectiveCropCycleSpecies/EffectiveCropCycleSpeciesImporter.java $
 * %%
 * Copyright (C) 2013 - 2015 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import fr.inra.agrosyst.api.entities.referential.RefClonePlantGrape;
import fr.inra.agrosyst.api.entities.referential.RefClonePlantGrapeTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefVariete;
import fr.inra.agrosyst.api.entities.referential.RefVarieteTopiaDao;
import fr.inra.agrosyst.api.services.domain.CroppingPlanEntryDto;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCycleSpeciesDto;
import fr.inra.agrosyst.api.services.effective.EffectivePerennialCropCycleDto;
import fr.inra.agrosyst.api.services.practiced.PracticedSystems;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.effective.Pz0EffectivePerennialCropCycle;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.services.pz0import.AbstractCSVImporter;
import fr.inra.agrosyst.services.pz0import.domain.dto.CroppingPlanSpeciesImportDto;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;

import java.io.InputStream;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Map;

/**
 * Created by davidcosse on 01/04/15.
 */
public class EffectiveCropCycleSpeciesImporter extends AbstractCSVImporter {

    private static final Log log = LogFactory.getLog(EffectiveCropCycleSpeciesImporter.class);

    protected RefVarieteTopiaDao refVarieteDao;
    protected RefClonePlantGrapeTopiaDao graftCloneDao;

    @Override
    public ImportResults importFromStream(InputStream is, Map<String, EntityAndDependencies> entitiesByCsvId) {
        ImportResults importResults = new ImportResults(EffectiveCropCycleSpeciesImportDto.class);
        log.debug("importfromstream");

        EffectiveCropCycleSpeciesImportModel model = new EffectiveCropCycleSpeciesImportModel();
        // récupère le DTO
        Import<EffectiveCropCycleSpeciesImportDto> importer = Import.newImport(model, is);

        // match the first csv line number with data (not header).
        long line = FIRST_LINE_NUMBER;
        for (EffectiveCropCycleSpeciesImportDto dto : importer) {
            boolean error;

            EffectiveCropCycleSpeciesDto speciesDto = new EffectiveCropCycleSpeciesDto();

            // required field
            error = validEffectivePerennialCropCycleAffiliation(importResults, line, dto, speciesDto, false);
            error = validPlantsCertified(importResults, line, dto, speciesDto, error);

            // valid required related object
            error = validSpecies(importResults, line, dto, speciesDto, error);

            // valid optional related objects
            error = validGraftSupport(importResults, line, dto, speciesDto, error);
            error = validGraftClone(importResults, line, dto, speciesDto, error);
            error = validOverGraftDate(importResults, line, dto, speciesDto, error);

            pz0IdToObject.put(EffectiveCropCycleSpeciesDto.class, dto.getId(), speciesDto);

            if (!error) {
                importResults.addInfoLine(line, ", " + "ESPÈCE PERENNE DU RÉALISÉ VALIDÉ, csvid: " + dto.getId());
                importResults.increaseAddedRecords();
            } else {
                importResults.increaseIgnoredRecords();
                importResults.addErrorLine(line, "PERENNIAL ESPÈCE PERENNE DU RÉALISÉ IGNORÉE csvid:" + dto.getId());
            }

            line++;
        }
        return importResults;
    }

    protected boolean validEffectivePerennialCropCycleAffiliation(ImportResults importResults, long line, EffectiveCropCycleSpeciesImportDto dto, EffectiveCropCycleSpeciesDto species, boolean error) {
        String effectivePerennialCropCycleId = dto.getEffectivePerennialCropCycleId();
        if (StringUtils.isNotBlank(effectivePerennialCropCycleId)) {
            Pz0EffectivePerennialCropCycle pz0PerennialCropCycleDto = (Pz0EffectivePerennialCropCycle) pz0IdToObject.get(Pz0EffectivePerennialCropCycle.class, effectivePerennialCropCycleId);
            if (pz0PerennialCropCycleDto != null) {
                EffectivePerennialCropCycleDto perennialCropCycleDto = pz0PerennialCropCycleDto.getPerennialCropCycleDto();
                perennialCropCycleDto.addSpeciesDto(species);
                // use later on to validate species
                dto.setCroppingPlanEntryId(perennialCropCycleDto.getCroppingPlanEntryId());
            } else {
                importResults.addErrorLine(line, String.format("ESPÈCE PERENNE DU RÉALISÉ IGNORÉE!, le cycle de culture pérenne avec comme id: '%s' n'est pas renseignée pour le cycle d'id: %s", effectivePerennialCropCycleId, dto.getId()));
                error = true;
            }
        } else {
            importResults.addErrorLine(line, "ESPÈCE PERENNE DU RÉALISÉ IGNORÉE!, la colonne 'plantscertified' n'est pas renseignée pour le cycle d'id: " + dto.getId());
            error = true;
        }
        return error;
    }

    protected boolean validPlantsCertified(ImportResults importResults, long line, EffectiveCropCycleSpeciesImportDto dto, EffectiveCropCycleSpeciesDto species, boolean error) {
        Boolean plantsCertified = dto.isPlantsCertified();
        if (plantsCertified != null) {
            species.setPlantsCertified(plantsCertified);
        } else {
            importResults.addErrorLine(line, "ESPÈCE PERENNE DU RÉALISÉ IGNORÉE!, la colonne 'plantscertified' n'est pas renseignée pour le cycle d'id: " + dto.getId());
            error = true;
        }
        return error;
    }

    protected boolean validSpecies(ImportResults importResults, long line, EffectiveCropCycleSpeciesImportDto dto, EffectiveCropCycleSpeciesDto species, boolean error) {
        String speciesId = dto.getCroppingPlanSpeciesId();
        if (StringUtils.isNotBlank(speciesId)) {
            String cropId = dto.getCroppingPlanEntryId();
            CroppingPlanSpeciesImportDto speciesImport = (CroppingPlanSpeciesImportDto) pz0IdToObject.get(CroppingPlanSpeciesImportDto.class, speciesId);
            if(speciesImport != null) {
                String speciesCropId = speciesImport.getCroppingPlanEntryId();
                if (cropId.contentEquals(speciesCropId)) {
                    String speciesAgrosystCode = csvCodeToAgrosystCode(CroppingPlanEntryDto.class, speciesImport.getCode());
                    species.setCode(speciesAgrosystCode);
                    species.setCroppingPlanSpeciesId(speciesId);
                } else {
                    importResults.addErrorLine(line, String.format("ESPÈCE PERENNE DU RÉALISÉ IGNORÉE!, l'espèce avec comme identifiant '%s' n'a pas été retrouvée pour l'espèce pérenne: %s", speciesId, dto.getId()));
                    error = true;
                }
            } else {
                importResults.addErrorLine(line, String.format("ESPÈCE PERENNE DU RÉALISÉ IGNORÉE!, l'espèce avec comme identifiant '%s' n'a pas été retrouvée pour l'espèce pérenne: %s", speciesId, dto.getId()));
                error = true;
            }

        } else {
            importResults.addErrorLine(line, "ESPÈCE PERENNE DU RÉALISÉ IGNORÉE!, la colonne 'croppingplanspecies' n'est pas renseignée pour le cycle d'id: " + dto.getId());
            error = true;
        }
        return error;
    }

    protected boolean validGraftSupport(ImportResults importResults, long line, EffectiveCropCycleSpeciesImportDto dto, EffectiveCropCycleSpeciesDto species, boolean error) {
        String graftSupportId = dto.getGraftSupportId();
        if (StringUtils.isNotBlank(graftSupportId)) {
            RefVariete graftSupport = refVarieteDao.forTopiaIdEquals(graftSupportId).findUniqueOrNull();
            //refVarieteDao.forTopiaIdEquals(speciesDto.getGraftSupport().getTopiaId()).findUnique();
            if (graftSupport != null) {
                species.setGraftSupport(PracticedSystems.REF_VARIETE_TO_GRAPH_DTO.apply(graftSupport));
            } else {
                importResults.addErrorLine(line, String.format("ESPÈCE PERENNE DU RÉALISÉ IGNORÉE!, le porte-greffe avec comme identifiant '%s' n'a pas été retrouvée pour l'espèce pérenne: %s", graftSupportId, dto.getId()));
                error = true;
            }
        }
        return error;
    }

    protected boolean validGraftClone(ImportResults importResults, long line, EffectiveCropCycleSpeciesImportDto dto, EffectiveCropCycleSpeciesDto species, boolean error) {
        String graftCloneId = dto.getGraftCloneId();
        if (StringUtils.isNotBlank(graftCloneId)) {
            RefClonePlantGrape graftClone = graftCloneDao.forTopiaIdEquals(graftCloneId).findUniqueOrNull();
            if (graftClone != null) {
                species.setGraftClone(PracticedSystems.REF_CLONE_TO_GRAPH_DTO.apply(graftClone));
            } else {
                importResults.addErrorLine(line, String.format("ESPÈCE PERENNE DU RÉALISÉ IGNORÉE!, le clone de la greffe avec comme identifiant '%s' n'a pas été retrouvée pour l'espèce pérenne: %s", graftCloneId, dto.getId()));
                error = true;
            }
        }
        return error;
    }

    protected boolean validOverGraftDate(ImportResults importResults, long line, EffectiveCropCycleSpeciesImportDto dto, EffectiveCropCycleSpeciesDto species, boolean error) {
        String date = dto.getOverGraftDate();
        if (StringUtils.isNotBlank(date)) {
            Date overGraftDate = null;
            try {
                SimpleDateFormat format = new SimpleDateFormat("dd/MM/yyyy");
                overGraftDate = format.parse(date);
            } catch (ParseException e) {
                importResults.addErrorLine(line, String.format("ESPÈCE PERENNE DU RÉALISÉ IGNORÉE!, la date de sur-greffage avec comme identifiant '%s' n'a pas été retrouvée pour l'espèce pérenne: %s", date, dto.getId()));
                error = true;
            }
            species.setOverGraftDate(overGraftDate);

        }
        return error;
    }

    @Override
    public void init(ServiceContext serviceContext) {
        super.init(serviceContext);
        graftCloneDao = getPersistenceContext().getRefClonePlantGrapeDao();
        refVarieteDao = getPersistenceContext().getRefVarieteDao();
    }

}
