package fr.inra.agrosyst.services.pz0import.practicedSystem;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: PracticedSystemImporter.java 5089 2015-09-09 07:44:14Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.1/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/practicedSystem/PracticedSystemImporter.java $
 * %%
 * Copyright (C) 2013 - 2015 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.GrowingPlan;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystem;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystemSource;
import fr.inra.agrosyst.api.services.practiced.PracticedSystemService;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.domains.DomainAndDependencies;
import fr.inra.agrosyst.api.services.pz0.growingPlan.GrowingPlanAndDependencies;
import fr.inra.agrosyst.api.services.pz0.growingSystem.GrowingSystemAndDependencies;
import fr.inra.agrosyst.api.services.pz0.practicedSystem.PracticedSystemAndDependencies;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.services.common.CommonService;
import fr.inra.agrosyst.services.pz0import.AbstractCSVImporter;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.InputStream;
import java.util.Map;

/**
 * Created by davidcosse on 16/12/14.
 */
public class PracticedSystemImporter extends AbstractCSVImporter {

    protected PracticedSystemService practicedSystemService;

    private static final Log log = LogFactory.getLog(PracticedSystemImporter.class);

    public static DomainAndDependencies getDomainAndDependenciesFromPracticedSystem(PracticedSystem practicedSystem) {
        DomainAndDependencies result = null;
        if (practicedSystem != null && practicedSystem.getGrowingSystem() != null && practicedSystem.getGrowingSystem().getGrowingPlan() != null) {
            GrowingPlan growingPlan = practicedSystem.getGrowingSystem().getGrowingPlan();
            GrowingPlanAndDependencies growingPlanAndDependencies = (GrowingPlanAndDependencies) pz0IdToRelatedObjectId.get(GrowingPlan.class, GrowingPlanAndDependencies.class, growingPlan);
            if (growingPlanAndDependencies != null && StringUtils.isNotBlank(growingPlanAndDependencies.getDomainPZ0Id())) {
                result = (DomainAndDependencies) pz0IdToObject.get(Domain.class, growingPlanAndDependencies.getDomainPZ0Id());
            }
        }
        return result;
    }

    @Override
    public ImportResults importFromStream(InputStream is, Map<String, EntityAndDependencies> growingSystemsByCsvId) {
        ImportResults importResults = new ImportResults(PracticedSystem.class);
        log.debug("importfromstream");

        PracticedSystemImportModel model = new PracticedSystemImportModel();
        // récupère le DTO
        Import<PracticedSystemImportDto> importer = Import.newImport(model, is);

        // match the first csv line number with data (not header).
        long line = FIRST_LINE_NUMBER;
        for (PracticedSystemImportDto dto : importer) {
            boolean error;
            PracticedSystem practicedSystem = practicedSystemService.getPracticedSystem(null);

            Binder<PracticedSystemImportDto, PracticedSystem> practicedSystemBinder = BinderFactory.newBinder(PracticedSystemImportDto.class, PracticedSystem.class);
            practicedSystemBinder.copyExcluding(dto, practicedSystem,
                    PracticedSystem.PROPERTY_TOPIA_ID,
                    PracticedSystem.PROPERTY_GROWING_SYSTEM);

            // valid required fields
            error = validGrowingSystemAffiliation(importResults, line, dto, practicedSystem, growingSystemsByCsvId, false);

            error = validCampaigns(importResults, line, dto, practicedSystem, error);

            error = validSource(importResults, line, dto, practicedSystem, error);

            PracticedSystemAndDependencies practicedSystemAndDependencies = new PracticedSystemAndDependencies(practicedSystem, line, dto.getId());
            importResults.addEntity(dto.getId(), practicedSystemAndDependencies);
            pz0IdToObject.put(PracticedSystem.class,  dto.getId(), practicedSystemAndDependencies);
            if (!error) {
                importResults.increaseAddedRecords();
                importResults.addInfoLine(line, ", " + "SYSTÈME SYNTHÉTISÉ PRÉ-VALIDÉ csvid:" + dto.getId() + ", vérifier qu'il n'y a pas d'echec de validation concernant les préconditions.");
            } else {
                importResults.increaseIgnoredRecords();
                importResults.addErrorLine(line, "SYSTÈME SYNTHÉTISÉ IGNORÉ csvid:" + dto.getId());
            }

            line++;
        }
        return importResults;
    }

    private boolean validSource(ImportResults importResults, long line, PracticedSystemImportDto dto, PracticedSystem practicedSystem, boolean error) {
        if (StringUtils.isNotBlank(dto.getSourceName())) {
            PracticedSystemSource source = parseSource(dto.getSourceName());
            if (source != null) {
                practicedSystem.setSource(source);
            } else {
                error = true;
                importResults.addErrorLine(line, String.format("SYSTÈME SYNTHÉTISÉ IGNORÉ, la source '%s' n'est pas valide", dto.getSourceName()));
            }
        }
        return error;
    }

    /**
     * value must not be null !
     */
    protected PracticedSystemSource parseSource(String value) {
        PracticedSystemSource result = null;
        if (value.equalsIgnoreCase("ENTRETIEN")) {
            result = PracticedSystemSource.ENTRETIEN;
        } else if (value.equalsIgnoreCase("ENREGISTREMENT")) {
            result = PracticedSystemSource.ENREGISTREMENT;
        } else if(value.equalsIgnoreCase("ENTRETIEN_ENREGISTREMENT")) {
            result = PracticedSystemSource.ENTRETIEN_ENREGISTREMENT;
        }
        return result;
    }

    protected boolean validCampaigns(ImportResults importResults, long line, PracticedSystemImportDto dto, PracticedSystem practicedSystem, boolean error) {
        if (StringUtils.isNotBlank(practicedSystem.getCampaigns())) {
            if (!CommonService.ARE_CAMPAIGNS_VALIDS(practicedSystem.getCampaigns())) {
                error = true;
                importResults.addErrorLine(line, String.format("SYSTÈME SYNTHÉTISÉ IGNORÉ, les campagnes suivantes %s ne sont pas valide, système synthétisé concerné csvid: %s", practicedSystem.getCampaigns(), dto.getId()));
            }
        } else {
            error = true;
            importResults.addErrorLine(line, "SYSTÈME SYNTHÉTISÉ IGNORÉ, aucune campagne n'est définie pour le système synthétisé csvid:" + dto.getId());
        }
        return error;
    }

    protected boolean validGrowingSystemAffiliation(ImportResults importResults, long line, PracticedSystemImportDto dto, PracticedSystem practicedSystem, Map<String, EntityAndDependencies> growingSystemsByCsvIds, boolean error) {
        try {
            if (StringUtils.isNotBlank(dto.getGrowingSystemId())) {
                GrowingSystemAndDependencies entityAndDependencies = (GrowingSystemAndDependencies) growingSystemsByCsvIds.get(dto.getGrowingSystemId());
                if (entityAndDependencies != null) {
                    GrowingSystem growingSystem = entityAndDependencies.getGrowingSystem();
                    practicedSystem.setGrowingSystem(growingSystem);
                } else {
                    importResults.addErrorLine(line, String.format("SYSTÈME SYNTHÉTISÉ IGNORÉ!, aucun système de culture n'est retrouvée avec l'identifiant suivant: %s", dto.getGrowingSystemId()));
                    error = true;
                }
            } else {
                importResults.addErrorLine(line, String.format("SYSTÈME SYNTHÉTISÉ IGNORÉ!, la colonne 'growingsystem' n'est pas renseignée pour le SDC \"%s\"!", dto.getName()));
                error = true;
            }
        } catch (ClassCastException e) {
            importResults.addErrorLine(line, String.format("SYSTÈME SYNTHÉTISÉ IGNORÉ!, système de culture non retrouvé pour le système synthétisé \"%s\"!", dto.getName()));
            error = true;
        }
        return error;
    }

    @Override
    public void init(ServiceContext serviceContext) {
        super.init(serviceContext);
        practicedSystemService = getServiceFactory().newService(PracticedSystemService.class);
    }
}
