package fr.inra.agrosyst.services.pz0import.domain;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: EquipmentImporter.java 5060 2015-08-18 16:20:23Z eancelet $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.1/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/domain/EquipmentImporter.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import fr.inra.agrosyst.api.entities.Equipment;
import fr.inra.agrosyst.api.entities.EquipmentImpl;
import fr.inra.agrosyst.api.entities.ToolsCoupling;
import fr.inra.agrosyst.api.entities.referential.RefMateriel;
import fr.inra.agrosyst.api.entities.referential.RefMaterielTopiaDao;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.domains.DomainAndDependencies;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.services.pz0import.AbstractCSVImporter;
import fr.inra.agrosyst.services.pz0import.domain.dto.EquipmentImportDto;
import fr.inra.agrosyst.services.pz0import.domain.model.EquipmentImportModel;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.InputStream;
import java.util.Map;

/**
 * Created by davidcosse on 09/12/14.
 */
public class EquipmentImporter extends AbstractCSVImporter {

    private static final Log log = LogFactory.getLog(EquipmentImporter.class);

    protected RefMaterielTopiaDao refMaterielTopiaDao;

    @Override
    public ImportResults importFromStream(InputStream is, Map<String, EntityAndDependencies> entitiesByCsvId) {
        ImportResults importResults = new ImportResults(Equipment.class);

        EquipmentImportModel model = new EquipmentImportModel();
        Import<EquipmentImportDto> importer = Import.newImport(model, is);

        // match the csv line number.
        Long line = FIRST_LINE_NUMBER;
        for (EquipmentImportDto dto : importer) {
            boolean error;

            Equipment equipment = new EquipmentImpl();

            Binder<EquipmentImportDto, Equipment> binder = BinderFactory.newBinder(EquipmentImportDto.class, Equipment.class);
            // the equipment topiaId has to be bind as it is use by service to link equipment and tool coupling.
            binder.copyExcluding(dto, equipment,
                    Equipment.PROPERTY_DOMAIN,
                    Equipment.PROPERTY_REF_MATERIEL);

            if (StringUtils.isEmpty(dto.getCode())) {
                error = true;
                importResults.addErrorLine(line, String.format("EQUIPEMENT IGNORÉ, l'équipement '%s' n'a pas de code", dto.getId()));
            }
            
            equipment.setCode(csvCodeToAgrosystCode(ToolsCoupling.class, dto.getCode()));

            error = addMaterielToEquipment(dto, equipment, importResults, line, false);

            error = addEquipmentToDomain(dto, equipment, entitiesByCsvId, importResults, line, error);

            pz0IdToObject.put(Equipment.class, dto.getId(), equipment);
            if (!error) {
                importResults.addInfoLine(line, ", " + "EQUIPEMENT VALIDÉ, csvid: " + dto.getId());
                importResults.increaseAddedRecords();
            } else {
                importResults.increaseIgnoredRecords();
                importResults.addErrorLine(line, "EQUIPEMENT IGNORÉ, csvid: " + dto.getId());
            }

            line++;
        }
        return  importResults;
    }

    protected boolean addEquipmentToDomain(EquipmentImportDto dto, Equipment equipment, Map<String, EntityAndDependencies> entitiesByCsvId, ImportResults importResults, long line, boolean error) {

        if (StringUtils.isNotBlank(dto.getDomainId())) {
            DomainAndDependencies domainAndDependencies = (DomainAndDependencies) entitiesByCsvId.get(dto.getDomainId());
            if (domainAndDependencies != null) {
                domainAndDependencies.addEquipement(equipment);
            } else {
                importResults.addErrorLine(line, String.format("EQUIPEMENT IGNORÉ, le domaine avec comme identifiant:%s, n'a pas été retrouvé pour l'equipment': %s",dto.getDomainId(), dto.getId()));
                error = true;
            }
        } else {
            importResults.addErrorLine(line, "EQUIPEMENT IGNORÉ, la colonne 'domain' n'est pas renseignée, csvid: " + dto.getId());
            error = true;
        }
        return error;
    }

    protected boolean addMaterielToEquipment(EquipmentImportDto dto, Equipment equipment, ImportResults importResults, long line, boolean error) {

        if (StringUtils.isNotBlank(dto.getRefmaterielID())) {
            RefMateriel materiel = refMaterielTopiaDao.forTopiaIdEquals(dto.getRefmaterielID()).findUniqueOrNull();
            if (materiel != null) {
                equipment.setRefMateriel(materiel);
            } else {
                importResults.addErrorLine(line, String.format("MATEIEL IGNORÉE, le materiel avec comme identifiant:%s, " +
                        "n'a pas été retrouvé, csvid: %s",dto.getRefmaterielID(), dto.getId()));
                error = true;
            }
        }
        return error;
    }

    @Override
    public void init(ServiceContext serviceContext) {
        super.init(serviceContext);

        refMaterielTopiaDao = getPersistenceContext().getRefMaterielDao();
    }
}
