package fr.inra.agrosyst.services.pz0import.zone;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: ZoneImporter.java 4923 2015-05-05 15:53:56Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.4.9/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/zone/ZoneImporter.java $
 * %%
 * Copyright (C) 2013 - 2015 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */


import com.google.common.base.Strings;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.ZoneImpl;
import fr.inra.agrosyst.api.entities.ZoneTopiaDao;
import fr.inra.agrosyst.api.entities.ZoneType;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.plot.PlotAndDependencies;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.services.pz0import.AbstractCSVImporter;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.InputStream;
import java.util.Map;

/**
 * Created by davidcosse on 25/03/15.
 */
public class ZoneImporter extends AbstractCSVImporter {

    protected ZoneTopiaDao zoneTopiaDao;

    private static final Log log = LogFactory.getLog(ZoneImporter.class);

    @Override
    public ImportResults importFromStream(InputStream is, Map<String, EntityAndDependencies> plotAndDependenciesByCsvId) {
        ImportResults importResults = new ImportResults(Zone.class);
        log.debug(String.format("importfromstream"));

        ZoneImportModel model = new ZoneImportModel();
        // récupère le DTO
        Import<ZoneDto> importer = Import.newImport(model, is);

        // match the first csv line number with data (not header).
        long line = FIRST_LINE_NUMBER;
        for (ZoneDto dto : importer) {
            boolean error = false;
            Zone zone = zoneTopiaDao.forTopiaIdEquals(dto.getId()).findUniqueOrNull();

            if (zone != null) {
                error = true;
                importResults.addErrorLine(line, String.format("ZONE IGNORÉE!, une zone avec l'identifiant %s existe déjà", dto.getId()));
            }

            if (!error) {
                Binder<ZoneDto, Zone> binder = BinderFactory.newBinder(ZoneDto.class, Zone.class);
                zone = new ZoneImpl();

                binder.copyExcluding(dto, zone, Zone.PROPERTY_TOPIA_ID);

                // valid required fields
                error = validPlotAffiliation(importResults, line, dto, zone, plotAndDependenciesByCsvId, error);
                error = validCode(importResults, line, dto, zone, error);

                // valid optional enums
                error = validZoneType(importResults, line, dto, zone, error);

                // to be able to find plot from zone
                pz0IdToRelatedObjectId.put(Zone.class, PlotAndDependencies.class, dto.getId(), plotAndDependenciesByCsvId.get(dto.getPlot()));
                if (!error) {
                    importResults.addInfoLine(line, "ZONE VALIDÉE, csvid: " + dto.getId());
                    importResults.increaseAddedRecords();
                } else {
                    importResults.increaseIgnoredRecords();
                    importResults.addErrorLine(line, "ZONE IGNORÉE csvid:" + dto.getId());
                }
            }

            line++;
        }
        return importResults;
    }

    protected boolean validCode(ImportResults results, long line, ZoneDto dto, Zone zone, boolean error){
        if (StringUtils.isNotBlank(dto.getCode())) {
            zone.setCode(csvCodeToAgrosystCode(Zone.class, dto.getCode()));
        } else {
            results.addErrorLine(line, String.format("ZONE IGNORÉE!, le code de la zone est requit pour la parcelle %s !", dto.getPlot()));
            error = true;
        }
        return error;
    }

    protected boolean validPlotAffiliation(ImportResults results, long line, ZoneDto dto, Zone zone, Map<String, EntityAndDependencies> plotAndDependenciesByCsvId, boolean error) {

        if (StringUtils.isNotBlank(dto.getPlot())) {
            PlotAndDependencies plotAndDependencies = plotAndDependenciesByCsvId != null ? (PlotAndDependencies) plotAndDependenciesByCsvId.get(dto.getPlot()) : null;
            if (plotAndDependencies != null) {
                plotAndDependencies.addZone(dto.getId(), zone);
            } else {
                error = true;
                results.addErrorLine(line, String.format("ZONE IGNORÉE!, la parcelle avec comme identifiant %s n'a pas été retrouvée !", dto.getPlot()));
            }
        } else {
            error = true;
            results.addErrorLine(line, String.format("ZONE IGNORÉE!, la colonne 'plot' n'est pas renseignée pour la parcelle avec comme identifiant %s!", dto.getId()));
        }
        return error;
    }


    protected boolean validZoneType(ImportResults result, long line, ZoneDto dto, Zone zone, boolean error) {
        String value = dto.getType();
            ZoneType zoneType = null;

            if (!Strings.isNullOrEmpty(value)) {
                try {
                    zoneType = ZoneType.valueOf(value);
                } catch (IllegalArgumentException e) {
                    result.addErrorLine(line, String.format("ZONE IGNORÉE! Type de zone, non supportée : %s", value));
                    error = true;
                }
            } else {
                result.addErrorLine(line, String.format("ZONE IGNORÉE! la colonne 'type' n'est pas renseignée pour la zone \"%s\"!", dto.getName()));
                error = true;
            }
        zone.setType(zoneType);
        return error;
    }

    @Override
    public void init(ServiceContext serviceContext) {
        super.init(serviceContext);

        zoneTopiaDao = getPersistenceContext().getZoneDao();
    }


}
