package fr.inra.agrosyst.services.pz0import.practicedSystem.practicedIntervention;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: PracticedInterventionImporter.java 4923 2015-05-05 15:53:56Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.4.9/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/practicedSystem/practicedIntervention/PracticedInterventionImporter.java $
 * %%
 * Copyright (C) 2013 - 2015 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.MaterielTransportUnit;
import fr.inra.agrosyst.api.entities.MaterielWorkRateUnit;
import fr.inra.agrosyst.api.entities.practiced.PracticedIntervention;
import fr.inra.agrosyst.api.services.practiced.PracticedCropCycleConnectionDto;
import fr.inra.agrosyst.api.services.practiced.PracticedCropCyclePhaseDto;
import fr.inra.agrosyst.api.services.practiced.PracticedInterventionDto;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.practicedSystem.Pz0PracticedCropCycleNode;
import fr.inra.agrosyst.api.services.pz0.practicedSystem.Pz0PracticedIntervention;
import fr.inra.agrosyst.api.services.pz0.practicedSystem.Pz0PracticedCropCycleConnection;
import fr.inra.agrosyst.api.services.pz0.practicedSystem.Pz0PracticedCropCyclePhase;
import fr.inra.agrosyst.services.pz0import.AbstractCSVImporter;
import fr.inra.agrosyst.services.pz0import.practicedSystem.practicedCropCycleNode.PracticedCropCycleNodeImporter;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.InputStream;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Created by davidcosse on 05/02/15.
 */
public class PracticedInterventionImporter extends AbstractCSVImporter {

    private static final Log log = LogFactory.getLog(PracticedInterventionImporter.class);
    protected static final Pattern DATE_PATTERN = Pattern.compile("^(0?[1-9]|[12][0-9]|3[01])/(0?[1-9]|1[012])");

    @Override
    public ImportResults importFromStream(InputStream is, Map<String, EntityAndDependencies> entitiesByCsvId) {
        ImportResults importResults = new ImportResults(PracticedIntervention.class);

        PracticedInterventionImportModel model = new PracticedInterventionImportModel();
        // récupère le DTO
        Import<PracticedInterventionImportDto> importer = Import.newImport(model, is);

        // match the first csv line number with data (not header).
        long line = FIRST_LINE_NUMBER;
        for (PracticedInterventionImportDto dto : importer) {
            boolean error = false;

            Pz0PracticedIntervention pz0Intervention = new Pz0PracticedIntervention(dto.getId(), dto.getPracticedCropCycleConnectionId(), dto.getPracticedCropCyclePhaseId());
            PracticedInterventionDto interventionDto = pz0Intervention.getPracticedInterventionDto();
            Binder<PracticedInterventionImportDto, PracticedInterventionDto> binder = BinderFactory.newBinder(PracticedInterventionImportDto.class, PracticedInterventionDto.class);
            binder.copy(dto, interventionDto);

            // valid required related objets
            error = validPhaseOrConnectionFields(importResults, line, dto, error);
            error = validPhase(importResults, line, dto, error);
            error = validConnection(importResults, line, dto, error);
            addInterventionToPhaseOrConnection(importResults, line, dto, pz0Intervention, error);

            // valid required field
            error = validName(importResults, line, interventionDto, error);
            error = validRank(importResults, line, dto ,interventionDto, error);
            error = validStatingPeriod(importResults, line, interventionDto, error);
            error = validEndingPeriod(importResults, line, interventionDto, error);

            // valid required Enums
            error = validType(importResults, line, dto, interventionDto, error);

            // valid optional Enums
            validTransitVolumeUnit(importResults, line, dto, interventionDto, error);
            validWorkRateUnit(importResults, line, dto, interventionDto, error);

            pz0IdToObject.put(Pz0PracticedIntervention.class, dto.getId(), pz0Intervention);

            if (!error) {
                importResults.addInfoLine(line, "INTERVENTION DU SYNTHÉTISÉ VALIDÉE, csvid: " + dto.getId());
                importResults.increaseAddedRecords();
            } else {
                importResults.increaseIgnoredRecords();
                importResults.addErrorLine(line, "INTERVENTION DU SYNTHÉTISÉ IGNORÉE csvid:" + dto.getId());
            }
            line++;
        }
        return importResults;
    }

    protected boolean validPhaseOrConnectionFields(ImportResults importResults, long line, PracticedInterventionImportDto from, boolean error) {
        String practicedCropCyclePhaseId = from.getPracticedCropCyclePhaseId();
        String practicedCropCycleConnectionId = from.getPracticedCropCycleConnectionId();
        if(StringUtils.isBlank(practicedCropCyclePhaseId) && StringUtils.isBlank(practicedCropCycleConnectionId)){
            importResults.addErrorLine(line, "INTERVENTION IGNORÉE!, une des colonnes suivante doit-être renseigné: 'practicedcropcyclephase' ou 'practicedcropcycleconnection' mais aucune ne l'est.");
            error = true;
        }
        return error;
    }

    protected boolean validPhase(ImportResults importResults, long line, PracticedInterventionImportDto from, boolean error) {
        String phaseId = from.getPracticedCropCyclePhaseId();
        if (!error && StringUtils.isNotBlank(phaseId)) {
            Pz0PracticedCropCyclePhase pz0Phase = (Pz0PracticedCropCyclePhase)pz0IdToObject.get(Pz0PracticedCropCyclePhase.class, phaseId);
            if(pz0Phase == null) {
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, la phase avec comme id %s n'esiste pas.", phaseId));
                error = true;
            }

        }
        return error;
    }

    protected boolean validConnection(ImportResults importResults, long line, PracticedInterventionImportDto from, boolean error) {
        String connectionId = from.getPracticedCropCycleConnectionId();
        if (!error && StringUtils.isNotBlank(connectionId)) {
            Pz0PracticedCropCycleConnection  pz0Connection = (Pz0PracticedCropCycleConnection)pz0IdToObject.get(Pz0PracticedCropCycleConnection.class, connectionId);
            if(pz0Connection == null) {
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, la connexion avec comme id %s n'esiste pas.", connectionId));
                error = true;
            }
        }
        return error;
    }

    protected boolean addInterventionToPhaseOrConnection(ImportResults importResults, long line, PracticedInterventionImportDto from, Pz0PracticedIntervention pz0Intervention, boolean error) {
        String connectionId = from.getPracticedCropCycleConnectionId();
        String phaseId = from.getPracticedCropCyclePhaseId();
        if (StringUtils.isNotBlank(connectionId)) {
            Pz0PracticedCropCycleConnection  pz0Connection = (Pz0PracticedCropCycleConnection)pz0IdToObject.get(Pz0PracticedCropCycleConnection.class, connectionId);
            if (pz0Connection != null) {
                pz0Connection.addPracticedPz0Intervention(pz0Intervention);
                PracticedCropCycleConnectionDto connectionDto = pz0Connection.getPracticedCropCycleConnectionDto();
                String cropCode = getSeasonnalCropCode(pz0Intervention, connectionDto);
                pz0Intervention.setCropCode(cropCode);
            } else {
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, la connexion avec comme id %s n'esiste pas.", connectionId));
                error = true;
            }
            pz0IdToObject.put(PracticedInterventionDto.class, PracticedCropCycleConnectionDto.class, from.getId(), connectionId);
        } else if (StringUtils.isNotBlank(phaseId)){
            Pz0PracticedCropCyclePhase  pz0Phase = (Pz0PracticedCropCyclePhase)pz0IdToObject.get(Pz0PracticedCropCyclePhase.class, phaseId);
            if (pz0Phase != null) {
                pz0Intervention.setCropCode(pz0Phase.getPracticedPerennialCropCycleId());
                pz0Phase.addPracticedPz0Intervention(pz0Intervention);
            } else {
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, la phase avec comme id %s n'esiste pas.", phaseId));
                error = true;
            }
            pz0IdToObject.put(PracticedInterventionDto.class, PracticedCropCyclePhaseDto.class, from.getId(), phaseId);
        }
        return error;
    }

    protected String getSeasonnalCropCode(Pz0PracticedIntervention pz0Intervention, PracticedCropCycleConnectionDto connectionDto) {
        String cropCode;
        if (pz0Intervention.getPracticedInterventionDto().isIntermediateCrop()) {
            cropCode = connectionDto.getIntermediateCroppingPlanEntryCode();
        } else {
            String csvNodeId = StringUtils.remove(connectionDto.getTargetId(), PracticedCropCycleNodeImporter.NEW_NODE);
            Pz0PracticedCropCycleNode node = (Pz0PracticedCropCycleNode) pz0IdToObject.get(Pz0PracticedCropCycleNode.class, csvNodeId);
            cropCode = node.getNodeDto().getCroppingPlanEntryCode();
        }
        return cropCode;
    }

    protected boolean validType(ImportResults importResults, long line, PracticedInterventionImportDto from, PracticedInterventionDto to, boolean error) {
        String typeName = from.getTypeName();
        if(StringUtils.isNotBlank(typeName)){
            try {
                AgrosystInterventionType type = AgrosystInterventionType.valueOf(typeName);
                to.setType(type);
            } catch (IllegalArgumentException e) {
                error = true;
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, le type d'intervention %s n'est pas reconnu.", typeName));
            }
        } else {
            error = true;
            importResults.addErrorLine(line, "INTERVENTION IGNORÉE!, la colonne 'type' n'est pas renseigné.");

        }
        return error;
    }

    protected boolean validRank(ImportResults importResults, long line, PracticedInterventionImportDto from, PracticedInterventionDto to, boolean error) {
        Integer rank = from.getRankNumber();
        if (rank != null) {
            to.setRank(rank);
        } else {
            error = true;
            importResults.addErrorLine(line, "INTERVENTION IGNORÉE!, la colonne 'rank' n'est pas renseigné.");
        }
        return error;
    }

    protected boolean validName(ImportResults importResults, long line, PracticedInterventionDto to, boolean error) {
        String name = to.getName();
        if (StringUtils.isBlank(name)) {
            error = true;
            importResults.addErrorLine(line, "INTERVENTION IGNORÉE!, la colonne 'rank' n'est pas renseigné.");
        }
        return error;
    }

    protected boolean validStatingPeriod(ImportResults importResults, long line, PracticedInterventionDto to, boolean error) {
        String startingPeriodDate = to.getStartingPeriodDate();
        if (StringUtils.isNotBlank(startingPeriodDate)) {
            Matcher matcher = DATE_PATTERN.matcher(startingPeriodDate);
            if (!matcher.matches()) {
                error = true;
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, la date d'intervention ou date de début %s n'est pas reconnue.", startingPeriodDate));
            }
        } else {
            error = true;
            importResults.addErrorLine(line, "INTERVENTION IGNORÉE!, la colonne 'startingperioddate' n'est pas renseigné.");
        }
        return error;
    }

    protected boolean validEndingPeriod(ImportResults importResults, long line, PracticedInterventionDto to, boolean error) {
        String endingPeriodDate = to.getEndingPeriodDate();
        if (StringUtils.isNotBlank(endingPeriodDate)) {
            Matcher matcher = DATE_PATTERN.matcher(endingPeriodDate);
            if (!matcher.matches()) {
                error = true;
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, la date de fin d'intervention %s n'est pas reconnue.", endingPeriodDate));
            }
        } else if (to.getStartingPeriodDate() != null) {
            to.setEndingPeriodDate(to.getStartingPeriodDate());
        } else {
            error = true;
            importResults.addErrorLine(line, "INTERVENTION IGNORÉE!, la colonne 'endingperioddate' n'est pas renseigné.");
        }
        return error;
    }

    protected boolean validTransitVolumeUnit(ImportResults importResults, long line, PracticedInterventionImportDto from, PracticedInterventionDto to, boolean error) {
        String transitVolumeName = from.getTransitVolumeUnitName();
        if (StringUtils.isNotBlank(transitVolumeName)) {
            try {
                MaterielTransportUnit type = MaterielTransportUnit.valueOf(transitVolumeName);
                to.setTransitVolumeUnit(type);
            } catch (IllegalArgumentException e) {
                error = true;
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, l'unité de volume par voyage %s n'est pas reconnu.", transitVolumeName));
            }
        }
        return error;
    }

    protected boolean validWorkRateUnit(ImportResults importResults, long line, PracticedInterventionImportDto from, PracticedInterventionDto to, boolean error) {
        String workRateUnitName = from.getWorkRateUnitName();
        if (StringUtils.isNotBlank(workRateUnitName)) {
            try {
                MaterielWorkRateUnit type = MaterielWorkRateUnit.valueOf(workRateUnitName);
                to.setWorkRateUnit(type);
            } catch (IllegalArgumentException e) {
                error = true;
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, l'unité de temps de travail %s n'est pas reconnue.", workRateUnitName));
            }
        }
        return error;
    }


}
