package fr.inra.agrosyst.services.pz0import.input;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: InputImporter.java 4923 2015-05-05 15:53:56Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.4.9/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/input/InputImporter.java $
 * %%
 * Copyright (C) 2013 - 2015 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.collect.Sets;
import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.action.AbstractAction;
import fr.inra.agrosyst.api.entities.action.AbstractInput;
import fr.inra.agrosyst.api.entities.action.BiologicalControlAction;
import fr.inra.agrosyst.api.entities.action.BiologicalProductInput;
import fr.inra.agrosyst.api.entities.action.BiologicalProductInputImpl;
import fr.inra.agrosyst.api.entities.action.MineralFertilizersSpreadingAction;
import fr.inra.agrosyst.api.entities.action.MineralProductInput;
import fr.inra.agrosyst.api.entities.action.MineralProductInputImpl;
import fr.inra.agrosyst.api.entities.action.OrganicFertilizersSpreadingAction;
import fr.inra.agrosyst.api.entities.action.OrganicProductInput;
import fr.inra.agrosyst.api.entities.action.OrganicProductInputImpl;
import fr.inra.agrosyst.api.entities.action.OrganicProductUnit;
import fr.inra.agrosyst.api.entities.action.OtherAction;
import fr.inra.agrosyst.api.entities.action.OtherProductInput;
import fr.inra.agrosyst.api.entities.action.OtherProductInputImpl;
import fr.inra.agrosyst.api.entities.action.PesticideProductInput;
import fr.inra.agrosyst.api.entities.action.PesticideProductInputImpl;
import fr.inra.agrosyst.api.entities.action.PesticidesSpreadingAction;
import fr.inra.agrosyst.api.entities.action.PhytoProductInput;
import fr.inra.agrosyst.api.entities.action.PhytoProductUnit;
import fr.inra.agrosyst.api.entities.action.SeedingAction;
import fr.inra.agrosyst.api.entities.action.SeedingActionSpecies;
import fr.inra.agrosyst.api.entities.action.SeedingProductInput;
import fr.inra.agrosyst.api.entities.action.SeedingProductInputImpl;
import fr.inra.agrosyst.api.entities.effective.EffectiveIntervention;
import fr.inra.agrosyst.api.entities.referential.RefActaTraitementsProduit;
import fr.inra.agrosyst.api.entities.referential.RefActaTraitementsProduitTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefFertiMinUNIFA;
import fr.inra.agrosyst.api.entities.referential.RefFertiMinUNIFATopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefFertiOrga;
import fr.inra.agrosyst.api.entities.referential.RefFertiOrgaTopiaDao;
import fr.inra.agrosyst.api.services.action.ActionService;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.effective.Pz0EffectiveIntervention;
import fr.inra.agrosyst.api.services.pz0.practicedSystem.Pz0PracticedIntervention;
import fr.inra.agrosyst.api.services.referential.ReferentialService;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.services.pz0import.AbstractCSVImporter;
import fr.inra.agrosyst.services.pz0import.input.dto.InputImportDto;
import fr.inra.agrosyst.services.pz0import.input.model.InputModel;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.InputStream;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Created by davidcosse on 24/02/15.
 **/
public class InputImporter extends AbstractCSVImporter {

    private static final Log log = LogFactory.getLog(InputImporter.class);

    protected ActionService actionService;

    protected RefFertiMinUNIFATopiaDao mineralProductDao;
    protected RefActaTraitementsProduitTopiaDao phytoProductDao;
    protected RefFertiOrgaTopiaDao organicProductDao;
    protected ReferentialService referentialService;
    protected Map<AgrosystInterventionType, List<String>> actaTreatmentProductTypes;

    public void setRefFertiMinUNIFATopiaDao(ReferentialService referentialService) {
        this.referentialService = referentialService;
    }

    protected Map<AgrosystInterventionType, List<String>> getActaTreatmentProductTypes() {
        if (actaTreatmentProductTypes == null || actaTreatmentProductTypes.isEmpty()) {
            actaTreatmentProductTypes = referentialService.getAllActiveActaTreatmentProductTypes();
        }
        return actaTreatmentProductTypes;
    }


    @Override
    public ImportResults importFromStream(InputStream is, Map<String, EntityAndDependencies> entitiesByCsvId) {
        ImportResults importResults = new ImportResults(AbstractInput.class);
        log.debug("importfromstream");

        InputModel model = new InputModel();
        // récupère le DTO
        Import<InputImportDto> importer = Import.newImport(model, is);

        // match the first csv line number with data (not header).
        long line = FIRST_LINE_NUMBER;
        for (InputImportDto dto : importer) {

            // valid required fields
            AbstractInput input = createInput(dto, importResults, line);
            Boolean error = input == null;

            error = validAction(importResults, input, line, dto, error);

            if (!error) {
                Pz0PracticedIntervention practicedInterventionDto = (Pz0PracticedIntervention) pz0IdToObject.get(AbstractAction.class, Pz0PracticedIntervention.class, dto.getPz0ActionId());
                Pz0EffectiveIntervention effectiveInterventionDto = (Pz0EffectiveIntervention) pz0IdToObject.get(AbstractAction.class, Pz0EffectiveIntervention.class, dto.getPz0ActionId());

                if (practicedInterventionDto != null) {
                    practicedInterventionDto.getPracticedInterventionDto().addInput(input);
                    importResults.addInfoLine(line, String.format("AJOUT DE L'INTRANT '%s' VALIDÉE", input.getInputType()));
                    importResults.increaseAddedRecords();
                } else if(effectiveInterventionDto != null) {
                    effectiveInterventionDto.getEffectiveInterventionDto().addInput(input);
                    importResults.addInfoLine(line, String.format("AJOUT DE L'INTRANT '%s' VALIDÉE", input.getInputType()));
                    importResults.increaseAddedRecords();
                } else {
                    importResults.increaseIgnoredRecords();
                    importResults.addErrorLine(line, "AJOUT DE L'INTRANT IGNORÉE, intervention non retrouvée");
                }
            } else {
                importResults.increaseIgnoredRecords();
                importResults.addErrorLine(line, "AJOUT DE L'INTRANT IGNORÉE");
            }
            line++;
        }
        return importResults;
    }

    protected Boolean validPhytoProductInput(PhytoProductInput input, InputImportDto from, ImportResults importResults, long line, Boolean error) {
        if (StringUtils.isNotBlank(from.getPhytoProductId())) {
            RefActaTraitementsProduit phytoProduct = phytoProductDao.forTopiaIdEquals(from.getPhytoProductId()).findUniqueOrNull();

            if (phytoProduct != null) {
                if(input != null) {
                    input.setPhytoProduct(phytoProduct);
                }
            } else {
                importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT '%s' IGNORÉ!, le produit avec comme identifiant %s n'a pas été retrouvée.", from.getInputTypeName(), from.getPhytoProductId()));
            }

        } else {
            importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT '%s' IGNORÉ!, la colonne 'phytoproduct' n'est pas renseignée.",from.getInputTypeName()));
        }

        if (StringUtils.isNotBlank(from.getPhytoProductUnitName())) {
            try {
                PhytoProductUnit phytoProductUnit = PhytoProductUnit.valueOf(from.getPhytoProductUnitName());
                if (input != null) {
                    input.setPhytoProductUnit(phytoProductUnit);
                }
            } catch (IllegalArgumentException e) {
                error = true;
                importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT '%s' IGNORÉ!, unité de produit phytosanitaire non reconnue: %s.",from.getInputTypeName(), from.getPhytoProductUnitName()));
            }
        }
        return error;
    }

    protected AbstractAction validAndGetAction(String actionId, String inputTypeName, ImportResults importResults, Long line) {
        AbstractAction action = null;
        if (StringUtils.isNotBlank(actionId)) {
            action = (AbstractAction) pz0IdToObject.get(AbstractAction.class, actionId);
            if (action == null) {
                importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT '%s' IGNORÉ!, l'action avec comme identifiant %s n'a pas été retrouvée.", inputTypeName, actionId));
            }
        } else {
            importResults.addErrorLine(line, "AJOUT DE L'INTRANT IGNORÉ!, le type d'intrant ne correspond pas à l'action associé.");
        }
        return action;
    }


    protected AbstractInput createInput(InputImportDto from, ImportResults importResults, long line) {
        AbstractInput input = null;
        AgrosystInterventionType type = null;
        if (StringUtils.isNotBlank(from.getInputTypeName())) {
            try {
                type = AgrosystInterventionType.valueOf(from.getInputTypeName());
                switch (type) {
                    case APPLICATION_DE_PRODUITS_FERTILISANTS_MINERAUX:
                        input = new MineralProductInputImpl();
                        break;
                    case EPANDAGES_ORGANIQUES:
                        input = new OrganicProductInputImpl();
                        break;
                    case APPLICATION_DE_PRODUITS_PHYTOSANITAIRES:
                        input = new PesticideProductInputImpl();
                        break;
                    case LUTTE_BIOLOGIQUE:
                        input = new BiologicalProductInputImpl();
                        break;
                    case SEMIS:
                        input = new SeedingProductInputImpl();
                        break;
                    case AUTRE:
                        input = new OtherProductInputImpl();
                        break;
                    default:
                        importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT IGNORÉ!, type d'intrant non reconnu %s.", from.getInputTypeName()));
                }
            } catch(IllegalArgumentException e){
                importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT IGNORÉ!, type d'intrant non reconnu %s.", from.getInputTypeName()));
            }
        } else {
            importResults.addErrorLine(line, "AJOUT DE L'INTRANT IGNORÉ!, la colonne 'inputtype' n'est pas renseignée.");
        }
        if (input != null) {
            input.setInputType(type);
        }
        return input;
    }

    protected boolean validAction(ImportResults importResults, AbstractInput input, long line, InputImportDto from, Boolean error) {

        if (input != null) {
            AgrosystInterventionType type = input.getInputType();

            if (type != null) {
                AbstractAction action = null;
                try {
                    if(AgrosystInterventionType.APPLICATION_DE_PRODUITS_FERTILISANTS_MINERAUX.equals(type)) {

                        action = validAndGetAction(from.getMineralFertilizersSpreadingActionId(), from.getInputTypeName(), importResults, line);
                        if (action != null) {
                            ((MineralProductInput)input).setMineralFertilizersSpreadingAction((MineralFertilizersSpreadingAction) action);
                        } else {
                            error = true;
                        }

                        if (StringUtils.isNotBlank(from.getMineralProductId())) {
                            RefFertiMinUNIFA mineralProduct = (RefFertiMinUNIFA) pz0IdToObject.get(RefFertiMinUNIFA.class, from.getMineralProductId());
                            if (mineralProduct == null) {
                                mineralProduct = mineralProductDao.forTopiaIdEquals(from.getMineralProductId()).findUniqueOrNull();
                            }
                            if (mineralProduct != null) {
                                ((MineralProductInput)input).setMineralProduct(mineralProduct);

                            } else {
                                error = true;
                                importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT 'MINERA'L IGNORÉ!, le produit avec comme identifiant %s n'a pas été retrouvé.", from.getMineralProductId()));
                            }
                        } else {
                            error = true;
                            importResults.addErrorLine(line, "AJOUT DE L'INTRANT 'MINERAL' IGNORÉ!, la colonne 'mineralproduct' n'est pas renseignée.");
                        }


                    } else if (AgrosystInterventionType.AUTRE.equals(type)) {
                        action = validAndGetAction(from.getOtherActionId(), from.getInputTypeName(), importResults, line);
                        if (action != null) {
                            ((OtherProductInput)input).setOtherAction((OtherAction) action);
                        } else {
                            error = true;
                        }
                    } else if (AgrosystInterventionType.APPLICATION_DE_PRODUITS_PHYTOSANITAIRES.equals(type)) {
                        action = validAndGetAction(from.getPesticidesSpreadingActionId(), from.getInputTypeName(), importResults, line);
                        if (action != null) {
                            ((PesticideProductInput)input).setPesticidesSpreadingAction((PesticidesSpreadingAction) action);
                        } else {
                            error = true;
                        }

                    error = validPhytoProductInput((PesticideProductInput) input, from, importResults, line, error);

                    } else if (AgrosystInterventionType.SEMIS.equals(type)) {
                        SeedingAction seedingAction = (SeedingAction) validAndGetAction(from.getSeedingActionId(), from.getInputTypeName(), importResults, line);

                        if (seedingAction != null) {
                            Collection<SeedingActionSpecies> specieses = seedingAction.getSeedingSpecies();

                            if (CollectionUtils.isNotEmpty(specieses)) {
                                SeedingProductInput seedingProductInput = (SeedingProductInput) input;

                                if (StringUtils.isNotBlank(seedingProductInput.getProductType())) {
                                    List<String> productTypes = actaTreatmentProductTypes.get(AgrosystInterventionType.SEMIS);
                                    Set<String> allowedProductTypes = Sets.newHashSet();
                                    for (SeedingActionSpecies species : specieses) {
                                        if (species.isTreatment()) {
                                            //Fongicides
                                            allowedProductTypes.add(productTypes.get(0));
                                            //Insecticides
                                            allowedProductTypes.add(productTypes.get(2));
                                        }
                                        if (species.isBiologicalSeedInoculation()) {
                                            // "Inoculation biologique de semences ou plants"
                                            allowedProductTypes.add(productTypes.get(1));
                                        }
                                    }

                                    if (!allowedProductTypes.contains(seedingProductInput.getProductType())) {
                                        error = true;
                                        importResults.addErrorLine(line, String.format("Pour l'intrant avec identifiant %s, aucune espece de l'action 'Semis' avec comme identifiant %s n'a de traitement sélectionné pour ce type d'intrant '%s' !", from.getId(), from.getSeedingActionId(), seedingProductInput.getProductType()));
                                    }
                                }
                                seedingProductInput.setSeedingAction(seedingAction);
                            } else {
                                error = true;
                                importResults.addErrorLine(line, String.format("Aucune espece présente sur l'action 'Semis' avec comme identifiant %s pour l'intrant '%s' !", from.getSeedingActionId() ,from.getId()));
                            }
                            error = validPhytoProductInput((SeedingProductInput) input, from, importResults, line, error);
                        } else {
                            error = true;
                        }

                    } else if (AgrosystInterventionType.LUTTE_BIOLOGIQUE.equals(type)) {
                        action = validAndGetAction(from.getBiologicalControlActionId(), from.getInputTypeName(), importResults, line);
                        if (action != null) {
                            ((BiologicalProductInput)input).setBiologicalControlAction((BiologicalControlAction) action);
                        } else {
                            error = true;
                        }

                        error = validPhytoProductInput((BiologicalProductInput) input, from, importResults, line, error);
                    } else if (AgrosystInterventionType.EPANDAGES_ORGANIQUES.equals(type)) {
                        action = validAndGetAction(from.getOrganicFertilizersSpreadingActionId(), from.getInputTypeName(), importResults, line);

                        if (action != null) {
                            ((OrganicProductInput) input).setOrganicFertilizersSpreadingAction((OrganicFertilizersSpreadingAction) action);
                        } else {
                            error = true;
                        }

                        if (StringUtils.isNotBlank(from.getOrganicProductId())) {
                            RefFertiOrga product = organicProductDao.forTopiaIdEquals(from.getOrganicProductId()).findUniqueOrNull();
                            if (product != null) {
                                ((OrganicProductInput) (input)).setOrganicProduct(product);
                            } else {
                                error = true;
                                importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT '%s' IGNORÉ!, le produit avec comme identifiant %s n'a pas été retrouvée.", from.getInputTypeName(), from.getOrganicProductId()));
                            }
                        } else {
                            error = true;
                            importResults.addErrorLine(line, "AJOUT DE L'INTRANT '%s' IGNORÉ!, la colonne 'organicproduct' n'est pas renseignée.");
                        }

                        if (StringUtils.isNotBlank(from.getOrganicProductUnitName())) {
                            try {
                                OrganicProductUnit organicProductUnit = OrganicProductUnit.valueOf(from.getOrganicProductUnitName());
                                ((OrganicProductInput) input).setOrganicProductUnit(organicProductUnit);
                            } catch (IllegalArgumentException e) {
                                error = true;
                                importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT '%s' IGNORÉ!, unité de produit organique non reconnues %s.", from.getInputTypeName(), from.getOrganicProductUnitName()));
                            }
                        }
                    }

                } catch (ClassCastException e) {
                    error = true;
                    String actionTypeName = "inconnu";
                    if (action != null) {
                        EffectiveIntervention actionType = action.getEffectiveIntervention();
                        actionTypeName = actionType != null ? actionType.getName() : actionTypeName;
                    }
                    importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT '%s' IGNORÉ!, le type d'intrant %s ne correspond pas au type d'action %s.", from.getInputTypeName(), type.name(), actionTypeName));
                }

            }
        }


        if (!error) {
            Binder<InputImportDto, AbstractInput> binder = BinderFactory.newBinder(InputImportDto.class, AbstractInput.class);
            binder.copy(from, input);
        }

        return error;
    }


    public void init(ServiceContext serviceContext) {
        super.init(serviceContext);
        mineralProductDao = getPersistenceContext().getRefFertiMinUNIFADao();
        phytoProductDao = getPersistenceContext().getRefActaTraitementsProduitDao();
        organicProductDao = getPersistenceContext().getRefFertiOrgaDao();
        referentialService = getServiceFactory().newService(ReferentialService.class);
    }

}
