package fr.inra.agrosyst.services.pz0import.growingSystem;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: GrowingSystemImporter.java 4907 2015-04-24 09:37:31Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.4.9/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/growingSystem/GrowingSystemImporter.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import fr.inra.agrosyst.api.entities.GrowingPlan;
import fr.inra.agrosyst.api.entities.GrowingPlanTopiaDao;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.Sector;
import fr.inra.agrosyst.api.entities.managementmode.CategoryStrategy;
import fr.inra.agrosyst.api.entities.referential.RefTypeAgriculture;
import fr.inra.agrosyst.api.entities.referential.RefTypeAgricultureTopiaDao;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemService;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.growingSystem.GrowingSystemAndDependencies;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.services.pz0import.AbstractCSVImporter;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.nuiton.csv.Import;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.InputStream;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
*
* @author eancelet
* 
* TODO le cas échéant :
* * * * ajout des characteritics_growingsystem
* 
* 
*/


public class GrowingSystemImporter extends AbstractCSVImporter {

    protected GrowingSystemService growingSystemService;
    protected GrowingPlanTopiaDao growingPlanDao;
    protected RefTypeAgricultureTopiaDao refTypeAgricultureDao;
    
    @Override
    public ImportResults importFromStream(InputStream is, Map<String, EntityAndDependencies> growingPlansByCsvIds) {
        ImportResults importResults = new ImportResults(GrowingSystem.class);

        GrowingSystemImportModel model = new GrowingSystemImportModel();

        // récupère le DTO
        Import<GrowingSystemImportDto> importer = Import.newImport(model, is);

        // match the csv line number.
        long line = FIRST_LINE_NUMBER;
        for (GrowingSystemImportDto dto : importer) {
            boolean error = false;
            GrowingSystem growingSystem = growingSystemService.newGrowingSystem();

            Binder<GrowingSystemImportDto, GrowingSystem> growingSystemBinder = BinderFactory.newBinder(GrowingSystemImportDto.class, GrowingSystem.class);
            growingSystemBinder.copyExcluding(dto, growingSystem, GrowingSystem.PROPERTY_TOPIA_ID);

            // valid required fields
            error = validGrowingPlanAffiliation(importResults, line, dto, growingSystem, error);
            error = validGrowingSystemName(importResults, line, growingSystem, error);
            error = validGrowingSystemUpdateDate(importResults, line, growingSystem, error);
            error = validGrowingSystemSector(importResults, dto, line, growingSystem, error);

            error = addGrowingSystemCategoryStrategy(importResults, dto, line, growingSystem, error);

            // Type Agriculture
            String refTypeAgricultureTopiaId = getTypeAgricultureId(importResults, line, dto, growingSystem);

            error = validDephyNumberAffiliation(importResults, line, dto, growingSystem, error);
            error = validGrowingSystemCode(importResults, line, dto, growingSystem, error);

            pz0IdToObject.put(GrowingSystem.class, dto.getId(), growingSystem);
            pz0CodeToObject.put(GrowingSystem.class, dto.getCode(), growingSystem);

            if (!error) {
                importResults.addInfoLine(line, "SDC VALIDÉ, csvid: " + dto.getId());
                importResults.increaseAddedRecords();
            } else {
                importResults.increaseIgnoredRecords();
                importResults.addErrorLine(line, "SDC IGNORÉ csvid:" + dto.getId());
            }
            GrowingSystemAndDependencies growingSystemAndDependencies = new GrowingSystemAndDependencies(growingSystem, refTypeAgricultureTopiaId, dto.getGrowingPlanId());
            importResults.addEntity(dto.getId(), growingSystemAndDependencies);

            line++;
        }


        return importResults;
    }

    protected String getTypeAgricultureId(ImportResults importResults, long line, GrowingSystemImportDto dto, GrowingSystem growingSystem) {
        String refTypeAgricultureTopiaId = null;
        if (StringUtils.isNotBlank(dto.getTypeAgricultureId())) {
            RefTypeAgriculture refTypeAgriculture = refTypeAgricultureDao.forTopiaIdEquals(dto.getTypeAgricultureId()).findUniqueOrNull();
            if (refTypeAgriculture==null) {
                importResults.addInfoLine(line, String.format("ATTENTION ! le Type d'agriculture de topiaId %s n'existe pas en base de données pour le SdC %s.", dto.getTypeAgricultureId(), dto.getName()));
            }
            else {
                refTypeAgricultureTopiaId = refTypeAgriculture.getTopiaId();
                growingSystem.setTypeAgriculture(refTypeAgriculture);
            }
        }
        return refTypeAgricultureTopiaId;
    }

    protected boolean validDephyNumberAffiliation(ImportResults importResults, long line, GrowingSystemImportDto dto, GrowingSystem growingSystem, boolean error) {
        String dephyNumber = dto.getDephyNumber();
        if (StringUtils.isNotBlank(dephyNumber)) {
            Integer campaign = growingSystem.getGrowingPlan().getDomain().getCampaign();
            LinkedHashMap<Integer, List<String>> relatedGrowingSystems = growingSystemService.getGSCodeGPCodeDomainCodeByCampaignForGrowingSystemDephyId(dephyNumber);
            if (relatedGrowingSystems != null && !relatedGrowingSystems.isEmpty()) {
                if (!relatedGrowingSystems.containsKey(campaign)) {
                    List<String> codes = relatedGrowingSystems.get(0);
                    if (CollectionUtils.isNotEmpty(codes)) {
                        growingSystem.setCode(codes.get(0));
                        growingSystem.getGrowingPlan().setCode(codes.get(1));
                        growingSystem.getGrowingPlan().getDomain().setCode(codes.get(2));
                    }
                } else {
                    error = true;
                    importResults.addErrorLine(line, String.format("SDC IGNORÉ!, un system de culture dont le numéro dephy est %s existe déjà pour la campagne %d",dto.getDephyNumber(), campaign));
                }
            } else {
                importResults.addInfoLine(line, String.format("ATTENTION !, le système de culture avec comme id %s n'a pas retouvé d'affiliation avec un système de culture ayant comme numéro DEPHY: %s", dto.getId(), dto.getDephyNumber()));
            }
        }
        return error;
    }

    protected boolean validGrowingPlanAffiliation(ImportResults importResults, long line, GrowingSystemImportDto dto, GrowingSystem growingSystem, boolean error) {
        try {
            if (StringUtils.isNotBlank(dto.getGrowingPlanId())) {
                GrowingPlan growingPlan = (GrowingPlan)pz0IdToObject.get(GrowingPlan.class, dto.getGrowingPlanId());
                if (growingPlan != null) {
                    growingSystem.setGrowingPlan(growingPlan);
                } else {
                    importResults.addErrorLine(line, String.format("SDC IGNORÉ!, aucun dispositif n'est retrouvée avec l'identifiant suivant: %s",dto.getGrowingPlanId()));
                    error = true;
                }
            } else {
                importResults.addErrorLine(line, String.format("SDC IGNORÉ!, la colonne 'growingplan' n'est pas renseignée pour le SDC %s", dto.getName()));
                error = true;
            }
        } catch (ClassCastException e) {
            importResults.addErrorLine(line, String.format("SDC IGNORÉ!, dispositif non retrouvé pour le SDC %s", dto.getName()));
            error = true;
        }
        return error;

    }

    protected boolean validGrowingSystemCode(ImportResults importResults, long line, GrowingSystemImportDto dto, GrowingSystem growingSystem, boolean error) {
        String code = growingSystem.getCode();
        if (StringUtils.isNotBlank(code)) {
            if (StringUtils.isNotBlank(dto.getCode())) {
                pz0CodeToAgrosystCode.put(GrowingSystem.class, dto.getCode(), growingSystem.getCode());
            }
        } else if (StringUtils.isNotBlank(dto.getCode())) {
            growingSystem.setCode(csvCodeToAgrosystCode(GrowingSystem.class,  dto.getCode()));
        } else {
            importResults.addErrorLine(line, "SDC IGNORÉ!, le code du système de culture n'a pu être retrouvé et la colonne 'code' n'est pas renseignée.");
            error = true;
        }
        return error;
    }

    protected boolean validGrowingSystemName(ImportResults importResults, long line, GrowingSystem growingSystem, boolean error) {
        if (StringUtils.isBlank(growingSystem.getName())) {
            importResults.addErrorLine(line, "SDC IGNORÉ!, Le nom du SDC n'est pas renseignés.");
            error = true;
        }
        return error;
    }

    protected boolean validGrowingSystemSector(ImportResults importResults, GrowingSystemImportDto dto, long line, GrowingSystem growingSystem, boolean error) {
        Sector sector;
        String value = dto.getSectorImporterValue();
        if (StringUtils.isNotBlank(value)) {
            try {
                sector = Sector.valueOf(value);
                growingSystem.setSector(sector);
            } catch (IllegalArgumentException ex) {
                error = true;
                importResults.addErrorLine(line, String.format("SDC IGNORÉ! Secteur non valide: %s", value));
            }
        } else {
            importResults.addErrorLine(line, "SDC IGNORÉ!, la colonne 'sector' n'est pas renseignés.");
            error = true;
        }
        return error;

    }

    protected boolean validGrowingSystemUpdateDate(ImportResults importResults, long line, GrowingSystem growingSystem, boolean error) {
        if (growingSystem.getUpdateDate() == null) {
            importResults.addErrorLine(line, "SDC IGNORÉ!, la date de mise à jour n'est pas renseignés.");
            error = true;
        }
        return error;
    }

    protected boolean addGrowingSystemCategoryStrategy(ImportResults importResults, GrowingSystemImportDto dto, long line, GrowingSystem growingSystem, boolean error) {
        CategoryStrategy categoryStrategy = null;
        String value = dto.getCategoryStrategyImporterValue();
        if (StringUtils.isNotBlank(value)) {
            try {
                categoryStrategy = CategoryStrategy.valueOf(value);
            } catch (IllegalArgumentException ex) {
                error = true;
                importResults.addErrorLine(line, String.format("SDC IGNORÉ! Catégorie de stratégie : (growingSystem.categoryStrategy), non supportée: %s", value));
            }
        }
        growingSystem.setCategoryStrategy(categoryStrategy);
        return error;

    }

    @Override
    public void init(ServiceContext serviceContext) {
        super.init(serviceContext);
        refTypeAgricultureDao = getPersistenceContext().getRefTypeAgricultureDao();
        growingPlanDao = getPersistenceContext().getGrowingPlanDao();
        growingSystemService = getServiceFactory().newService(GrowingSystemService.class);
    }

}
