package fr.inra.agrosyst.services.pz0import.effective.effectivePerennialCropCycle;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: EffectivePerennialCropCycleImporter.java 4923 2015-05-05 15:53:56Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.4.9/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/effective/effectivePerennialCropCycle/EffectivePerennialCropCycleImporter.java $
 * %%
 * Copyright (C) 2013 - 2015 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import fr.inra.agrosyst.api.entities.OrchardFrutalForm;
import fr.inra.agrosyst.api.entities.PollinatorSpreadMode;
import fr.inra.agrosyst.api.entities.VineFrutalForm;
import fr.inra.agrosyst.api.entities.WeedType;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.effective.EffectivePerennialCropCycle;
import fr.inra.agrosyst.api.entities.referential.RefOrientationEDI;
import fr.inra.agrosyst.api.entities.referential.RefOrientationEDITopiaDao;
import fr.inra.agrosyst.api.services.domain.CroppingPlanEntryDto;
import fr.inra.agrosyst.api.services.effective.EffectivePerennialCropCycleDto;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.domains.DomainAndDependencies;
import fr.inra.agrosyst.api.services.pz0.effective.EffectiveCropCycleAndDependencies;
import fr.inra.agrosyst.api.services.pz0.effective.Pz0EffectiveCropCyclePhase;
import fr.inra.agrosyst.api.services.pz0.effective.Pz0EffectivePerennialCropCycle;
import fr.inra.agrosyst.api.services.pz0.plot.PlotAndDependencies;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.services.pz0import.AbstractCSVImporter;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.InputStream;
import java.util.Map;

/**
 * Created by davidcosse on 01/04/15.
 */
public class EffectivePerennialCropCycleImporter extends AbstractCSVImporter {

    private static final Log log = LogFactory.getLog(EffectivePerennialCropCycleImporter.class);

    protected RefOrientationEDITopiaDao orientationEDIDao;

    protected static final Integer LOWER_VALID_YEAR = 1700;
    protected static final Integer MAX_VALID_YEAR = 2100;

    @Override
    public ImportResults importFromStream(InputStream is, Map<String, EntityAndDependencies> entitiesByCsvId) {
        ImportResults importResults = new ImportResults(EffectivePerennialCropCycleDto.class);
        log.debug("importfromstream");

        EffectivePerennialCropCycleImportModel model = new EffectivePerennialCropCycleImportModel();

        Import<EffectivePerennialCropCycleImportDto> importer = Import.newImport(model, is);

        // match the first csv line number with data (not header).
        long line = FIRST_LINE_NUMBER;
        for (EffectivePerennialCropCycleImportDto dto : importer) {
            boolean error = false;

            Pz0EffectivePerennialCropCycle pz0EffectivePerennialCropCycle = new Pz0EffectivePerennialCropCycle(dto.getId());
            EffectivePerennialCropCycleDto effectivePerennialCropCycleDto = pz0EffectivePerennialCropCycle.getPerennialCropCycleDto();

            // EffectivePerennialCropCycle to persist
            Binder<EffectivePerennialCropCycleImportDto, EffectivePerennialCropCycleDto> binder = BinderFactory.newBinder(EffectivePerennialCropCycleImportDto.class, EffectivePerennialCropCycleDto.class);
            binder.copyExcluding(dto, effectivePerennialCropCycleDto,
                    EffectivePerennialCropCycle.PROPERTY_TOPIA_ID,
                    EffectivePerennialCropCycle.PROPERTY_CROPPING_PLAN_ENTRY,
                    EffectivePerennialCropCycle.PROPERTY_POLLINATOR,
                    EffectivePerennialCropCycle.PROPERTY_WEED_TYPE,
                    EffectivePerennialCropCycle.PROPERTY_POLLINATOR_SPREAD_MODE,
                    EffectivePerennialCropCycle.PROPERTY_ORCHARD_FRUTAL_FORM,
                    EffectivePerennialCropCycle.PROPERTY_VINE_FRUTAL_FORM,
                    EffectivePerennialCropCycle.PROPERTY_ORIENTATION);

            // valid required related objects
            error = validZoneAffiliation(importResults, line, dto, error);
            error = validCrops(importResults, line, dto, effectivePerennialCropCycleDto, error);
            error = validPhase(importResults, line, dto, pz0EffectivePerennialCropCycle, error);

            // valid required attributes
            error = validPollinator(importResults, line, dto, effectivePerennialCropCycleDto, error);

            // valid required enums
            error = validWeedType(importResults, line, dto, effectivePerennialCropCycleDto, error);

            // valid optional enums
            error = validPollinatorSpreadMode(importResults, line, dto, effectivePerennialCropCycleDto, error);
            error = validOrchardFrutalForm(importResults, line, dto, effectivePerennialCropCycleDto, error);
            error = validVineFrutalForm(importResults, line, dto, effectivePerennialCropCycleDto, error);

            // valid optional related objects
            error = validOrientation(importResults, line, dto, effectivePerennialCropCycleDto, error);

            // valid specific import control
            error = validPlantingDensity(importResults, line, dto, effectivePerennialCropCycleDto, error);
            error = validPlantingYear(importResults, line, dto, effectivePerennialCropCycleDto, error);

            addCycleToEntityAndDependencies(dto, pz0EffectivePerennialCropCycle);
            // need to be able to find it out EffectivePerennialCropCycleDto to add phase ont it
            pz0IdToObject.put(Pz0EffectivePerennialCropCycle.class, dto.getId(), pz0EffectivePerennialCropCycle);
            if (!error) {
                importResults.addInfoLine(line, "CYCLE DE CULTURE DU RÉALISÉ VALIDÉ, csvid: " + dto.getId());
                importResults.increaseAddedRecords();
            } else {
                importResults.increaseIgnoredRecords();
                importResults.addErrorLine(line, "CYCLE DE CULTURE DU RÉALISÉ IGNORÉ csvid:" + dto.getId());
            }


            line++;
        }
        return importResults;
    }

    protected void addCycleToEntityAndDependencies(EffectivePerennialCropCycleImportDto dto, Pz0EffectivePerennialCropCycle effectivePerennialCropCycleDto) {
        EffectiveCropCycleAndDependencies effectiveCropCycleAndDependencies = getEffectiveCropCycleAndDependencies(dto.getZoneId());
        effectiveCropCycleAndDependencies.addEffectivePerennialCropCycleDto(dto.getId(), effectivePerennialCropCycleDto);
    }


    protected boolean validZoneAffiliation(ImportResults importResults, long line, EffectivePerennialCropCycleImportDto dto, boolean error) {
        String target = dto.getZoneId();
        if (StringUtils.isNotBlank(target)) {
            PlotAndDependencies plotAndDependencies = (PlotAndDependencies) pz0IdToRelatedObjectId.get(Zone.class, PlotAndDependencies.class, target);
            if (plotAndDependencies == null) {
                importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU RÉALISÉ IGNORÉ!, aucune zone n'est retrouvée avec l'identifiant suivant: %s", target));
                error = true;
            }

        } else {
            importResults.addErrorLine(line, "CYCLE DE CULTURE PÉRENNE DU RÉALISÉ IGNORÉ!, la colonne 'zone' n'est pas renseignée");
            error = true;
        }
        return error;
    }


    protected boolean validCrops(ImportResults importResults, long line, EffectivePerennialCropCycleImportDto dto, EffectivePerennialCropCycleDto effectivePerennialCropCycleDto, boolean error) {
        String cropId = dto.getCroppingPlanEntryId();
        if (StringUtils.isNotBlank(cropId)) {
            try {
                DomainAndDependencies domainAndDependencies = getDomainAndDependenciesForZone(dto.getZoneId());
                if (domainAndDependencies != null) {
                    Map<String, CroppingPlanEntryDto> cropIds = domainAndDependencies.getCroppingPlanEntryDtosByPZ0CsvId();
                    if (cropIds != null) {
                        if (cropIds.get(cropId) != null){
                            effectivePerennialCropCycleDto.setCroppingPlanEntryId(cropId);
                        } else {
                            error = true;
                            importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU RÉALISÉ IGNORÉ!, la culture avec comme identifiant:%s n'a pas été retrouvée sur le domaine %s", cropId, domainAndDependencies.getCsvId()));
                        }
                    } else {
                        error = true;
                        importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU RÉALISÉ IGNORÉ!, la culture avec comme identifiant:%s n'a pas été retrouvée sur le domaine %s", cropId, domainAndDependencies.getCsvId()));
                    }
                } else {
                    error = true;
                    importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU RÉALISÉ IGNORÉ!, le domaine associé à la zone %s n'a pu être retrouvé", dto.getId()));
                }
            } catch (NullPointerException e) {
                error = true;
                importResults.addErrorLine(line, "CYCLE DE CULTURE PÉRENNE DU RÉALISÉ IGNORÉ!, l'affiliation de la culture au domaine n'a pu être vérifiée");
            }

        } else {
            error = true;
            importResults.addErrorLine(line, "CYCLE DE CULTURE PÉRENNE DU RÉALISÉ IGNORÉ!, la colone 'croppingplanentry' n'est pas renseignée");
        }
        return  error;
    }

    protected boolean validPhase(ImportResults importResults, long line, EffectivePerennialCropCycleImportDto dto, Pz0EffectivePerennialCropCycle effectivePerennialCropCycleDto, boolean error) {
        String phaseId = dto.getPhaseId();
        if (StringUtils.isNotBlank(phaseId)) {
            Pz0EffectiveCropCyclePhase pz0Phase = (Pz0EffectiveCropCyclePhase) pz0IdToObject.get(Pz0EffectiveCropCyclePhase.class, phaseId);
            if (pz0Phase != null) {
                effectivePerennialCropCycleDto.addPz0Phase(pz0Phase);
                pz0Phase.setPerennialCropCycleId(dto.getId());// use to be able to find out perennial cycle from phase
                pz0Phase.setZoneId(dto.getZoneId());
                pz0Phase.setCropId(dto.getCroppingPlanEntryId());
            } else {
                error = true;
                importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU RÉALISÉ IGNORÉ!, la phase avec comme identifiant:%s n'a pas été retrouvée", phaseId));

            }
        } else {
            error = true;
            importResults.addErrorLine(line, "CYCLE DE CULTURE PÉRENNE DU RÉALISÉ IGNORÉ!, la colone 'phase' n'est pas renseignée");
        }
        return  error;
    }

    protected boolean validPollinator(ImportResults importResults, long line, EffectivePerennialCropCycleImportDto dto, EffectivePerennialCropCycleDto effectivePerennialCropCycleDto, boolean error) {
        Boolean isPollinator = dto.isPollinator();
        if (isPollinator == null) {
            importResults.addErrorLine(line, "CYCLE DE CULTURE PÉRENNE DU RÉALISÉ IGNORÉ!, la colonne 'pollinator' n'est pas renseignée pour le cycle d'id: " + dto.getId());
            error = true;
        } else {
            effectivePerennialCropCycleDto.setPollinator(isPollinator);
        }
        return error;
    }

    protected boolean validWeedType(ImportResults importResults, long line, EffectivePerennialCropCycleImportDto dto, EffectivePerennialCropCycleDto effectivePerennialCropCycleDto, boolean error) {
        String weedTypeSt = dto.getWeedType();
        if (StringUtils.isNotBlank(weedTypeSt)) {
            WeedType weedType = null;
            try {
                weedType = WeedType.valueOf(weedTypeSt);
            } catch (IllegalArgumentException e) {
                importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU RÉALISÉ IGNORÉ!, type d'enherbement '%s' est non valide pour le cycle d'id: %s", weedTypeSt, dto.getId()));
                error = true;
            }
            effectivePerennialCropCycleDto.setWeedType(weedType);
        } else {
            importResults.addErrorLine(line, "CYCLE DE CULTURE PÉRENNE DU RÉALISÉ IGNORÉ!, la colonne 'weedType' n'est pas renseignée pour le cycle d'id: " + dto.getId());
            error = true;
        }
        return error;
    }

    protected boolean validPollinatorSpreadMode(ImportResults importResults, long line, EffectivePerennialCropCycleImportDto dto, EffectivePerennialCropCycleDto effectivePerennialCropCycleDto, boolean error) {
        String pollinatorSpreadSt = dto.getPollinatorSpreadMode();
        if (StringUtils.isNotBlank(pollinatorSpreadSt)) {
            PollinatorSpreadMode pollinatorSpreadMode = null;
            try {
                pollinatorSpreadMode = PollinatorSpreadMode.valueOf(pollinatorSpreadSt);
            } catch (IllegalArgumentException e) {
                importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU RÉALISÉ IGNORÉ!, mode de répartition des pollinisateurs '%s' non valide pour le cycle d'id: %s", pollinatorSpreadSt, dto.getId()));
                error = true;
            }
            effectivePerennialCropCycleDto.setPollinatorSpreadMode(pollinatorSpreadMode);
        }
        return error;
    }

    protected boolean validOrchardFrutalForm(ImportResults importResults, long line, EffectivePerennialCropCycleImportDto dto, EffectivePerennialCropCycleDto effectivePerennialCropCycleDto, boolean error) {
        String orchardFrutalFormSt = dto.getOrchardFrutalForm();
        if (StringUtils.isNotBlank(orchardFrutalFormSt)) {
            OrchardFrutalForm orchardFrutalForm = null;
            try {
                orchardFrutalForm = OrchardFrutalForm.valueOf(orchardFrutalFormSt);
            } catch (IllegalArgumentException e) {
                importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU RÉALISÉ IGNORÉ!, forme fruitière vergers '%s' non valide pour le cycle d'id: %s", orchardFrutalFormSt, dto.getId()));
                error = true;
            }
            effectivePerennialCropCycleDto.setOrchardFrutalForm(orchardFrutalForm);
        }
        return error;
    }

    protected boolean validVineFrutalForm(ImportResults importResults, long line, EffectivePerennialCropCycleImportDto dto, EffectivePerennialCropCycleDto effectivePerennialCropCycleDto, boolean error) {
        String vineFrutalFormSt = dto.getVineFrutalForm();
        if (StringUtils.isNotBlank(vineFrutalFormSt)) {
            VineFrutalForm frutalForm = null;
            try {
                frutalForm = VineFrutalForm.valueOf(vineFrutalFormSt);
            } catch (IllegalArgumentException e) {
                importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU RÉALISÉ IGNORÉ!, Forme fruitière vigne '%s' non valide pour le cycle d'id: %s", vineFrutalFormSt, dto.getId()));
                error = true;
            }
            effectivePerennialCropCycleDto.setVineFrutalForm(frutalForm);
        }
        return error;
    }

    protected boolean validOrientation(ImportResults importResults, long line, EffectivePerennialCropCycleImportDto dto, EffectivePerennialCropCycleDto effectivePerennialCropCycleDto, boolean error) {
        String orientationId = dto.getOrientation();
        if (StringUtils.isNotBlank(orientationId)) {
            RefOrientationEDI orientationEDI = orientationEDIDao.forTopiaIdEquals(orientationId).findUniqueOrNull();
            if (orientationEDI != null) {
                effectivePerennialCropCycleDto.setOrientation(orientationEDI);
            } else {
                error = true;
                importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU RÉALISÉ IGNORÉ!, l'orientation des rangs avec comme identifiant %s n'a pas été retrouvée pour le cycle d'id: %s", orientationId, dto.getId()));
            }
        }
        return error;
    }

    protected boolean validPlantingDensity(ImportResults importResults, long line, EffectivePerennialCropCycleImportDto dto, EffectivePerennialCropCycleDto effectivePerennialCropCycleDto, boolean error) {
        // A renseigner si différent de : 10000 / ((interFurrow / 100) * (spacing / 100)
        if ((effectivePerennialCropCycleDto.getPlantingInterFurrow() == null || effectivePerennialCropCycleDto.getPlantingSpacing() == null) && effectivePerennialCropCycleDto.getPlantingDensity() == null) {
            importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU RÉALISÉ IGNORÉ!, la densité de plantation doit être renseigné " +
                    "dans le cas ou celle-ci ne peut pas être calculée à partir de " +
                    "l'inter-rang de plantation et l'espacement de plantation sur le rang pour le cycle d'id: %s", dto.getId()));
            error = true;
        }

        if (effectivePerennialCropCycleDto.getPlantingDensity() != null && (effectivePerennialCropCycleDto.getPlantingInterFurrow() != null && effectivePerennialCropCycleDto.getPlantingSpacing() != null)) {
            double plantingDensity = 10000 / ((effectivePerennialCropCycleDto.getPlantingInterFurrow() / 100) * (effectivePerennialCropCycleDto.getPlantingSpacing() / 100));
            if ((effectivePerennialCropCycleDto.getPlantingDensity().compareTo(plantingDensity) != 0)) {
                importResults.addErrorLine(line, String.format(
                        "CYCLE DE CULTURE PÉRENNE DU RÉALISÉ IGNORÉ!, conflit sur la densité de plantation pour le cycle d'id: %s, la densité de plantation est renseignée " +
                        "alors que l'inter-rang de plantation et l'espacement de plantation sur le rang le sont aussi. Valeur importée: %f, valeur calculée: %f", dto.getId(), effectivePerennialCropCycleDto.getPlantingDensity(), plantingDensity));
                error = true;
            }
        } else if (effectivePerennialCropCycleDto.getPlantingInterFurrow() != null && effectivePerennialCropCycleDto.getPlantingSpacing() != null) {
            double plantingDensity = 10000 / ((effectivePerennialCropCycleDto.getPlantingInterFurrow() / 100) * (effectivePerennialCropCycleDto.getPlantingSpacing() / 100));
            effectivePerennialCropCycleDto.setPlantingDensity(plantingDensity);
        }
        return error;
    }

    protected boolean validPlantingYear(ImportResults importResults, long line, EffectivePerennialCropCycleImportDto dto, EffectivePerennialCropCycleDto effectivePerennialCropCycleDto, boolean error) {
        Integer plantingYear = dto.getPlantingYear();
        if (plantingYear != null) {
            if (plantingYear > LOWER_VALID_YEAR && plantingYear < MAX_VALID_YEAR) {
                effectivePerennialCropCycleDto.setPlantingYear(Integer.valueOf(plantingYear));
            } else {
                error = true;
                importResults.addErrorLine(line, String.format("CYCLE DE CULTURE PÉRENNE DU RÉALISÉ IGNORÉ!, année de plantation %s non valide pour le cycle d'id: %s", dto.getPlantingYear(), dto.getId()));
            }
        }
        return error;
    }

    @Override
    public void init(ServiceContext serviceContext) {
        super.init(serviceContext);
        orientationEDIDao = getPersistenceContext().getRefOrientationEDIDao();
    }

}
