package fr.inra.agrosyst.services.pz0import.effective.effectiveIntervention;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: EffectiveInterventionImporter.java 4923 2015-05-05 15:53:56Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.4.9/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/effective/effectiveIntervention/EffectiveInterventionImporter.java $
 * %%
 * Copyright (C) 2013 - 2015 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.MaterielTransportUnit;
import fr.inra.agrosyst.api.entities.MaterielWorkRateUnit;
import fr.inra.agrosyst.api.entities.effective.EffectiveIntervention;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCycleConnectionDto;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCycleNodeDto;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCyclePhaseDto;
import fr.inra.agrosyst.api.services.effective.EffectiveInterventionDto;
import fr.inra.agrosyst.api.services.effective.EffectiveSeasonalCropCycleDto;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.effective.EffectiveCropCycleAndDependencies;
import fr.inra.agrosyst.api.services.pz0.effective.Pz0EffectiveCropCycleNode;
import fr.inra.agrosyst.api.services.pz0.effective.Pz0EffectiveCropCyclePhase;
import fr.inra.agrosyst.api.services.pz0.effective.Pz0EffectiveIntervention;
import fr.inra.agrosyst.services.pz0import.AbstractCSVImporter;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.InputStream;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Map;

/**
 * Created by davidcosse on 13/04/15.
 */
public class EffectiveInterventionImporter extends AbstractCSVImporter {

    private static final Log log = LogFactory.getLog(EffectiveInterventionImporter.class);
    public static final String YYYY_MM_DD_HH_MM_SS_SSS = "yyyy-MM-dd HH:mm:ss.SSS";

    @Override
    public ImportResults importFromStream(InputStream is, Map<String, EntityAndDependencies> entitiesByCsvId) {
        ImportResults importResults = new ImportResults(EffectiveIntervention.class);
        log.debug("importfromstream");

        EffectiveInterventionImportModel model = new EffectiveInterventionImportModel();
        // récupère le DTO
        Import<EffectiveInterventionImportDto> importer = Import.newImport(model, is);

        // match the first csv line number with data (not header).
        long line = FIRST_LINE_NUMBER;
        for (EffectiveInterventionImportDto dto : importer) {
            boolean error = false;

            Pz0EffectiveIntervention interventionImport = new Pz0EffectiveIntervention(dto.getId());
            EffectiveInterventionDto interventionDto = interventionImport.getEffectiveInterventionDto();

            Binder<EffectiveInterventionImportDto, EffectiveInterventionDto> binder = BinderFactory.newBinder(EffectiveInterventionImportDto.class, EffectiveInterventionDto.class);
            binder.copy(dto, interventionDto);

            // valid required related objets
            error = validPhaseOrNodeFields(importResults, line, dto, error);
            error = validPhase(importResults, line, dto, error);
            error = validNode(importResults, line, dto, error);
            error = addInterventionToPhaseOrNode(importResults, line, dto, interventionImport, error);

            // valid required Enum
            error = validType(importResults, line, dto, interventionDto, error);

            // valid required field
            error = validName(importResults, line, interventionDto, error);
            error = validStartInterventionDate(importResults, line, dto, interventionDto, error);
            error = validEndInterventionDate(importResults, line, dto, interventionDto, error);
            error = validTransitCount(importResults, line, dto, interventionDto, error);

            // valid optional Enums
            error = validTransitVolumeUnit(importResults, line, dto, interventionDto, error);
            error = validWorkRateUnit(importResults, line, dto, interventionDto, error);

            pz0IdToObject.put(Pz0EffectiveIntervention.class, dto.getId(), interventionImport);

            if (!error) {
                importResults.addInfoLine(line, "INTERVENTION DU RÉALISÉ VALIDÉE, csvid: " + dto.getId());
                importResults.increaseAddedRecords();
            } else {
                importResults.increaseIgnoredRecords();
                importResults.addErrorLine(line, "INTERVENTION DU RÉALISÉ IGNORÉE csvid:" + dto.getId());
            }
            line++;
        }
        return importResults;
    }

    protected boolean validPhaseOrNodeFields(ImportResults importResults, long line, EffectiveInterventionImportDto from, boolean error) {
        String cropCyclePhaseId = from.getEffectiveCropCyclePhaseId();
        String cropCycleNodeId = from.getEffectiveCropCycleNodeId();
        if (StringUtils.isBlank(cropCyclePhaseId) && StringUtils.isBlank(cropCycleNodeId)) {
            importResults.addErrorLine(line, "INTERVENTION IGNORÉE!, une des colonnes suivante doit-être renseigné: 'effectivecropcyclephase' ou 'effectivecropcyclenode' mais aucune ne l'est.");
            error = true;
        }
        return error;
    }

    protected boolean validPhase(ImportResults importResults, long line, EffectiveInterventionImportDto from, boolean error) {
        String phaseId = from.getEffectiveCropCyclePhaseId();
        if (!error && StringUtils.isNotBlank(phaseId)) {
            Pz0EffectiveCropCyclePhase phaseDto = (Pz0EffectiveCropCyclePhase) pz0IdToObject.get(Pz0EffectiveCropCyclePhase.class, phaseId);
            if (phaseDto == null) {
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, la phase avec comme id %s n'esiste pas.", phaseId));
                error = true;
            }

        }
        return error;
    }

    protected boolean validNode(ImportResults importResults, long line, EffectiveInterventionImportDto from, boolean error) {
        String nodeId = from.getEffectiveCropCycleNodeId();
        if (!error && StringUtils.isNotBlank(nodeId)) {
            Pz0EffectiveCropCycleNode node = (Pz0EffectiveCropCycleNode) pz0IdToObject.get(Pz0EffectiveCropCycleNode.class, nodeId);
            if (node == null) {
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, le noeud avec comme id %s n'esiste pas.", nodeId));
                error = true;
            }
        }
        return error;
    }

    protected boolean addInterventionToPhaseOrNode(ImportResults importResults, long line, EffectiveInterventionImportDto from, Pz0EffectiveIntervention pz0Intervention, boolean error) {
        String nodeId = from.getEffectiveCropCycleNodeId();
        String phaseId = from.getEffectiveCropCyclePhaseId();

        Pz0EffectiveCropCycleNode node = null;
        Pz0EffectiveCropCyclePhase phaseDto = null;

        if (StringUtils.isNotBlank(nodeId)) {
            node = (Pz0EffectiveCropCycleNode) pz0IdToObject.get(Pz0EffectiveCropCycleNode.class, nodeId);
            if (node != null) {
                node.addEffectivePz0Intervention(pz0Intervention);
            } else {
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, le noeud avec comme id %s n'esiste pas.", nodeId));
                error = true;
            }
            pz0IdToObject.put(EffectiveInterventionDto.class, EffectiveCropCycleNodeDto.class, from.getId(), nodeId);
        } else if (StringUtils.isNotBlank(phaseId)) {
            phaseDto = (Pz0EffectiveCropCyclePhase) pz0IdToObject.get(Pz0EffectiveCropCyclePhase.class, phaseId);
            if (phaseDto != null) {
                phaseDto.addEffectivePz0Intervention(pz0Intervention);
            } else {
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, la phase avec comme id %s n'esiste pas.", phaseId));
                error = true;
            }
            pz0IdToObject.put(EffectiveInterventionDto.class, EffectiveCropCyclePhaseDto.class, from.getId(), phaseId);
        }

        addTargetedZoneIdToIntervention(pz0Intervention, node, phaseDto);
        addTargetedCropIdToIntervention(pz0Intervention, node, phaseDto);

        return error;
    }

    protected void addTargetedCropIdToIntervention(Pz0EffectiveIntervention to, Pz0EffectiveCropCycleNode pz0Node, Pz0EffectiveCropCyclePhase pz0Phase) {
        if (pz0Node != null) {
            addTargetedCropIdToIntervention(to, pz0Node);
        }
        if (pz0Phase != null) {
            addTargetedCropIdToIntervention(to, pz0Phase);
        }
    }

    protected void addTargetedCropIdToIntervention(Pz0EffectiveIntervention to, Pz0EffectiveCropCycleNode pz0Node) {
        if (to.getEffectiveInterventionDto().isIntermediateCrop()) {
            String targetId = pz0Node.getTargetId();
            if (StringUtils.isNotBlank(targetId)) {
                EffectiveCropCycleConnectionDto connectionDto = (EffectiveCropCycleConnectionDto) pz0IdToObject.get(EffectiveCropCycleConnectionDto.class, targetId);
                if (connectionDto != null) {
                    to.setCropId(connectionDto.getIntermediateCroppingPlanEntryId());
                }
            }
        } else {
            to.setCropId(pz0Node.getNode().getCroppingPlanEntryId());
        }
    }

    protected void addTargetedCropIdToIntervention(Pz0EffectiveIntervention to, Pz0EffectiveCropCyclePhase pz0Phase) {
        to.setCropId(pz0Phase.getCropId());
    }

    protected void addTargetedZoneIdToIntervention(Pz0EffectiveIntervention to, Pz0EffectiveCropCycleNode pz0Node, Pz0EffectiveCropCyclePhase pz0Phase) {
        if (pz0Node != null) {
            addTargetedZoneIdToIntervention(to, pz0Node);
        }
        if (pz0Phase != null) {
            addTargetedZoneIdToIntervention(to, pz0Phase);
        }
    }

    protected void addTargetedZoneIdToIntervention(Pz0EffectiveIntervention to, Pz0EffectiveCropCycleNode pz0Node) {
        String effectiveSeasonalCropCycleId = pz0Node.getEffectiveSeasonalCropCycleId();
        if (StringUtils.isNotBlank(effectiveSeasonalCropCycleId)) {
            EffectiveCropCycleAndDependencies effectiveCropCycleAndDependencies = (EffectiveCropCycleAndDependencies) pz0IdToObject.get(EffectiveSeasonalCropCycleDto.class, EffectiveCropCycleAndDependencies.class, effectiveSeasonalCropCycleId);
            to.setZoneId(effectiveCropCycleAndDependencies.getZoneId());
        }
    }

    protected void addTargetedZoneIdToIntervention(Pz0EffectiveIntervention to, Pz0EffectiveCropCyclePhase pz0Phase) {
        String zoneId = pz0Phase.getZoneId();
        to.setZoneId(zoneId);
    }

    protected boolean validType(ImportResults importResults, long line, EffectiveInterventionImportDto from, EffectiveInterventionDto to, boolean error) {
        String typeName = from.getTypeName();
        if (StringUtils.isNotBlank(typeName)) {
            try {
                AgrosystInterventionType type = AgrosystInterventionType.valueOf(typeName);
                to.setType(type);
            } catch (IllegalArgumentException e) {
                error = true;
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, le type d'intervention %s n'est pas reconnu.", typeName));
            }
        } else {
            error = true;
            importResults.addErrorLine(line, "INTERVENTION IGNORÉE!, la colonne 'type' n'est pas renseignée.");

        }
        return error;
    }

    protected boolean validName(ImportResults importResults, long line, EffectiveInterventionDto to, boolean error) {
        String name = to.getName();
        if (StringUtils.isBlank(name)) {
            error = true;
            importResults.addErrorLine(line, "INTERVENTION IGNORÉE!, la colonne 'name' n'est pas renseignée.");
        }
        return error;
    }

    protected boolean validStartInterventionDate(ImportResults importResults, long line, EffectiveInterventionImportDto from, EffectiveInterventionDto to, boolean error) {
        String startInterventionDate = from.getStartInterventionDateSt();
        if (StringUtils.isNotBlank(startInterventionDate)) {
            SimpleDateFormat format = new SimpleDateFormat(YYYY_MM_DD_HH_MM_SS_SSS);
            try {
                Date date = format.parse(startInterventionDate);
                to.setStartInterventionDate(date);
            } catch (ParseException e) {
                error = true;
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, la date de début d'intervention %s n'est pas valide.", startInterventionDate));
            }

        } else {
            error = true;
            importResults.addErrorLine(line, "INTERVENTION IGNORÉE!, la colonne 'startinterventiondate' n'est pas renseignée.");
        }
        return error;
    }

    protected boolean validEndInterventionDate(ImportResults importResults, long line, EffectiveInterventionImportDto from, EffectiveInterventionDto to, boolean error) {
        String endInterventionDateSt = from.getEndInterventionDateSt();
        if (StringUtils.isNotBlank(endInterventionDateSt)) {
            SimpleDateFormat format = new SimpleDateFormat(YYYY_MM_DD_HH_MM_SS_SSS);
            try {
                Date date = format.parse(endInterventionDateSt);
                to.setEndInterventionDate(date);
            } catch (ParseException e) {
                error = true;
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, la date de fin d'intervention %s n'est pas valide.", endInterventionDateSt));
            }

        } else if (to.getStartInterventionDate() != null){
            to.setEndInterventionDate(to.getStartInterventionDate());
        } else {
            error = true;
            importResults.addErrorLine(line, "INTERVENTION IGNORÉE!, la colonne 'endinterventiondate' n'est pas renseignée.");
        }
        return error;
    }

    protected boolean validTransitCount(ImportResults importResults, long line, EffectiveInterventionImportDto from, EffectiveInterventionDto to, boolean error) {
        Integer transitCount = from.getTransitCountInteger();
        if (transitCount != null) {
            to.setTransitCount(transitCount);
        } else {
            error = true;
            importResults.addErrorLine(line, "INTERVENTION IGNORÉE!, la colonne 'transitcount' n'est pas renseignée.");
        }
        return error;
    }

    protected boolean validTransitVolumeUnit(ImportResults importResults, long line, EffectiveInterventionImportDto from, EffectiveInterventionDto to, boolean error) {
        String transitVoulmeName = from.getTransitVolumeUnitName();
        if (StringUtils.isNotBlank(transitVoulmeName)) {
            try {
                MaterielTransportUnit type = MaterielTransportUnit.valueOf(transitVoulmeName);
                to.setTransitVolumeUnit(type);
            } catch (IllegalArgumentException e) {
                error = true;
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, l'unité de volume par voyage %s n'est pas reconnu.", transitVoulmeName));
            }
        }
        return error;
    }

    protected boolean validWorkRateUnit(ImportResults importResults, long line, EffectiveInterventionImportDto from, EffectiveInterventionDto to, boolean error) {
        String workRateUnitName = from.getWorkRateUnitName();
        if (StringUtils.isNotBlank(workRateUnitName)) {
            try {
                MaterielWorkRateUnit type = MaterielWorkRateUnit.valueOf(workRateUnitName);
                to.setWorkRateUnit(type);
            } catch (IllegalArgumentException e) {
                error = true;
                importResults.addErrorLine(line, String.format("INTERVENTION IGNORÉE!, l'unité de temps de travail %s n'est pas reconnue.", workRateUnitName));
            }
        }
        return error;
    }

}