package fr.inra.agrosyst.services.pz0import.domain;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: ToolsCouplingImporter.java 4901 2015-04-20 10:50:56Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.4.9/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/domain/ToolsCouplingImporter.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.collect.Sets;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.Equipment;
import fr.inra.agrosyst.api.entities.EquipmentImpl;
import fr.inra.agrosyst.api.entities.MaterielWorkRateUnit;
import fr.inra.agrosyst.api.entities.ToolsCoupling;
import fr.inra.agrosyst.api.entities.ToolsCouplingImpl;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.domains.DomainAndDependencies;
import fr.inra.agrosyst.services.pz0import.AbstractCSVImporter;
import fr.inra.agrosyst.services.pz0import.domain.dto.ToolCouplingImportDto;
import fr.inra.agrosyst.services.pz0import.domain.model.ToolsCouplingImportModel;
import org.apache.commons.lang3.StringUtils;
import org.nuiton.csv.Import;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.InputStream;
import java.util.Map;
import java.util.Set;

/**
 * Created by davidcosse on 09/12/14.
 */
public class ToolsCouplingImporter extends AbstractCSVImporter {

    @Override
    public ImportResults importFromStream(InputStream is, Map<String, EntityAndDependencies> entitiesByCsvId) {
        ImportResults importResults = new ImportResults(ToolsCoupling.class);

        ToolsCouplingImportModel model = new ToolsCouplingImportModel();
        Import<ToolCouplingImportDto> importer = Import.newImport(model, is);

        // match the csv line number.
        Long line = FIRST_LINE_NUMBER;
        for (ToolCouplingImportDto dto : importer) {
            boolean error = false;
            dto.setLine(line);

            ToolsCoupling toolsCoupling = new ToolsCouplingImpl();

            Binder<ToolCouplingImportDto, ToolsCoupling> binder = BinderFactory.newBinder(ToolCouplingImportDto.class, ToolsCoupling.class);
            binder.copyExcluding(dto, toolsCoupling,
                    ToolsCoupling.PROPERTY_TOPIA_ID,
                    ToolsCoupling.PROPERTY_TRACTOR,
                    ToolsCoupling.PROPERTY_DOMAIN);

            toolsCoupling.setCode(csvCodeToAgrosystCode(ToolsCoupling.class, dto.getCode()));

            error = validToolCouplingToDomain(importResults, entitiesByCsvId, dto, toolsCoupling, line, error);

            error = validTractor(importResults, dto, toolsCoupling, line, error);

            error = addToolCouplingWorkRateUnit(importResults, dto, toolsCoupling, line, error);

            pz0IdToObject.put(ToolsCoupling.class, dto.getId(), toolsCoupling);

            DomainAndDependencies domainAndDependencies = (DomainAndDependencies) entitiesByCsvId.get(dto.getDomainId());
            stockToolsCouplingCodeForDomainAndCampain(dto, domainAndDependencies);

            if (!error) {
                importResults.addInfoLine(line, "COMBINAISON D'OUTILS VALIDÉE, csvid: " + dto.getId() + ", elle sera persistée avec le domaine: " + dto.getDomainId());
                importResults.increaseAddedRecords();
            } else {
                importResults.increaseIgnoredRecords();
                importResults.addErrorLine(line, "COMBINAISON D'OUTILS IGNORÉE, csvid: " + dto.getId());
            }

            line++;
        }
        return  importResults;
    }

    protected void stockToolsCouplingCodeForDomainAndCampain(ToolCouplingImportDto toolsCoupling, DomainAndDependencies domainAndDependencies) {
        if (domainAndDependencies != null && domainAndDependencies.getDomain() != null) {
            Domain domain = domainAndDependencies.getDomain();
            Set<String> toolsCouplingCodesForCampaign = domainCodeCampaignTCCode.get(domainAndDependencies.getCsvCode(), domain.getCampaign());
            if (toolsCouplingCodesForCampaign == null) {
                toolsCouplingCodesForCampaign = Sets.newHashSet();
            }
            toolsCouplingCodesForCampaign.add(toolsCoupling.getCode());// we save the csv code
            domainCodeCampaignTCCode.put(domainAndDependencies.getCsvCode(), domain.getCampaign(), toolsCouplingCodesForCampaign);
        }
    }

    protected boolean validToolCouplingToDomain(ImportResults importResults, Map<String, EntityAndDependencies> entitiesByCsvId, ToolCouplingImportDto dto, ToolsCoupling toolsCoupling, long line, boolean error) {

        if (StringUtils.isNotBlank(dto.getDomainId())) {
            DomainAndDependencies domainAndDependencies = (DomainAndDependencies) entitiesByCsvId.get(dto.getDomainId());
            if (domainAndDependencies != null) {
                domainAndDependencies.addToolsCoupling(dto.getId(), toolsCoupling);
            } else {
                importResults.addErrorLine(line, String.format("COMBINAISON D'OUTILS IGNORÉE, le domaine avec comme identifiant:%s, n'a pas été retrouvé pour la combinaison d'outil: %s",dto.getDomainId(), dto.getId()));
                error = true;
            }
        } else {
            importResults.addErrorLine(line, "COMBINAISON D'OUTILS IGNORÉE, la colonne 'domain' n'est pas renseignée, csvid: " + dto.getId());
            error = true;
        }
        return error;
    }

    protected boolean validTractor(ImportResults importResults, ToolCouplingImportDto dto, ToolsCoupling toolsCoupling, long line, boolean error) {
        if (dto.isManualIntervention()) {
            if (StringUtils.isNotBlank(dto.getTractorId())) {
                error = true;
                importResults.addErrorLine(line, String.format("INTEVENTION MANUELLE IGNORÉE, un tracteur est renseigné csvid: %s", dto.getId()));
            }
        } else if (StringUtils.isNotBlank(dto.getTractorId())) {
            DomainAndDependencies domainAndDependencies = dto.getDomainAndDependencies();
            Equipment tractor = (Equipment) pz0IdToObject.get(Equipment.class, dto.getTractorId());
            if (tractor != null) {
                // to be able to save tools coupling it's necessary to have diferent objet instance that the targeted one.
                Equipment equipment = new EquipmentImpl();
                Binder<Equipment, Equipment> tractorBinder = BinderFactory.newBinder(Equipment.class, Equipment.class);
                tractorBinder.copy(tractor, equipment);
                toolsCoupling.setTractor(equipment);

                if (domainAndDependencies != null) {
                    if (!domainAndDependencies.getEquipments().contains(tractor)) {
                        error = true;
                        importResults.addErrorLine(line, String.format("COMBINAISON D'OUTILS IGNORÉE, le tracteur avec comme identifiant %s n'a pas été retrouvé sur sur le domaine %s", dto.getTractorId(), domainAndDependencies.getCsvId()));
                    }
                }
            } else {
                error = true;
                importResults.addErrorLine(line, String.format("COMBINAISON D'OUTILS IGNORÉE, le tracteur %s n'a pas été retrouvé csvid: %s", dto.getTractor() , dto.getId()));
            }



        }
        return error;
    }

    protected boolean addToolCouplingWorkRateUnit(ImportResults importResults, ToolCouplingImportDto dto, ToolsCoupling toolsCoupling, long line, boolean error) {
        MaterielWorkRateUnit workRateUnit;
        String value = dto.getWorkrateUnitImporterValue();
        if (StringUtils.isNotBlank(value)) {
            try {
                workRateUnit = MaterielWorkRateUnit.valueOf(value);
                toolsCoupling.setWorkRateUnit(workRateUnit);
            } catch (IllegalArgumentException ex) {
                error = true;
                importResults.addErrorLine(line, String.format("COMBINAISON D'OUTILS IGNORÉE! Unité de rendement non supportée, %s\"!", value));
            }
        }
        return error;
    }
}
