package fr.inra.agrosyst.services.pz0import.domain;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: CroppingPanEntryImporter.java 4914 2015-04-30 08:10:37Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.4.9/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/domain/CroppingPanEntryImporter.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.collect.Sets;
import fr.inra.agrosyst.api.entities.CroppingEntryType;
import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.services.domain.CroppingPlanEntryDto;
import fr.inra.agrosyst.api.services.domain.DomainService;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.domains.DomainAndDependencies;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.services.pz0import.AbstractCSVImporter;
import fr.inra.agrosyst.services.pz0import.domain.dto.CroppingPlanEntryImportDto;
import fr.inra.agrosyst.services.pz0import.domain.model.CroppingPlanEntryImportModel;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.InputStream;
import java.util.Map;
import java.util.Set;

/**
 * Created by davidcosse on 05/12/14.
 */
public class CroppingPanEntryImporter extends AbstractCSVImporter {

    private static final Log log = LogFactory.getLog(CroppingPanEntryImporter.class);
    protected DomainService domainService;

    @Override
    public ImportResults importFromStream(InputStream is,  Map<String, EntityAndDependencies> entitiesByCsvIds) {

        ImportResults importResults = new ImportResults(CroppingPlanEntry.class);

        CroppingPlanEntryImportModel model = new CroppingPlanEntryImportModel();
        Import<CroppingPlanEntryImportDto> importer = Import.newImport(model, is);

        // match the csv line number.
        Long line = FIRST_LINE_NUMBER;
        for (CroppingPlanEntryImportDto dto : importer) {
            boolean error = false;

            // we use CroppingPlanEntryDto because the service used to save the domain required a collection of this type object.
            CroppingPlanEntryDto croppingPlanEntryDto = new CroppingPlanEntryDto();

            Binder<CroppingPlanEntryImportDto, CroppingPlanEntryDto> croppingPlanEntryDtoBinder = BinderFactory.newBinder(CroppingPlanEntryImportDto.class, CroppingPlanEntryDto.class);
            croppingPlanEntryDtoBinder.copy(dto, croppingPlanEntryDto);

            error = validCroppingPlanEntryCroppingEntryType(dto, croppingPlanEntryDto, importResults, line, error);

            error = addCroppingPlanToDomain(importResults, line, dto, error, croppingPlanEntryDto);

            croppingPlanEntryDto.setCode(csvCodeToAgrosystCode(CroppingPlanEntryDto.class, dto.getCode()));

            pz0IdToObject.put(croppingPlanEntryDto.getClass(), dto.getId(), croppingPlanEntryDto);
            if (!error) {
                importResults.addInfoLine(line, "CULTURE VALIDÉE, csvid: " + dto.getId() + ", elle sera persistée avec le domaine: " + dto.getDomainId());
                importResults.increaseAddedRecords();
            } else {
                importResults.increaseIgnoredRecords();
                importResults.addErrorLine(line, "CULTURE IGNORÉE, csvid: " + dto.getId());
            }

            line++;
        }
        return  importResults;
    }

    protected boolean addCroppingPlanToDomain(ImportResults importResults, Long line, CroppingPlanEntryImportDto dto, boolean error, CroppingPlanEntryDto croppingPlanEntryDto) {
        String domainId = dto.getDomainId();
        if (StringUtils.isNotBlank(domainId)) {
            DomainAndDependencies domainAndDependencies = (DomainAndDependencies) pz0IdToObject.get(Domain.class, domainId);
            if (domainAndDependencies != null) {
                pz0IdToObject.put(CroppingPlanEntryDto.class, DomainAndDependencies.class, dto.getId(), domainAndDependencies);
                domainAndDependencies.addCroppingPlanEntryDto(dto.getCode(), dto.getId(), croppingPlanEntryDto);
                stockCroppingPlanCodeForDomainAndCampain(dto, domainAndDependencies);
            } else {
                error = true;
                importResults.addErrorLine(line, "CULTURE IGNORÉE, Le domain dont d'identifiant est:'" + domainId + "' n'a pas été retrouvé.");
            }
        } else {
            error = true;
            importResults.addErrorLine(line, "CULTURE IGNORÉE, La colonne 'domain' n'est pas renseignée.");
        }
        return error;
    }

    protected void stockCroppingPlanCodeForDomainAndCampain(CroppingPlanEntryImportDto dto, DomainAndDependencies domainAndDependencies) {
        if (domainAndDependencies != null && domainAndDependencies.getDomain() != null) {
            Domain domain = domainAndDependencies.getDomain();
            Set<String> cropsCouplingCodesForCampaign = domainCodeCampaignCropCode.get(domainAndDependencies.getCsvCode(), domain.getCampaign());
            if (cropsCouplingCodesForCampaign == null) {
                cropsCouplingCodesForCampaign = Sets.newHashSet();
            }
            cropsCouplingCodesForCampaign.add(dto.getCode());// we save the csv code
            domainCodeCampaignCropCode.put(domainAndDependencies.getCsvCode(), domain.getCampaign(), cropsCouplingCodesForCampaign);
        }
    }

    protected boolean validCroppingPlanEntryCroppingEntryType(CroppingPlanEntryImportDto from, CroppingPlanEntryDto to, ImportResults importResults, long line, boolean error) {
        CroppingEntryType type;
        String value = from.getTypeImporterValue();
        if (StringUtils.isNotBlank(value)) {
            try {
                type = CroppingEntryType.valueOf(value);
                to.setType(type);
            } catch (IllegalArgumentException ex) {
                error = true;
                importResults.addErrorLine(line, String.format("CULTURE IGNORÉE ! Type de culture non supportée, %s !", value));
            }
        } else {
            error = true;
            importResults.addErrorLine(line, "CULTURE IGNORÉE, La colonne 'type' n'est pas renseignée.");
        }
        return error;
    }

    @Override
    public void init(ServiceContext serviceContext) {
        super.init(serviceContext);
        domainService = getServiceFactory().newService(DomainService.class);
    }

}
