package fr.inra.agrosyst.services.pz0import;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: CliStarter.java 4925 2015-05-05 21:11:04Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.4.9/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/CliStarter.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import fr.inra.agrosyst.api.entities.AgrosystTopiaPersistenceContext;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.Equipment;
import fr.inra.agrosyst.api.entities.Ground;
import fr.inra.agrosyst.api.entities.GrowingPlan;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.Plot;
import fr.inra.agrosyst.api.entities.ToolsCoupling;
import fr.inra.agrosyst.api.entities.action.AbstractAction;
import fr.inra.agrosyst.api.entities.action.HarvestingYeald;
import fr.inra.agrosyst.api.entities.action.SeedingActionSpecies;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCycleConnection;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCycleNode;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCyclePhase;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCycleSpecies;
import fr.inra.agrosyst.api.entities.effective.EffectiveIntervention;
import fr.inra.agrosyst.api.entities.effective.EffectivePerennialCropCycle;
import fr.inra.agrosyst.api.entities.effective.EffectiveSeasonalCropCycle;
import fr.inra.agrosyst.api.entities.effective.EffectiveSpeciesStade;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCycle;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCycleConnection;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCycleNode;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCyclePhase;
import fr.inra.agrosyst.api.entities.practiced.PracticedIntervention;
import fr.inra.agrosyst.api.entities.practiced.PracticedPerennialCropCycle;
import fr.inra.agrosyst.api.entities.practiced.PracticedPlot;
import fr.inra.agrosyst.api.entities.practiced.PracticedSeasonalCropCycle;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystem;
import fr.inra.agrosyst.api.entities.referential.RefFertiMinUNIFA;
import fr.inra.agrosyst.api.entities.referential.RefMaterielAutomoteur;
import fr.inra.agrosyst.api.entities.referential.RefMaterielIrrigation;
import fr.inra.agrosyst.api.entities.security.AgrosystUser;
import fr.inra.agrosyst.api.entities.security.AgrosystUserTopiaDao;
import fr.inra.agrosyst.api.entities.security.RoleType;
import fr.inra.agrosyst.api.entities.security.UserRole;
import fr.inra.agrosyst.api.entities.security.UserRoleTopiaDao;
import fr.inra.agrosyst.api.services.ServiceFactory;
import fr.inra.agrosyst.api.services.domain.CroppingPlanEntryDto;
import fr.inra.agrosyst.api.services.domain.DomainService;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCycleService;
import fr.inra.agrosyst.api.services.growingplan.GrowingPlanService;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemService;
import fr.inra.agrosyst.api.services.input.Inputs;
import fr.inra.agrosyst.api.services.plot.PlotService;
import fr.inra.agrosyst.api.services.practiced.PracticedPlotService;
import fr.inra.agrosyst.api.services.practiced.PracticedSystemService;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.domains.DomainAndDependencies;
import fr.inra.agrosyst.api.services.pz0.effective.EffectiveCropCycleAndDependencies;
import fr.inra.agrosyst.api.services.pz0.effective.Pz0EffectiveCropCycleNode;
import fr.inra.agrosyst.api.services.pz0.effective.Pz0EffectiveCropCyclePhase;
import fr.inra.agrosyst.api.services.pz0.effective.Pz0EffectiveIntervention;
import fr.inra.agrosyst.api.services.pz0.effective.Pz0EffectivePerennialCropCycle;
import fr.inra.agrosyst.api.services.pz0.effective.Pz0EffectiveSeasonalCropCycle;
import fr.inra.agrosyst.api.services.pz0.plot.PlotAndDependencies;
import fr.inra.agrosyst.api.services.pz0.practicedSystem.PracticedSystemAndDependencies;
import fr.inra.agrosyst.api.services.pz0.practicedSystem.Pz0PracticedCropCycleConnection;
import fr.inra.agrosyst.api.services.pz0.practicedSystem.Pz0PracticedCropCyclePhase;
import fr.inra.agrosyst.api.services.pz0.practicedSystem.Pz0PracticedIntervention;
import fr.inra.agrosyst.api.services.pz0.practicedSystem.Pz0PracticedPerennialCropCycle;
import fr.inra.agrosyst.api.services.pz0.practicedSystem.Pz0PracticedSeasonalCropCycle;
import fr.inra.agrosyst.api.services.referential.ReferentialService;
import fr.inra.agrosyst.api.services.security.AuthenticationService;
import fr.inra.agrosyst.api.services.security.AuthorizationService;
import fr.inra.agrosyst.api.services.security.UserRoleDto;
import fr.inra.agrosyst.api.services.users.UserDto;
import fr.inra.agrosyst.api.services.users.UserService;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.services.pz0import.action.ActionImporter;
import fr.inra.agrosyst.services.pz0import.action.harvestingYeald.HarvestingYealdImporter;
import fr.inra.agrosyst.services.pz0import.action.seedingActionSpecies.SeedingActionSpeciesImporter;
import fr.inra.agrosyst.services.pz0import.domain.CroppingPanEntryImporter;
import fr.inra.agrosyst.services.pz0import.domain.CroppingPanEntrySpeciesImporter;
import fr.inra.agrosyst.services.pz0import.domain.DomainImporter;
import fr.inra.agrosyst.services.pz0import.domain.EquipmentForToolsCouplingImporter;
import fr.inra.agrosyst.services.pz0import.domain.EquipmentImporter;
import fr.inra.agrosyst.services.pz0import.domain.GroundImporter;
import fr.inra.agrosyst.services.pz0import.domain.MainActionForToolsCouplingImporter;
import fr.inra.agrosyst.services.pz0import.domain.ToolsCouplingImporter;
import fr.inra.agrosyst.services.pz0import.domain.dto.CroppingPlanSpeciesImportDto;
import fr.inra.agrosyst.services.pz0import.effective.EffectiveCropCycleImporter;
import fr.inra.agrosyst.services.pz0import.effective.effectiveCropCycleConnection.EffectiveCropCycleConnectionImporter;
import fr.inra.agrosyst.services.pz0import.effective.effectiveCropCycleNode.EffectiveCropCycleNodeImporter;
import fr.inra.agrosyst.services.pz0import.effective.effectiveCropCycleSpecies.EffectiveCropCycleSpeciesImporter;
import fr.inra.agrosyst.services.pz0import.effective.effectiveIntervention.EffectiveInterventionImporter;
import fr.inra.agrosyst.services.pz0import.effective.effectivePerennialCropCycle.EffectivePerennialCropCycleImporter;
import fr.inra.agrosyst.services.pz0import.effective.effectivePhase.EffectivePhaseImporter;
import fr.inra.agrosyst.services.pz0import.effective.effectiveSeasonalCropCycle.EffectiveSeasonalCropCycleImporter;
import fr.inra.agrosyst.services.pz0import.effective.effectiveSpeciesStade.EffectiveSpeciesStadeImporter;
import fr.inra.agrosyst.services.pz0import.effective.effectiveToolsCoupling.EffectiveToolsCouplingImporter;
import fr.inra.agrosyst.services.pz0import.growingPlan.GrowingPlanImporter;
import fr.inra.agrosyst.services.pz0import.growingSystem.GrowingSystemImporter;
import fr.inra.agrosyst.services.pz0import.growingSystem.GrowingSystemNetworkImportDto;
import fr.inra.agrosyst.services.pz0import.growingSystem.GrowingSystemNetworkImporter;
import fr.inra.agrosyst.services.pz0import.input.InputImporter;
import fr.inra.agrosyst.services.pz0import.input.RefFertiMinUnifaImporter;
import fr.inra.agrosyst.services.pz0import.plot.PlotImporter;
import fr.inra.agrosyst.services.pz0import.practicedSystem.PracticedPerennialCropCyclePhase.PracticedPerennialCropCyclePhaseImporter;
import fr.inra.agrosyst.services.pz0import.practicedSystem.PracticedSystemImporter;
import fr.inra.agrosyst.services.pz0import.practicedSystem.practicedCropCycle.PracticedCropCycleImporter;
import fr.inra.agrosyst.services.pz0import.practicedSystem.practicedCropCycleConnection.PracticedCropCycleConnectionImporter;
import fr.inra.agrosyst.services.pz0import.practicedSystem.practicedCropCycleNode.PracticedCropCycleNodeImporter;
import fr.inra.agrosyst.services.pz0import.practicedSystem.practicedIntervention.PracticedInterventionImporter;
import fr.inra.agrosyst.services.pz0import.practicedSystem.practicedInterventionToolsCoupling.PracticedToolsCouplingImportDto;
import fr.inra.agrosyst.services.pz0import.practicedSystem.practicedInterventionToolsCoupling.PracticedToolsCouplingImporter;
import fr.inra.agrosyst.services.pz0import.practicedSystem.practicedPerennialCropCycle.PracticedPerennialCropCycleImporter;
import fr.inra.agrosyst.services.pz0import.practicedSystem.practicedPlot.PracticedPlotImporter;
import fr.inra.agrosyst.services.pz0import.practicedSystem.practicedSeasonalCropCycle.PracticedSeasonalCropCycleImporter;
import fr.inra.agrosyst.services.pz0import.security.UserRoleImporter;
import fr.inra.agrosyst.services.pz0import.zone.ZoneImporter;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 *
 * @author antoine Schellenberger
 */
public class CliStarter {

    protected static final Log log = LogFactory.getLog(CliStarter.class);

    protected final static String LAUNCH_INFO = "Un seul argument est authorisé, il sagit du chemin vers le fichier de configuration, si aucun n'est spécifié le fichier 'agrosyst-import.properties' sera utilisé s'il existe sinon les paramêtres par défaut s'appliqueront.";

    protected String configFilePath;
    protected ServiceFactory serviceFactory;
    protected AgrosystCliServiceContext serviceContext;
    protected DomainService domainService;
    protected GrowingSystemService growingSystemService;
    protected GrowingPlanService growingPlanService;
    protected ReferentialService referentialService;
    protected PracticedSystemService practicedSystemService;
    protected PracticedPlotService practicedPlotService;
    protected PlotService plotService;
    protected EffectiveCropCycleService effectiveCropCycleService;
    protected AuthorizationService authorizationService;

    protected String domainFilePath;
    protected String croppingPlanFilePath;
    protected String croppingPlanSpeciesFilePath;

    protected String equipmentFilePath;
    protected String toolCouplingFilePath;
    protected String equipmentToToolsCouplingFilePath;
    protected String mainActionToToolsCouplingFilePath;
    protected String groundFilePath;

    protected String growingPlanFilePath;
    protected String growingSystemFilePath;
    protected String growingSystemNetworksFilePath;

    protected String practicedSystemFilePath;
    protected String practicedPlotFilePath;
    protected String practicedCropCycleFilePath;

    protected String practicedPerennialCropCycleFilePath;
    protected String practicedCropCyclePhaseFilePath;
    protected String practicedCropCycleSpeciesFilePath;

    protected String practicedSeasonalCropCycleFilePath;
    protected String practicedCropCycleNodeFilePath;
    protected String practicedCropCycleConnectionFilePath;

    protected String practicedInterventionFilePath;
    protected String practiceInterventionToolsCouplingFilePath;

    protected String effectivePerennialCropCycleFilePath;
    protected String effectiveCropCycleSpeciesFilePath;
    protected String effectivePhaseFilePath;
    protected String effectiveSeasonalCropCycleFilePath;
    protected String effectiveCropCycleNodeFilePath;
    protected String effectiveCropCycleConnectionFilePath;

    protected String effectiveInterventionFilePath;
    protected String effectiveSpeciesStadeFilePath;
    protected String effectiveToolsCouplingFilePath;
    protected String effectiveInvolvedRuleFilePath;

    protected String abstractActionFilePath;
    protected String seedingActionSpeciesFilePath;
    protected String harverstingYealdFilePath;
    protected String abstractInputFilePath;
    protected String fertiMinUnifaFilePath;
    protected String plotFilePath;
    protected String zoneFilePAth;

    protected String usersRolesFilePath;

    public CliStarter(String configFilePath) {
        this.configFilePath = configFilePath;
    }

    /**
     * les résultats des tests peuvent être retrouvés dans le fichier /tmp/pz0import.out
     * Si les informations sont trop verbeuses vous pouvez modifier le ficher log4j.properties du package agrosyst-cli
     * @param args Optional Parameter: config file path (if null 'agrosyst-import.properties' will be used)
     */
    public static void main(String[] args) {

        String dbConfigFilePath = null;

        if(args.length == 1) {
            dbConfigFilePath = args[0];
        } else if (args.length > 1) {
            System.out.println(LAUNCH_INFO);
        }

        CliStarter importer = new CliStarter(dbConfigFilePath);
        importer.init();
        importer.processImport();
    }

    public AgrosystTopiaPersistenceContext getPersistenceContext() {
        return serviceContext.getPersistenceContext();
    }

    protected Map<Class, ImportResults> processImport() {
        Map<Class, ImportResults> allResults = Maps.newHashMap();

        try {
            Preconditions.checkArgument(StringUtils.isNotBlank(domainFilePath) && (new File(domainFilePath)).exists(), "Le fichier d'import des domains est requis");
            Preconditions.checkArgument(StringUtils.isNotBlank(growingPlanFilePath) && (new File(growingPlanFilePath)).exists(), "Le fichier d'import des dispositifs est requis");
            Preconditions.checkArgument(StringUtils.isNotBlank(growingSystemFilePath) && (new File(growingSystemFilePath)).exists(), "Le fichier d'import des systèmes de culture est requis");

            // process file parsing, create entity and save result
            processPzOCSVFilesParsing(allResults);

            // if no errors, entities will be saved
            processEntitiesPersistence(allResults);

        } catch (FileNotFoundException ex) {
            getPersistenceContext().rollback();
            Logger.getLogger(CliStarter.class.getName()).log(Level.SEVERE, null, ex);
        } catch (Exception e) {
            getPersistenceContext().rollback();
            System.out.println("rollback\n");
            e.printStackTrace();
        }

        return allResults;
    }

    protected void processPzOCSVFilesParsing(Map<Class, ImportResults> allResults) throws FileNotFoundException {
        importDomainUniverse(domainFilePath,
                croppingPlanFilePath,
                croppingPlanSpeciesFilePath,
                equipmentFilePath,
                toolCouplingFilePath,
                equipmentToToolsCouplingFilePath,
                mainActionToToolsCouplingFilePath,
                groundFilePath, allResults);

        importGrowingPlans(growingPlanFilePath, allResults);
        importGrowingSystems(growingSystemFilePath, growingSystemNetworksFilePath, allResults);
        importPlots(plotFilePath, zoneFilePAth, allResults);

        importPracticedSystemUniverse(
                practicedSystemFilePath,
                practicedPlotFilePath,
                practicedCropCycleFilePath,
                practicedSeasonalCropCycleFilePath, practicedCropCycleNodeFilePath, practicedCropCycleConnectionFilePath,
                practicedPerennialCropCycleFilePath, practicedCropCyclePhaseFilePath, practicedCropCycleSpeciesFilePath,
                practiceInterventionToolsCouplingFilePath,
                practicedInterventionFilePath,
                allResults);

        importEffectiveCropCycleUniverse(
                effectivePerennialCropCycleFilePath, effectiveCropCycleSpeciesFilePath, effectivePhaseFilePath,
                effectiveSeasonalCropCycleFilePath, effectiveCropCycleNodeFilePath, effectiveCropCycleConnectionFilePath,
                effectiveInterventionFilePath, effectiveSpeciesStadeFilePath, effectiveToolsCouplingFilePath, effectiveInvolvedRuleFilePath,
                allResults
        );

        importAbstractAction(abstractActionFilePath, allResults);
        importHarverstingYeald(harverstingYealdFilePath, allResults);
        importSeedingActionSpecies(seedingActionSpeciesFilePath, allResults);
        importAbstractInput(abstractInputFilePath, fertiMinUnifaFilePath, allResults);
        importUsersRolesImporter(usersRolesFilePath, allResults);

        log.info("POST VALIDATION DES INTERVENTIONS, valide la présence d'actions sur les interventions.");
        validPracticedInterventionMainActionExists(allResults);
        validEffectiveInterventionMainActionExists();

    }

    protected void validPracticedInterventionMainActionExists(Map<Class, ImportResults> allResults) {
        ImportResults importResults = allResults.get(PracticedSystem.class);
        if (importResults != null) {
            Map<String, EntityAndDependencies> practicedSystemsByCsvId = importResults.getEntityAndDepsByCsvIds();
            for (Map.Entry<String, EntityAndDependencies> practicedSystemByCsvId : practicedSystemsByCsvId.entrySet()) {
                PracticedSystemAndDependencies practicedSystemAndDependencies = (PracticedSystemAndDependencies) practicedSystemByCsvId.getValue();

                List<Pz0PracticedSeasonalCropCycle> pz0SeasonalCropCycles = practicedSystemAndDependencies.getPz0PracticedSeasonalCropCycles();
                if (pz0SeasonalCropCycles != null) {
                    for (Pz0PracticedSeasonalCropCycle seasonalCropCycleDto : pz0SeasonalCropCycles) {
                        Collection<Pz0PracticedCropCycleConnection> pz0Connections = seasonalCropCycleDto.getPz0Connections();
                        if (pz0Connections != null) {
                            for (Pz0PracticedCropCycleConnection pz0Connection : pz0Connections) {
                                Collection<Pz0PracticedIntervention> pz0Interventions = pz0Connection.getPz0InterventionByCsvId().values();
                                validPracticedInterventionMainAction(importResults, practicedSystemByCsvId, pz0Interventions);
                            }
                        }
                    }
                }

                List<Pz0PracticedPerennialCropCycle> pz0PracticedPerennialCropCycles = practicedSystemAndDependencies.getPz0PracticedPerennialCropCycles();
                if (pz0PracticedPerennialCropCycles != null) {
                    for (Pz0PracticedPerennialCropCycle pz0PracticedPerennialCropCycle : pz0PracticedPerennialCropCycles) {
                        Collection<Pz0PracticedCropCyclePhase> pz0Phases = pz0PracticedPerennialCropCycle.getPz0PhasesByCsvId().values();
                        for (Pz0PracticedCropCyclePhase pz0Phase : pz0Phases) {
                            Collection<Pz0PracticedIntervention> pz0Interventions = pz0Phase.getPz0InterventionByCsvId().values();
                            validPracticedInterventionMainAction(importResults, practicedSystemByCsvId, pz0Interventions);
                        }
                    }
                }
            }
            displayImportResult(importResults);
        }

    }

    protected void validEffectiveInterventionMainActionExists() {
        ImportResults importResults = new ImportResults(Pz0EffectiveIntervention.class);
        EffectiveCropCycleImporter effectiveCycleImporter = new EffectiveCropCycleImporter();
        Collection<EffectiveCropCycleAndDependencies> effectiveCropCycleAndDependencies = effectiveCycleImporter.getEffectiveCropCycleAndDependencies();
        for (EffectiveCropCycleAndDependencies effectiveCropCycleAndDependency : effectiveCropCycleAndDependencies) {
            String zoneId = effectiveCropCycleAndDependency.getZoneId();
            Map<String, Pz0EffectivePerennialCropCycle> perennialCropCycles = effectiveCropCycleAndDependency.getEffectivePz0PerennialCropCyclesByCsvId();
            for (Pz0EffectivePerennialCropCycle pz0EffectivePerennialCropCycle : perennialCropCycles.values()) {
                Collection<Pz0EffectiveCropCyclePhase> phases = pz0EffectivePerennialCropCycle.getPhasesByCsvId().values();
                for (Pz0EffectiveCropCyclePhase phase : phases) {
                    Collection<Pz0EffectiveIntervention> pz0Interventions = phase.getPz0InterventionByCsvId().values();
                    validEffectiveInterventions(importResults, zoneId, pz0Interventions);
                }
            }

            Map<String, Pz0EffectiveSeasonalCropCycle> seasonalCropCycles = effectiveCropCycleAndDependency.getEffectivePz0SeasonalCropCyclesByCsvId();
            for (Pz0EffectiveSeasonalCropCycle seasonalCropCycle : seasonalCropCycles.values()) {
                Collection<Pz0EffectiveCropCycleNode> nodes =  seasonalCropCycle.getPz0NodesByRank().values();
                for (Pz0EffectiveCropCycleNode node : nodes) {
                    Collection<Pz0EffectiveIntervention> pz0Interventions = node.getPz0InterventionByCsvId().values();
                    validEffectiveInterventions(importResults, zoneId, pz0Interventions);
                }
            }
        }
        displayImportResult(importResults);
    }

    protected void validEffectiveInterventions(ImportResults importResults, String zoneId, Collection<Pz0EffectiveIntervention> pz0Interventions) {
        for (Pz0EffectiveIntervention pz0Intervention : pz0Interventions) {
            Collection<AbstractAction> actions = pz0Intervention.getEffectiveInterventionDto().getActions();
            if (CollectionUtils.isNotEmpty(actions)) {
                importResults.increaseAddedRecords();
                importResults.addInfoLine(0L, String.format("INTERVENTION VALIDÉE: %s", pz0Intervention.getCsvId()));
            } else {
                importResults.increaseIgnoredRecords();
                importResults.addErrorLine(0L, String.format("INTERVENTION IGNORÉE, aucune action trouvée sur l'intervention: %s de la zone %s", pz0Intervention.getCsvId(), zoneId));
            }
        }
    }

    private void validPracticedInterventionMainAction(ImportResults importResults, Map.Entry<String, EntityAndDependencies> stringEntityAndDependenciesEntry, Collection<Pz0PracticedIntervention> pz0Interventions) {
        if (pz0Interventions != null) {
            for (Pz0PracticedIntervention pz0Intervention : pz0Interventions) {
                Collection<AbstractAction> actions = pz0Intervention.getPracticedInterventionDto().getActions();
                if (CollectionUtils.isNotEmpty(actions)) {
                    importResults.increaseAddedRecords();
                    importResults.addInfoLine(0L, String.format("INTERVENTION VALIDÉE: %s", pz0Intervention.getCsvId()));
                } else {
                    importResults.increaseIgnoredRecords();
                    importResults.addErrorLine(0L, String.format("INTERVENTION IGNORÉE, aucune action trouvée sur l'intervention: '%s' du système synthétisé %s", pz0Intervention.getCsvId(), stringEntityAndDependenciesEntry.getKey()));
                }
            }
        }
    }

    protected void importDomainUniverse(
            String domainFilePath,
            String croppingPlanFilePath,
            String croppingPlanSpeciesFilePath,
            String equipmentFilePath,
            String toolCouplingFilePath,
            String equipmentToToolsCouplingFilePath,
            String mainActionToToolsCouplingFilePath,
            String groundFilePath,
            Map<Class, ImportResults> allResults) throws FileNotFoundException {
        importDomains(domainFilePath, allResults);
        importCroppingPlans(croppingPlanFilePath, allResults);
        importCroppingPlanSpecies(croppingPlanSpeciesFilePath, allResults);
        importEquipments(equipmentFilePath, allResults);
        importGrounds(groundFilePath, allResults);
        importToolsCouplings(toolCouplingFilePath, equipmentToToolsCouplingFilePath, mainActionToToolsCouplingFilePath, allResults);
    }

    protected void validPracticedCropCycle(Map<Class, ImportResults> allResults) {

        ImportResults practicedSystemsResult = allResults.get(PracticedSystem.class);
        if (practicedSystemsResult != null) {
            Map<String, EntityAndDependencies> entitiesAndDependencies = practicedSystemsResult.getEntityAndDepsByCsvIds();

            if (entitiesAndDependencies != null) {
                for (Map.Entry<String, EntityAndDependencies> entityAndDependenciesByCsvId : entitiesAndDependencies.entrySet()) {
                    PracticedSystemAndDependencies practicedSystemAndDependencies = (PracticedSystemAndDependencies) entityAndDependenciesByCsvId.getValue();
                    List<Pz0PracticedSeasonalCropCycle> pz0SeasonalCropCycles = practicedSystemAndDependencies.getPz0PracticedSeasonalCropCycles();
                    if (pz0SeasonalCropCycles != null) {
                        for (Pz0PracticedSeasonalCropCycle pz0SeasonalCropCycle : pz0SeasonalCropCycles) {
                            if (CollectionUtils.isEmpty(pz0SeasonalCropCycle.getPracticedSeasonalCropCycleDto().getCropCycleNodeDtos())) {
                                practicedSystemsResult.addErrorLine(practicedSystemAndDependencies.getImportedNumber(), String.format("PRECONDITION NON VALIDÉE POUR LE CYCLE SAISONNIER AVEC COMME IDENTIFIANT '%s', aucun noeud n'est définit pour ce cycle.", entityAndDependenciesByCsvId.getKey()));
                                practicedSystemsResult.increaseIgnoredRecords();
                            }
                            if (CollectionUtils.isEmpty(pz0SeasonalCropCycle.getPz0Connections())) {
                                practicedSystemsResult.addErrorLine(practicedSystemAndDependencies.getImportedNumber(), String.format("PRECONDITION NON VALIDÉE POUR LE CYCLE SAISONNIER AVEC COMME IDENTIFIANT '%s', aucune connection n'est définie pour ce cycle.", entityAndDependenciesByCsvId.getKey()));
                                practicedSystemsResult.increaseIgnoredRecords();
                            }
                        }
                    }
                }
            }

            displayImportResult(practicedSystemsResult);
        } else {
            log.info("Aucun système synthétisé importé !");
        }
    }

    protected void processEntitiesPersistence(Map<Class, ImportResults> allResults) {
        boolean importErrors = isImportErrors(allResults);
        if (!importErrors) {
            if (log.isInfoEnabled()) {
                log.info("PERSISTANCE en cours...");
            }
            try {
                if (log.isInfoEnabled()) {
                    log.info("Sauvegarde des domaines en cours...");
                }
                domainService.importPZ0Domains(allResults);

                if (log.isInfoEnabled()) {
                    log.info("Sauvegarde des dispositifs en cours...");
                }
                growingPlanService.importPZ0GrowingPlans(allResults);

                if (log.isInfoEnabled()) {
                    log.info("Sauvegarde des systèmes de culture en cours...");
                }
                growingSystemService.importPZ0GrowingSystems(allResults);

                addInterventionDomainId(allResults);
                if (log.isInfoEnabled()) {
                    log.info("Sauvegarde des parcelles en cours...");
                }
                plotService.importPZ0Plots(allResults);

                if (log.isInfoEnabled()) {
                    log.info("Sauvegarde des produits fertilisant minéraux en cours...");
                }
                referentialService.importRefFertiMinUnifaPz0(allResults);

                if (log.isInfoEnabled()) {
                    log.info("Sauvegarde des systèmes synthétisé en cours...");
                }
                practicedSystemService.importPZ0PracticedSystems(allResults);

                if (log.isInfoEnabled()) {
                    log.info("Sauvegarde des parcelles du synthétisé en cours...");
                }
                practicedPlotService.importPZ0PracticedPlots(allResults);

                EffectiveCropCycleImporter effectiveCropCycleImporter = new EffectiveCropCycleImporter();
                Collection<EffectiveCropCycleAndDependencies> effectiveCropCycleAndDependencies = effectiveCropCycleImporter.propagateTopiaIdAndGetAllEffectiveCropCycleAndDependencies();
                if (log.isInfoEnabled()) {
                    log.info("Sauvegarde des cycles de culture du réalisé en cours...");
                }
                effectiveCropCycleService.importPZ0EffectiveCropCycles(effectiveCropCycleAndDependencies);

                authorizationService.importPz0UsersRoles(allResults);

                getPersistenceContext().commit();
                displayPersistenceImportResult(allResults);
            } catch (Exception e) {
                log.error("ECHEC DE PERSISTANCE, aucun n'objet n'a été persisté !");
                e.printStackTrace();
                log.error(e);
            }

        } else {
            if (log.isErrorEnabled()) {
                log.error("LES FICHIERS COMPORTANT DES ERREURS ET DOIVENT ÊTRE CORRIGÉS AVANT DE PROCEDER À LA SAUVEGARDE DES ENTITÉS !\nPROCESSUS INTERROMPU");
            }
        }
    }

    protected void addInterventionDomainId(Map<Class, ImportResults> allResults) {
        ImportResults practicedSystemsResult = allResults.get(PracticedSystem.class);
        if (practicedSystemsResult != null && practicedSystemsResult.getIgnoredRecords() == 0) {
            Map<String, EntityAndDependencies> entitiesAndDependencies = practicedSystemsResult.getEntityAndDepsByCsvIds();

            for (Map.Entry<String, EntityAndDependencies> practicedSystemAndDependenciesByCsvId : entitiesAndDependencies.entrySet()) {
                PracticedSystemAndDependencies practicedSystemAndDependencies = (PracticedSystemAndDependencies) practicedSystemAndDependenciesByCsvId.getValue();

                List<Pz0PracticedSeasonalCropCycle> pz0SeasonalCropCycles = practicedSystemAndDependencies.getPz0PracticedSeasonalCropCycles();
                PracticedSystem ps = practicedSystemAndDependencies.getEntity();
                String domainId = ps.getGrowingSystem().getGrowingPlan().getDomain().getCode();

                setPracticeSeasonalInterventionsDomainId(practicedSystemsResult, practicedSystemAndDependenciesByCsvId, practicedSystemAndDependencies, pz0SeasonalCropCycles, domainId);
                setPracticedPhasesInterventionsDomainId(practicedSystemAndDependencies, domainId);
            }
        }
    }

    protected void setPracticeSeasonalInterventionsDomainId(ImportResults practicedSystemsResult, Map.Entry<String, EntityAndDependencies> stringEntityAndDependenciesEntry, PracticedSystemAndDependencies practicedSystemAndDependencies, Collection<Pz0PracticedSeasonalCropCycle> pz0SeasonalCropCycles, String domainId) {
        if (pz0SeasonalCropCycles != null) {
            for (Pz0PracticedSeasonalCropCycle pz0SeasonalCropCycle : pz0SeasonalCropCycles) {
                if (!CollectionUtils.isEmpty(pz0SeasonalCropCycle.getPracticedSeasonalCropCycleDto().getCropCycleNodeDtos()) && !CollectionUtils.isEmpty(pz0SeasonalCropCycle.getPz0Connections())) {
                    if (pz0SeasonalCropCycle.getPz0Connections() != null) {
                        for (Pz0PracticedCropCycleConnection pz0Connection : pz0SeasonalCropCycle.getPz0Connections()) {
                            Collection<Pz0PracticedIntervention> pz0Interventions = pz0Connection.getPz0InterventionByCsvId().values();
                            for (Pz0PracticedIntervention pz0Intervention : pz0Interventions) {
                                pz0Intervention.getPracticedInterventionDto().setDomainId(domainId);
                            }
                        }
                    }
                    practicedSystemsResult.addInfoLine(practicedSystemAndDependencies.getImportedNumber(), "SYSTÈME SYNTHÉTISÉ VALIDÉ, csvid: " + stringEntityAndDependenciesEntry.getKey());
                }
            }
        }
    }

    protected void setPracticedPhasesInterventionsDomainId(PracticedSystemAndDependencies practicedSystemAndDependencies, String domainId) {
        List<Pz0PracticedPerennialCropCycle> pz0PracticedPerennialCropCycles = practicedSystemAndDependencies.getPz0PracticedPerennialCropCycles();
        if (pz0PracticedPerennialCropCycles != null) {
            for (Pz0PracticedPerennialCropCycle pz0PerennialCropCycle : pz0PracticedPerennialCropCycles) {
                Collection<Pz0PracticedCropCyclePhase> pz0Phases = pz0PerennialCropCycle.getPz0PhasesByCsvId().values();
                for (Pz0PracticedCropCyclePhase phase : pz0Phases) {
                    Collection<Pz0PracticedIntervention> pz0Interventions = phase.getPz0InterventionByCsvId().values();
                    for (Pz0PracticedIntervention interventionDto : pz0Interventions) {
                        interventionDto.getPracticedInterventionDto().setDomainId(domainId);
                    }

                }
            }
        }
    }

    protected boolean isImportErrors(Map<Class, ImportResults> allResults) {
        boolean errors = false;
        for (ImportResults result : allResults.values()) {
            if (!result.isImportSuccess()) {
                errors = true;
                break;
            }
        }
        return errors;
    }

    protected void importDomains(String domainFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        log.info("IMPORT DES DOMAINES EN COURS...");
        DomainImporter domainImporter = new DomainImporter();
        domainImporter.init(getServiceContext());
        ImportResults domainImportResults = domainImporter.importFromStream(new FileInputStream(domainFilePath), null);
        displayImportResult(domainImportResults);
        allResults.put(Domain.class, domainImportResults);
    }

    protected void importCroppingPlans(String croppingPlanFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        log.info("IMPORT DES CULTURES EN COURS...");
        CroppingPanEntryImporter croppingPanEntryImporter = new CroppingPanEntryImporter();
        croppingPanEntryImporter.init(getServiceContext());
        if (StringUtils.isNotBlank(croppingPlanFilePath) && (new File(croppingPlanFilePath).exists())) {
            ImportResults croppingPlanEntryImportResults = croppingPanEntryImporter.importFromStream(new FileInputStream(croppingPlanFilePath), null);
            displayImportResult(croppingPlanEntryImportResults);
            allResults.put(CroppingPlanEntryDto.class, croppingPlanEntryImportResults);
        } else {
            log.info("Pas de fichier d'import de cultures");
        }
    }

    protected void importCroppingPlanSpecies(String croppingPlanSpeciesFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        log.info("IMPORT DES ESPÈCES EN COURS...");
        CroppingPanEntrySpeciesImporter croppingPanEntrySpeciesImporter = new CroppingPanEntrySpeciesImporter();
        croppingPanEntrySpeciesImporter.init(getServiceContext());
        if (StringUtils.isNotBlank(croppingPlanSpeciesFilePath) && (new File(croppingPlanSpeciesFilePath).exists())) {
            ImportResults croppingPlanSpeciesImportResults = croppingPanEntrySpeciesImporter.importFromStream(new FileInputStream(croppingPlanSpeciesFilePath), null);
            displayImportResult(croppingPlanSpeciesImportResults);
            allResults.put(CroppingPlanSpeciesImportDto.class, croppingPlanSpeciesImportResults);
        } else {
            log.info("Pas de fichier d'import des espèces");
        }
    }

    protected void importEquipments(String equipmentFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        log.info("IMPORT DES MATÉRIELS EN COURS...");

        EquipmentImporter equipmentsImporter = new EquipmentImporter();
        equipmentsImporter.init(getServiceContext());
        if (StringUtils.isNotBlank(equipmentFilePath) && (new File(equipmentFilePath).exists())) {
            ImportResults domainsImportResults = allResults.get(Domain.class);
            ImportResults equipmentsImportResults = equipmentsImporter.importFromStream(new FileInputStream(equipmentFilePath), domainsImportResults.getEntityAndDepsByCsvIds());
            displayImportResult(equipmentsImportResults);
            allResults.put(Equipment.class, equipmentsImportResults);
        } else {
            log.info("Pas de fichier d'import des équipements");
        }
    }

    protected void importGrounds(String groundFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        log.info("IMPORT DES SOLS EN COURS...");

        GroundImporter groundImporter = new GroundImporter();
        groundImporter.init(getServiceContext());
        if (StringUtils.isNotBlank(groundFilePath) && (new File(groundFilePath).exists())) {
            ImportResults domainsImportResults = allResults.get(Domain.class);
            ImportResults equipmentsImportResults = groundImporter.importFromStream(new FileInputStream(groundFilePath), domainsImportResults.getEntityAndDepsByCsvIds());
            displayImportResult(equipmentsImportResults);
            allResults.put(Ground.class, equipmentsImportResults);
        } else {
            log.info("Pas de fichier d'import des sols");
        }
    }
    protected void importToolsCouplings(String toolCouplingFilePath, String equipmentToToolCouplingFilePath, String mainActionToToolCouplingFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        log.info("IMPORT DES COMBINAISONS D'OUTILS EN COURS...");

        ToolsCouplingImporter toolsCouplingImporter = new ToolsCouplingImporter();
        toolsCouplingImporter.init(getServiceContext());
        if (StringUtils.isNotBlank(toolCouplingFilePath) && (new File(toolCouplingFilePath).exists())) {
            ImportResults domainsImportResults = allResults.get(Domain.class);
            ImportResults toolsCouplingsImportResults = toolsCouplingImporter.importFromStream(new FileInputStream(toolCouplingFilePath), domainsImportResults.getEntityAndDepsByCsvIds());

            log.info("AJOUT DES ÉQUIPEMENTS AUX COMBINAISONS D'OUTILS EN COURS...");
            importEquipmentToToolCoupling(equipmentToToolCouplingFilePath, allResults, toolsCouplingsImportResults);
            log.info("AJOUT DES ACTION PRINCIPALES AUX COMBINAISONS D'OUTILS EN COURS...");
            importToolsCouplingMainsActions(mainActionToToolCouplingFilePath, allResults, toolsCouplingsImportResults);

            log.info("VALIDATION QUE TOUTES LES COMBINAISONS D'OUTILS ON AU MOINS UNE ACTION PRINCIPALE EN COURS...");
            validToolsCouplingMainActionExist(domainsImportResults.getEntityAndDepsByCsvIds(), toolsCouplingsImportResults);

            log.info("VALIDATION DES EQUIPEMENTS DES COMBINAISONS D'OUTILS EN COURS...");
            validToolsCouplingsEquipmentPreconditions(domainsImportResults.getEntityAndDepsByCsvIds(), toolsCouplingsImportResults);

            allResults.put(ToolsCoupling.class, toolsCouplingsImportResults);

            displayImportResult(toolsCouplingsImportResults);
        } else {
            log.warn("Pas de fichier d'import des combinaisons d'outils");
        }
    }

    protected void validToolsCouplingMainActionExist(Map<String, EntityAndDependencies> domainsAndDependencies, ImportResults toolsCouplingsImportResults) {
        for (Map.Entry<String, EntityAndDependencies> entityAndDependencies : domainsAndDependencies.entrySet()) {
            DomainAndDependencies domainAndDependencies = (DomainAndDependencies) entityAndDependencies.getValue();
            Collection<ToolsCoupling> toolsCouplings = domainAndDependencies.getToolsCouplings();

            for (ToolsCoupling toolsCoupling : toolsCouplings) {
                if(CollectionUtils.isEmpty(toolsCoupling.getMainsActions())) {
                    String scvId = domainAndDependencies.getCsvIdForToolsCouplings(toolsCoupling);
                    toolsCouplingsImportResults.increaseIgnoredRecords();
                    toolsCouplingsImportResults.addErrorLine(domainAndDependencies.getLine(), String.format("COMBINAISON D'OUTILS IGNORÉE, la combinaison d'outil %s du domaine %s doit avoir au moins une action principale associée.", scvId, entityAndDependencies.getKey()));
                }
            }
        }
    }

    protected void validToolsCouplingsEquipmentPreconditions(Map<String, EntityAndDependencies> domainsAndDependencies, ImportResults toolsCouplingsImportResults) {
        for (Map.Entry<String, EntityAndDependencies> entityAndDependencies : domainsAndDependencies.entrySet()) {
            DomainAndDependencies domainAndDependencies = (DomainAndDependencies) entityAndDependencies.getValue();
            Collection<ToolsCoupling> toolsCouplings = domainAndDependencies.getToolsCouplings();

            Map<String, Equipment> equipmentsByCode =  domainAndDependencies.getEquipmentsByCode();

            for (ToolsCoupling toolsCoupling : toolsCouplings) {
                String csvId = domainAndDependencies.getCsvIdForToolsCouplings(toolsCoupling);
                if (!toolsCoupling.isManualIntervention()) {
                    if (!(toolsCoupling.getTractor() != null && toolsCoupling.getTractor().getRefMateriel() != null && toolsCoupling.getTractor().getRefMateriel() instanceof RefMaterielAutomoteur)) {
                        if(CollectionUtils.isEmpty(toolsCoupling.getEquipments())){
                            toolsCouplingsImportResults.increaseIgnoredRecords();
                            toolsCouplingsImportResults.addErrorLine(domainAndDependencies.getLine(), String.format("COMBINAISON D'OUTILS IGNORÉE, la combinaison d'outil %s du domaine %s doit avoir au moins un equipement associé.", csvId, entityAndDependencies.getKey()));
                        }

                    }

                    Collection<Equipment> tcEquipments = toolsCoupling.getEquipments();
                    boolean isIrrigationEquipments = false;
                    boolean isIrrigationEquipmentAndNonIrrigationEquipments = false;
                    if (CollectionUtils.isNotEmpty(tcEquipments)) {
                        for (Equipment tcEquipment : tcEquipments) {

                            // valid equipment belong to domain
                            if (!equipmentsByCode.containsKey(tcEquipment.getCode())) {
                                toolsCouplingsImportResults.increaseIgnoredRecords();
                                toolsCouplingsImportResults.addErrorLine(domainAndDependencies.getLine(), String.format("COMBINAISON D'OUTILS IGNORÉE, l'équipement '%s' de la combinaison d'outil %s n'existe pas sur le domaine %s.", tcEquipment.getName(), csvId, entityAndDependencies.getKey()));
                            }

                            boolean isIrrigationEquipment = tcEquipment.getRefMateriel() != null && tcEquipment.getRefMateriel() instanceof RefMaterielIrrigation;
                            if (isIrrigationEquipment) {
                                isIrrigationEquipments = true;
                            } else {
                                isIrrigationEquipmentAndNonIrrigationEquipments = true;
                            }
                        }
                    }

                    Equipment tractor = toolsCoupling.getTractor();
                    if (tractor == null) {
                        if (isIrrigationEquipments) {
                            toolsCouplingsImportResults.increaseIgnoredRecords();
                            toolsCouplingsImportResults.addErrorLine(domainAndDependencies.getLine(),
                                    String.format("Combinaison d'outils non valide, " +
                                            "aucun tracteur/automoteur/matériel d'irrigation présent sur la combinaison d'outils ayant comme identifiant: %s !", csvId));
                        }
                        if (isIrrigationEquipmentAndNonIrrigationEquipments) {
                            toolsCouplingsImportResults.increaseIgnoredRecords();
                            toolsCouplingsImportResults.addErrorLine(domainAndDependencies.getLine(),
                                    String.format("Combinaison d'outils non valide," +
                                            " du matériel d'irrigation ne peut être associé avec du matériel autre qu'un tracteur ou un automoteur, combinaison d'outils concernée: %s !", csvId));
                        }
                    } else {
                        // if tractor is not auto-motor's type, it must be associated to 1+n equipment
                        boolean isTractor = tractor.getRefMateriel() != null && !(tractor.getRefMateriel() instanceof RefMaterielAutomoteur);
                        if (isTractor) {
                            if (CollectionUtils.isEmpty(tcEquipments)) {
                                toolsCouplingsImportResults.increaseIgnoredRecords();
                                toolsCouplingsImportResults.addErrorLine(domainAndDependencies.getLine(),
                                        String.format("Combinaison d'outils non valide, aucun équipement présent avec le tracteur, combinaison d'outils concernée: %s !", csvId));
                            }
                        }

                        if (CollectionUtils.isNotEmpty(tcEquipments)) {
                            if (CollectionUtils.isNotEmpty(tcEquipments)) {
                                if (isIrrigationEquipments && isIrrigationEquipmentAndNonIrrigationEquipments) {
                                    toolsCouplingsImportResults.increaseIgnoredRecords();
                                    toolsCouplingsImportResults.addErrorLine(domainAndDependencies.getLine(),
                                            String.format("Combinaison d'outils non valide, " +
                                                    "du matériel d'irrigation ne peut être associé avec du matériel autre qu'un tracteur ou un automoteur, combinaison d'outils concernée: %s !", csvId));
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    protected void importEquipmentToToolCoupling(String equipmentToToolCouplingFilePath, Map<Class, ImportResults> allResults, ImportResults toolsCouplingsImportResults) throws FileNotFoundException {
        if (StringUtils.isNotBlank(equipmentToToolCouplingFilePath) && (new File(equipmentToToolCouplingFilePath).exists())) {
            EquipmentForToolsCouplingImporter equipmentForToolCouplingImporter = new EquipmentForToolsCouplingImporter();
            equipmentForToolCouplingImporter.init(getServiceContext());
            ImportResults equipmentForToolCoupling = equipmentForToolCouplingImporter.importFromStream(new FileInputStream(equipmentToToolCouplingFilePath), null);
            displayImportResult(equipmentForToolCoupling);
            allResults.put(ToolsCoupling.class, equipmentForToolCoupling);
        } else {
            log.warn("Pas de fichier d'import des équipements trouvé !");
        }
    }

    protected void importToolsCouplingMainsActions(String mainsActionsToolCouplingFilePath, Map<Class, ImportResults> allResults, ImportResults toolsCouplingsImportResults) throws FileNotFoundException {
        if (StringUtils.isNotBlank(mainsActionsToolCouplingFilePath) && (new File(mainsActionsToolCouplingFilePath).exists())) {
            MainActionForToolsCouplingImporter mainActionForToolsCouplingImporter = new MainActionForToolsCouplingImporter();
            mainActionForToolsCouplingImporter.init(getServiceContext());
            ImportResults mainsActionsForToolsCouplings = mainActionForToolsCouplingImporter.importFromStream(new FileInputStream(mainsActionsToolCouplingFilePath), null);
            displayImportResult(mainsActionsForToolsCouplings);
            allResults.put(ToolsCoupling.class, mainsActionsForToolsCouplings);
        } else {
            toolsCouplingsImportResults.addErrorLine(0L, "FICHIER DES ACTIONS PRINCIPALES NON TROUVÉ, une combinaison d'outils doit obligatoirement être associée avec au moins une action principale.");
        }
    }

    protected void importGrowingPlans(String growingPlanFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        log.info("IMPORT DES DISPOSITIFS EN COURS...");
        if (StringUtils.isNotBlank(growingPlanFilePath) && (new File(growingPlanFilePath).exists())) {
            GrowingPlanImporter growingPlanImporter = new GrowingPlanImporter();
            growingPlanImporter.init(getServiceContext());
            ImportResults domainImportResults = allResults.get(Domain.class);
            ImportResults growingPlanImportResults = growingPlanImporter.importFromStream(new FileInputStream(growingPlanFilePath), domainImportResults.getEntityAndDepsByCsvIds());
            displayImportResult(growingPlanImportResults);
            allResults.put(GrowingPlan.class, growingPlanImportResults);
        } else {
            log.warn("Pas de fichier d'import des dispositifs trouvé !");
        }

    }

    protected void importGrowingSystems(String growingSystemFilePath, String growingSystemNetworksFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        log.info("IMPORT DES SYSTÈMES DE CULTURE EN COURS...");
        GrowingSystemImporter growingSystemImporter = new GrowingSystemImporter();
        growingSystemImporter.init(getServiceContext());
        ImportResults growingPlanImportResults = allResults.get(GrowingPlan.class);
        ImportResults growingSystemImportResults = growingSystemImporter.importFromStream(new FileInputStream(growingSystemFilePath), growingPlanImportResults.getEntityAndDepsByCsvIds());
        displayImportResult(growingSystemImportResults);

        if (StringUtils.isNotBlank(growingSystemNetworksFilePath) && (new File(growingSystemNetworksFilePath).exists())) {
            log.info("APPAIRAGE DES SDS AVEC LES RÉSEAUX EN COURS...");
            GrowingSystemNetworkImporter growingSystemNetworkImporter = new GrowingSystemNetworkImporter();
            growingSystemNetworkImporter.init(getServiceContext());
            ImportResults growingSystemNetworksImportResults = growingSystemNetworkImporter.importFromStream(new FileInputStream(growingSystemNetworksFilePath), growingSystemImportResults.getEntityAndDepsByCsvIds());
            displayImportResult(growingSystemNetworksImportResults);
            allResults.put(GrowingSystemNetworkImportDto.class, growingSystemNetworksImportResults);
        } else {
            log.warn("Pas de fichier d'appairage des SDC et réseaux trouvé !");
        }
        allResults.put(GrowingSystem.class, growingSystemImportResults);
    }

    protected void importPlots(String plotFilePath, String zoneFilePAth, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        log.info("IMPORT DES PARCELLE EN COURS...");
        ImportResults plotImportResults = null;
        if (StringUtils.isNotBlank(plotFilePath)  && (new File(plotFilePath).exists())) {
            PlotImporter importer = new PlotImporter();
            importer.init(getServiceContext());
            ImportResults domainImportResults = allResults.get(Domain.class);
            ImportResults growingSystemImportResults = allResults.get(GrowingSystem.class);
            Map<String, EntityAndDependencies> requiredEntities = Maps.newHashMap(domainImportResults.getEntityAndDepsByCsvIds());
            requiredEntities.putAll(growingSystemImportResults.getEntityAndDepsByCsvIds());
            plotImportResults = importer.importFromStream(new FileInputStream(plotFilePath), requiredEntities);
            allResults.put(Plot.class, plotImportResults);
        } else {
            log.warn("Pas de fichier d'import des parcelles trouvé !");
        }

        log.info("IMPORT DES ZONES EN COURS...");
        if (StringUtils.isNotBlank(zoneFilePAth)  && (new File(zoneFilePAth).exists())) {
            ZoneImporter importer = new ZoneImporter();
            importer.init(getServiceContext());
            Map<String, EntityAndDependencies> plotAndDependenciesByCsvId = plotImportResults != null ? plotImportResults.getEntityAndDepsByCsvIds() : null;
            ImportResults zoneImportResults = importer.importFromStream(new FileInputStream(zoneFilePAth), plotAndDependenciesByCsvId);
            displayImportResult(zoneImportResults);
        } else {
            log.warn("Pas de fichier d'import des zones trouvé !");
        }

        log.info("VALIDE LE FAIT QU'UNE PARCELLE EST RATTACHÉE À UNE ZONE...");
        if (plotImportResults != null) {
            Map<String, EntityAndDependencies> plotByCsvId = plotImportResults.getEntityAndDepsByCsvIds();
            for (Map.Entry<String, EntityAndDependencies> csvIdEntityAndDependenciesEntry : plotByCsvId.entrySet()) {
                PlotAndDependencies plotAndDependencies = (PlotAndDependencies) csvIdEntityAndDependenciesEntry.getValue();
                if (plotAndDependencies.getZones() == null) {
                    plotImportResults.increaseIgnoredRecords();
                    plotImportResults.addErrorLine(0L, String.format("PARCELLE IGNORÉE, la parcelle avec comme identifiant %s n'a pas de zone de définie", csvIdEntityAndDependenciesEntry.getKey()));
                }
            }
            displayImportResult(plotImportResults);
        }

    }

    protected void importPracticedSystems(String practicedSystemFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        log.info("IMPORT DES SYSTÈMES SYNTHÉTISÉS EN COURS...");
        if (StringUtils.isNotBlank(practicedSystemFilePath)  && (new File(practicedSystemFilePath).exists())) {
            PracticedSystemImporter importer = new PracticedSystemImporter();
            importer.init(getServiceContext());
            ImportResults growingSystemImportResults = allResults.get(GrowingSystem.class);
            ImportResults practicedSystemImportResults = importer.importFromStream(new FileInputStream(practicedSystemFilePath), growingSystemImportResults.getEntityAndDepsByCsvIds());
            // do not display result at this stade as it need more validation
            allResults.put(PracticedSystem.class, practicedSystemImportResults);
        } else {
            log.warn("Pas de fichier d'import des systèmes synthétisé trouvé !");
        }
    }

    protected void importPracticedPlots(String practicedPlotFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        log.info("IMPORT DES PARCELLES DU SYNTHÉTISÉ EN COURS...");
        if (StringUtils.isNotBlank(practicedPlotFilePath)  && (new File(practicedPlotFilePath).exists())) {
            PracticedPlotImporter importer = new PracticedPlotImporter();
            importer.init(getServiceContext());
            ImportResults practicedSystemImportResults = allResults.get(PracticedSystem.class);
            ImportResults practicedPlotImportResults = importer.importFromStream(new FileInputStream(practicedPlotFilePath), practicedSystemImportResults.getEntityAndDepsByCsvIds());
            displayImportResult(practicedPlotImportResults);
            allResults.put(PracticedPlot.class, practicedPlotImportResults);
        } else {
            log.warn("Pas de fichier d'import des parcelle du synthétisé trouvé !");
        }
    }

    protected void importPracticedCropCycle(
            String practicedCropCycleFilePath,
            String practicedSeasonalCropCycleFilePath,
            String practicedCropCycleNodeFilePath,
            String practicedCropCycleConnectionFilePath,
            String practicedPerennialCropCycleFilePath,
            String practicedCropCyclePhaseFilePath,
            String practicedCropCycleSpeciesFilePath,
            Map<Class, ImportResults> allResults) throws FileNotFoundException {
        log.info("IMPORT DES CYCLES DE CULTURES DU SYNTHÉTISÉ EN COURS...");
        if (StringUtils.isNotBlank(practicedCropCycleFilePath) && (new File(practicedCropCycleFilePath).exists())) {
            PracticedCropCycleImporter importer = new PracticedCropCycleImporter();
            ImportResults practicedSystemImportResults = allResults.get(PracticedSystem.class);
            ImportResults importResults = importer.importFromStream(new FileInputStream(practicedCropCycleFilePath), practicedSystemImportResults.getEntityAndDepsByCsvIds());
            displayImportResult(importResults);
            allResults.put(PracticedCropCycle.class, importResults);
        }  else {
            log.warn("Pas de fichier d'import des cycles de cultures du synthétisé trouvé !");
        }
        importPracticedSeasonalCropCycle(practicedSeasonalCropCycleFilePath, practicedCropCycleNodeFilePath, practicedCropCycleConnectionFilePath, allResults);
        importPracticedPerennialCropCycle(practicedPerennialCropCycleFilePath, practicedCropCyclePhaseFilePath, practicedCropCycleSpeciesFilePath, allResults);
    }

    protected void importPracticedSeasonalCropCycle(String practicedSeasonalCropCycleFilePath, String practicedCropCycleNodeFilePath, String practicedCropCycleConnectionFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        log.info("IMPORT DES CYCLES DE CULTURES SAISONNIÈRES DU SYNTHÉTISÉ EN COURS...");
        if (StringUtils.isNotBlank(practicedSeasonalCropCycleFilePath) && (new File(practicedSeasonalCropCycleFilePath).exists())) {
            PracticedSeasonalCropCycleImporter importer = new PracticedSeasonalCropCycleImporter();
            ImportResults importResults = importer.importFromStream(new FileInputStream(practicedSeasonalCropCycleFilePath), null);
            allResults.put(PracticedSeasonalCropCycle.class, importResults);
            displayImportResult(importResults);

        } else {
            log.warn("Pas de fichier d'import des cycle de cultures saisonnières du synthétisé trouvé !");
        }

        log.info("AJOUT DES NOEUD AU CYCLES DE CULTURES SAISONNIÈRES EN COURS...");
        if (StringUtils.isNotBlank(practicedCropCycleNodeFilePath) && (new File(practicedCropCycleNodeFilePath).exists())) {
            PracticedCropCycleNodeImporter importer = new PracticedCropCycleNodeImporter();
            ImportResults importResults = importer.importFromStream(new FileInputStream(practicedCropCycleNodeFilePath), null);
            allResults.put(PracticedCropCycleNode.class, importResults);
            displayImportResult(importResults);
        } else {
            log.warn("Pas de fichier d'import des noeuds du synthétisé trouvé !");
        }

        log.info("AJOUT DES CONNEXIONS AU CYCLES DE CULTURES SAISONNIÈRES EN COURS...");
        if (StringUtils.isNotBlank(practicedCropCycleConnectionFilePath) && (new File(practicedCropCycleConnectionFilePath).exists())) {
            PracticedCropCycleConnectionImporter importer = new PracticedCropCycleConnectionImporter();
            ImportResults importResults = importer.importFromStream(new FileInputStream(practicedCropCycleConnectionFilePath), null);
            allResults.put(PracticedCropCycleConnection.class, importResults);
            displayImportResult(importResults);
        } else {
            log.warn("Pas de fichier d'import des connexion du synthétisé trouvé !");
        }
    }

    protected void importPracticedPerennialCropCycle(String practicedPerennialCropCycleFilePath, String practicedCropCyclePhaseFilePath, String practicedCropCycleSpeciesFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        log.info("IMPORT DES CYCLES DE CULTURES PÉRENNES DU SYNTHÉTISÉ EN COURS...");
        ImportResults practicedPerennialCropCycleImportResults;
        if (StringUtils.isNotBlank(practicedPerennialCropCycleFilePath) && (new File(practicedPerennialCropCycleFilePath).exists())) {
            PracticedPerennialCropCycleImporter importer = new PracticedPerennialCropCycleImporter();
            importer.init(getServiceContext());
            practicedPerennialCropCycleImportResults = importer.importFromStream(new FileInputStream(practicedPerennialCropCycleFilePath), allResults.get(PracticedSystem.class).getEntityAndDepsByCsvIds());
            displayImportResult(practicedPerennialCropCycleImportResults);
            allResults.put(PracticedPerennialCropCycle.class, practicedPerennialCropCycleImportResults);
        }  else {
            log.warn("Pas de fichier d'import des cycles de cultures perennes du synthétisé trouvé !");
        }

        log.info("AJOUT DES PHASE AUX CYCLES DE CULTURES PÉRENNES DU SYNTHÉTISÉ EN COURS...");
        if (StringUtils.isNotBlank(practicedCropCyclePhaseFilePath) && (new File(practicedCropCyclePhaseFilePath).exists())) {
            PracticedPerennialCropCyclePhaseImporter importer = new PracticedPerennialCropCyclePhaseImporter();
            ImportResults importResults = importer.importFromStream(new FileInputStream(practicedCropCyclePhaseFilePath), null);
            displayImportResult(importResults);
            allResults.put(PracticedCropCyclePhase.class, importResults);
        } else {
            log.warn("Pas de fichier d'import des phases du synthétisé trouvé !");
        }

        log.warn("AJOUT DES ESPÈCES AUX CYCLES DE CULTURES PÉRENNES DU SYNTHÉTISÉ, Import non géré actuellement !");
        if (StringUtils.isNotBlank(practicedCropCycleSpeciesFilePath) && (new File(practicedCropCycleSpeciesFilePath).exists())) {
            // TODO DCossé 01/04/15
            log.error("Import non implémenté !");
        }  else {
            log.warn("Pas de fichier d'import des espèces du synthétisé trouvé !");
        }
    }

    protected void importPracticedIntervention(String practicedInterventionFilePath, String practiceInterventionToolsCouplingFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        log.info("IMPORT DES INTERVENTIONS DU SYNTHÉTISÉ EN COURS...");
        boolean canDoIt = StringUtils.isNotBlank(practicedInterventionFilePath) && (new File(practicedInterventionFilePath).exists());
        if (canDoIt) {
            PracticedInterventionImporter importer = new PracticedInterventionImporter();
            ImportResults importResults = importer.importFromStream(new FileInputStream(practicedInterventionFilePath), null);
            displayImportResult(importResults);
            allResults.put(PracticedIntervention.class, importResults);
        } else {
            log.warn("Pas de fichier d'import des interventions du synthétisé trouvé !");
        }
        log.info("AJOUT DES COMBINAISONS D'OUTILS AUX INTERVENTIONS DU SYNTHÉTISÉ EN COURS...");
        if (canDoIt) {
            if (StringUtils.isBlank(practiceInterventionToolsCouplingFilePath) && !(new File(practiceInterventionToolsCouplingFilePath).exists())) {
                log.warn("Il manque le fichier d'import des intervention du synthétisé pour procéder à l'import des combinaisons d'outils liées au interventions du synthétisé !");
            } else {
                importPracticeInterventionToolsCoupling(practiceInterventionToolsCouplingFilePath, allResults);
            }
        }
    }

    protected void importPracticeInterventionToolsCoupling(String practiceInterventionToolsCouplingFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        if (StringUtils.isNotBlank(practiceInterventionToolsCouplingFilePath) && (new File(practiceInterventionToolsCouplingFilePath).exists())) {
            PracticedToolsCouplingImporter importer = new PracticedToolsCouplingImporter();
            ImportResults importResults = importer.importFromStream(new FileInputStream(practiceInterventionToolsCouplingFilePath), null);
            displayImportResult(importResults);
            allResults.put(PracticedToolsCouplingImportDto.class, importResults);
        } else {
            log.warn("Pas de fichier d'import des combinaisons d'outils des interventions du synthétisé trouvé !");
        }
    }

    protected void importEffectiveCropCycleUniverse(
            String effectivePerennialCropCycleFilePath,
            String effectiveCropCycleSpeciesFilePath,
            String effectivePhaseFilePath,

            String effectiveSeasonalCropCycleFilePath ,
            String effectiveCropCycleNodeFilePath,
            String effectiveCropCycleConnectionFilePath,

            String effectiveInterventionFilePath,
            String effectiveSpeciesStadeFilePath,
            String effectiveToolsCouplingFilePath,
            String effectiveInvolvedRuleFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {

        importEffectivePerennialCropCycle(effectivePerennialCropCycleFilePath, effectiveCropCycleSpeciesFilePath, effectivePhaseFilePath, allResults);
        importEffectiveSeasonalCropCycle(effectiveSeasonalCropCycleFilePath, effectiveCropCycleNodeFilePath, effectiveCropCycleConnectionFilePath, allResults);
        importEffectiveIntervention(effectiveInterventionFilePath, effectiveSpeciesStadeFilePath, effectiveToolsCouplingFilePath, effectiveInvolvedRuleFilePath, allResults);
    }

    private void importEffectivePerennialCropCycle(String effectivePerennialCropCycleFilePath, String effectiveCropCycleSpeciesFilePath, String effectivePhaseFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        log.info("IMPORT DES PHASE DES CYCLES DE CULTURE PÉRENNES DU RÉALISÉ EN COURS...");
        if (StringUtils.isNotBlank(effectivePhaseFilePath)  && (new File(effectivePhaseFilePath).exists())) {
            AbstractCSVImporter importer = new EffectivePhaseImporter();
            importer.init(getServiceContext());
            ImportResults importResults = importer.importFromStream(new FileInputStream(effectivePhaseFilePath), null);
            allResults.put(EffectiveCropCyclePhase.class, importResults);
            displayImportResult(importResults);
        } else {
            log.warn("Pas de fichier d'import des phases du réalisé trouvé !");
        }

        log.info("IMPORT DES CYCLES DE CULTURE PÉRENNES DU RÉALISÉ EN COURS...");
        if (StringUtils.isNotBlank(effectivePerennialCropCycleFilePath)  && (new File(effectivePerennialCropCycleFilePath).exists())) {
            AbstractCSVImporter importer = new EffectivePerennialCropCycleImporter();
            importer.init(getServiceContext());
            ImportResults importResults = importer.importFromStream(new FileInputStream(effectivePerennialCropCycleFilePath), null);
            allResults.put(EffectivePerennialCropCycle.class, importResults);
            displayImportResult(importResults);
        } else {
            log.warn("Pas de fichier d'import des cycles de culture perennes du réalisé trouvé !");
        }

        log.info("IMPORT DES ESPÈCES AUX CYCLE DE CULTURE PÉRENNES DU RÉALISÉ EN COURS...");
        if (StringUtils.isNotBlank(effectiveCropCycleSpeciesFilePath)  && (new File(effectiveCropCycleSpeciesFilePath).exists())) {
            AbstractCSVImporter importer = new EffectiveCropCycleSpeciesImporter();
            importer.init(getServiceContext());
            ImportResults importResults = importer.importFromStream(new FileInputStream(effectiveCropCycleSpeciesFilePath), null);
            allResults.put(EffectiveCropCycleSpecies.class, importResults);
            displayImportResult(importResults);
        } else {
            log.warn("Pas de fichier d'import des espèces du cycle de culture du réalisé trouvé !");
        }
    }

    private void importEffectiveSeasonalCropCycle(String effectiveSeasonalCropCycleFilePath, String effectiveCropCycleNodeFilePath, String effectiveCropCycleConnectionFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        log.info("IMPORT DES CYCLES DE CULTURE SAISONNIÈRES DU RÉALISÉ EN COURS...");

        if (StringUtils.isNotBlank(effectiveSeasonalCropCycleFilePath)  && (new File(effectiveSeasonalCropCycleFilePath).exists())) {
            AbstractCSVImporter importer = new EffectiveSeasonalCropCycleImporter();
            importer.init(getServiceContext());
            ImportResults importResults = importer.importFromStream(new FileInputStream(effectiveSeasonalCropCycleFilePath), null);
            allResults.put(EffectiveSeasonalCropCycle.class, importResults);
            displayImportResult(importResults);
        } else {
            log.warn("Pas de fichier d'import des cycles de cultures saisonnières du réalisé trouvé !");
        }

        log.info("AJOUT DES NOEUDS AUX CYCLES DE CULTURE SAISONNIÈRES DU RÉALISÉ EN COURS...");
        if (StringUtils.isNotBlank(effectiveCropCycleNodeFilePath)  && (new File(effectiveCropCycleNodeFilePath).exists())) {
            AbstractCSVImporter importer = new EffectiveCropCycleNodeImporter();
            importer.init(getServiceContext());
            ImportResults importResults = importer.importFromStream(new FileInputStream(effectiveCropCycleNodeFilePath), null);
            allResults.put(EffectiveCropCycleNode.class, importResults);
            displayImportResult(importResults);
        } else {
            log.warn("Pas de fichier d'import des noeud du réalisé trouvé !");
        }

        log.info("AJOUT DES CONNEXIONS AUX CYCLES DE CULTURE SAISONNIÈRES DU RÉALISÉ EN COURS...");
        if (StringUtils.isNotBlank(effectiveCropCycleConnectionFilePath)  && (new File(effectiveCropCycleConnectionFilePath).exists())) {
            AbstractCSVImporter importer = new EffectiveCropCycleConnectionImporter();
            importer.init(getServiceContext());
            ImportResults importResults = importer.importFromStream(new FileInputStream(effectiveCropCycleConnectionFilePath), null);
            allResults.put(EffectiveCropCycleConnection.class, importResults);
            displayImportResult(importResults);
        } else {
            log.warn("Pas de fichier d'import des connexions du réalisé trouvé !");
        }
    }

    private void importEffectiveIntervention(String effectiveInterventionFilePath, String effectiveSpeciesStadeFilePath, String effectiveToolsCouplingFilePath, String effectiveInvolvedRuleFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        log.info("AJOUT DES INTERVENTIONS AUX CYCLES DE CULTURE SAISONNIÈRES DU RÉALISÉ EN COURS...");

        if (StringUtils.isNotBlank(effectiveInterventionFilePath)  && (new File(effectiveInterventionFilePath).exists())) {
            AbstractCSVImporter importer = new EffectiveInterventionImporter();
            importer.init(getServiceContext());
            ImportResults importResults = importer.importFromStream(new FileInputStream(effectiveInterventionFilePath), null);
            allResults.put(EffectiveIntervention.class, importResults);
            displayImportResult(importResults);
        } else {
            log.warn("Pas de fichier d'import des interventions du réalisé trouvé !");
        }

        log.info("AJOUT DES STADES DE CULTURE AUX INTERVENTIONS DU RÉALISÉ EN COURS...");
        if (StringUtils.isNotBlank(effectiveSpeciesStadeFilePath)  && (new File(effectiveSpeciesStadeFilePath).exists())) {
            AbstractCSVImporter importer = new EffectiveSpeciesStadeImporter();
            importer.init(getServiceContext());
            ImportResults importResults = importer.importFromStream(new FileInputStream(effectiveSpeciesStadeFilePath), null);
            allResults.put(EffectiveSpeciesStade.class, importResults);
            displayImportResult(importResults);
        } else {
            log.warn("Pas de fichier d'import des stades de culture du réalisé trouvé !");
        }

        log.info("AJOUT DES COMBINAISONS D'OUTILS AUX INTERVENTIONS DU RÉALISÉ EN COURS...");
        if (StringUtils.isNotBlank(effectiveToolsCouplingFilePath)  && (new File(effectiveToolsCouplingFilePath).exists())) {
            AbstractCSVImporter importer = new EffectiveToolsCouplingImporter();
            importer.init(getServiceContext());
            ImportResults importResults = importer.importFromStream(new FileInputStream(effectiveToolsCouplingFilePath), null);
            allResults.put(EffectiveCropCycleConnection.class, importResults);
            displayImportResult(importResults);
        } else {
            log.warn("Pas de fichier d'import des combinaison d'outils pour intervention du réalisé trouvé !");
        }

        if (StringUtils.isNotBlank(effectiveInvolvedRuleFilePath)  && (new File(effectiveInvolvedRuleFilePath).exists())) {
            // TODO DCossé 13/04/15
        }

    }

    protected void importPracticedSystemUniverse(
            String practicedSystemFilePath,
            String practicedPlotFilePath,
            String practicedCropCycleFilePath,
            String practicedSeasonalCropCycleFilePath,
            String practicedCropCycleNodeFilePath,
            String practicedCropCycleConnectionFilePath,
            String practicedPerennialCropCycleFilePath,
            String practicedCropCyclePhaseFilePath,
            String practicedCropCycleSpeciesFilePath,
            String practiceInterventionToolsCouplingFilePath,
            String practicedInterventionFilePath,
            Map<Class, ImportResults> allResults) throws FileNotFoundException {

        importPracticedSystems(practicedSystemFilePath, allResults);
        importPracticedPlots(practicedPlotFilePath, allResults);

        importPracticedCropCycle(practicedCropCycleFilePath, practicedSeasonalCropCycleFilePath, practicedCropCycleNodeFilePath, practicedCropCycleConnectionFilePath, practicedPerennialCropCycleFilePath, practicedCropCyclePhaseFilePath, practicedCropCycleSpeciesFilePath, allResults);

        validPracticedCropCycle(allResults);
        
        importPracticedIntervention(practicedInterventionFilePath, practiceInterventionToolsCouplingFilePath, allResults);
    }


    protected void importAbstractAction(String abstractActionFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        log.info("IMPORT DES ACTIONS EN COURS...");

        if (StringUtils.isNotBlank(abstractActionFilePath) && (new File(abstractActionFilePath).exists())) {
            ActionImporter importer = new ActionImporter();
            importer.init(getServiceContext());
            ImportResults importResults = importer.importFromStream(new FileInputStream(abstractActionFilePath), null);
            displayImportResult(importResults);
            allResults.put(AbstractAction.class, importResults);
        } else {
            log.warn("Pas de fichier d'import des action trouvé !");
        }
    }

    protected void importHarverstingYeald(String harverstingYealdFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        log.info("IMPORT DES RENDEMENTS DE RÉCOLTE EN COURS...");

        if (StringUtils.isNotBlank(harverstingYealdFilePath) && (new File(harverstingYealdFilePath).exists())){
            HarvestingYealdImporter importer = new HarvestingYealdImporter();
            ImportResults importResults = importer.importFromStream(new FileInputStream(harverstingYealdFilePath), null);
            displayImportResult(importResults);
            allResults.put(HarvestingYeald.class, importResults);
        } else {
            log.warn("Pas de fichier d'import des rendement de récolte trouvé !");
        }
    }

    protected void importSeedingActionSpecies(String seedingActionSpeciesFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        log.info("IMPORT DES ESPÈCES DES ACTIONS DE TYPE SEMIS EN COURS...");

        if(StringUtils.isNotBlank(seedingActionSpeciesFilePath) && (new File(seedingActionSpeciesFilePath).exists())) {
            SeedingActionSpeciesImporter importer = new SeedingActionSpeciesImporter();
            ImportResults importResults = importer.importFromStream(new FileInputStream(seedingActionSpeciesFilePath), null);
            displayImportResult(importResults);
            allResults.put(SeedingActionSpecies.class, importResults);
        } else {
            log.warn("Pas de fichier d'import des espèces des actions de type SEMIS trouvé !");
        }
    }

    protected void importAbstractInput(String inputFilePath, String fertiMinFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        log.info("IMPORT DES PRODUITS FERTILISANT MINÉRAUX EN COURS...");

        if (StringUtils.isNotBlank(fertiMinFilePath) && (new File(fertiMinFilePath).exists())) {
            RefFertiMinUnifaImporter ferMinImporter = new RefFertiMinUnifaImporter();
            ferMinImporter.init(getServiceContext());
            ImportResults fertiMinImportResults = ferMinImporter.importFromStream(new FileInputStream(fertiMinFilePath), null);
            displayImportResult(fertiMinImportResults);
            allResults.put(RefFertiMinUNIFA.class, fertiMinImportResults);
        }  else {
            log.warn("Pas de fichier d'import du référentiel FertiMinUnifa trouvé !");
        }

        log.info("IMPORT DES INTRANTS EN COURS...");

        if (StringUtils.isNotBlank(inputFilePath) && (new File(inputFilePath).exists())) {
            InputImporter importer = new InputImporter();
            importer.init(getServiceContext());
            ImportResults importResults = importer.importFromStream(new FileInputStream(inputFilePath), null);
            displayImportResult(importResults);
            allResults.put(Inputs.class, importResults);
        } else {
            log.warn("Pas de fichier d'import des intrants trouvé !");
        }
    }

    protected void importUsersRolesImporter(String usersRolesFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        log.info("IMPORT DES RÔLES UTILISATEUR EN COURS...");

        if (StringUtils.isNotBlank(usersRolesFilePath) && (new File(usersRolesFilePath).exists())) {
            UserRoleImporter userRoleImporter = new UserRoleImporter();
            userRoleImporter.init(getServiceContext());
            ImportResults usersRolesImportResults = userRoleImporter.importFromStream(new FileInputStream(usersRolesFilePath), null);
            displayImportResult(usersRolesImportResults);
            allResults.put(UserRoleDto.class, usersRolesImportResults);
        }  else {
            log.warn("Pas de fichier d'import des rôles utilisateur trouvé !");
        }
    }

    protected void displayImportResult(ImportResults importResults) {
        if (log.isInfoEnabled()){
            log.info(String.format("Import des %s", importResults.getImportedEntityClass().getSimpleName()));
        }

        if (log.isErrorEnabled()) {
            Map<Long,List<String>> errorMessages = importResults.getErrorLineNumbersWithMessage();
            for (Map.Entry<Long,List<String>> errorMessagesByLine : errorMessages.entrySet()) {
                long key = errorMessagesByLine.getKey();
                for (String message :errorMessagesByLine.getValue()) {
                    log.error(String.format("ligne %d, %s", key, message));
                }
            }
            errorMessages.clear();
        }
        if (log.isInfoEnabled()){
            LinkedHashMap<Long,List<String>> infoMessages = importResults.getInfoLineNumbersWithMessage();
            for (Map.Entry<Long,List<String>> infoMessageByLine : infoMessages.entrySet()) {
                long key = infoMessageByLine.getKey();
                for (String message :infoMessageByLine.getValue()) {
                    log.info(String.format("ligne %d, %s", key, message));
                }
            }
            infoMessages.clear();
        }
    }

    protected void displayPersistenceImportResult(Map<Class, ImportResults> allResults) {
        if (log.isInfoEnabled()){
            log.info("Persistance achevée avec succées");
            for (ImportResults importResults : allResults.values()) {
                log.info(String.format("%d %s importés.", importResults.getAddedRecords(), importResults.getImportedEntityClass().getSimpleName()));
                if (importResults.getIgnoredRecords() > 0) {
                    log.info((String.format("%d %s ignorés.",  importResults.getIgnoredRecords(), importResults.getImportedEntityClass().getSimpleName())));
                }
            }
        }
    }


    protected void userAuthentication() {
        String email = serviceContext.getAgrosystCliConfig().getUserEmail();
        String password = serviceContext.getAgrosystCliConfig().getUserPassword();
        String firstName = serviceContext.getAgrosystCliConfig().getUserFirstName();
        String lastName = serviceContext.getAgrosystCliConfig().getUserLastName();

        AuthenticationService authenticationService = serviceFactory.newService(AuthenticationService.class);

        AgrosystUserTopiaDao userDao = getPersistenceContext().getAgrosystUserDao();

        // if user does not exist a new Administrateur one is created
        if (!userDao.forEmailEquals(email).tryFindAny().isPresent()) {
            createAdminUser(email, password, firstName, lastName, userDao);
        }

        UserDto result = authenticationService.login(email, password);
        serviceContext.setAuthenticationToken(result.getAuthenticationToken());

        log.info("connecté avec l'utilisateur:" + result.getFirstName() + " " + result.getLastName() + " " + result.getEmail());
    }

    protected void createAdminUser(String email, String password, String firstName, String lastName, AgrosystUserTopiaDao userDao) {
        UserService userService = serviceFactory.newService(UserService.class);
        UserDto user = new UserDto();
        user.setActive(true);
        user.setEmail(email);
        user.setFirstName(firstName);
        user.setLastName(lastName);
        user = userService.createUser(user, password);

        AgrosystUser userEntity = userDao.forTopiaIdEquals(user.getTopiaId()).findUnique();

        UserRoleTopiaDao userRoleTopiaDao = getPersistenceContext().getUserRoleDao();
        userRoleTopiaDao.create(UserRole.PROPERTY_AGROSYST_USER, userEntity, UserRole.PROPERTY_TYPE, RoleType.ADMIN);
    }

    protected ServiceContext getServiceContext() {
        return serviceContext;
    }


    protected void init() {
        serviceContext = new AgrosystCliServiceContext(configFilePath);
        serviceFactory = serviceContext.getServiceFactory();
        domainService = serviceFactory.newService(DomainService.class);
        growingPlanService = serviceFactory.newService(GrowingPlanService.class);
        growingSystemService = serviceFactory.newService(GrowingSystemService.class);
        referentialService = serviceFactory.newService(ReferentialService.class);
        practicedSystemService = serviceFactory.newService(PracticedSystemService.class);
        practicedPlotService = serviceFactory.newService(PracticedPlotService.class);
        plotService= serviceFactory.newService(PlotService.class);

        domainFilePath  = serviceContext.getAgrosystCliConfig().getDomainFilePath();
        croppingPlanFilePath = serviceContext.getAgrosystCliConfig().getCroppingPanFilePath();
        croppingPlanSpeciesFilePath = serviceContext.getAgrosystCliConfig().getCroppingPlanSpeciesFilePath();

        equipmentFilePath = serviceContext.getAgrosystCliConfig().getEquipmentFilePath();
        toolCouplingFilePath = serviceContext.getAgrosystCliConfig().getToolCouplingFilePath();
        equipmentToToolsCouplingFilePath = serviceContext.getAgrosystCliConfig().getEquipmentToToolsCouplingFilePath();
        mainActionToToolsCouplingFilePath = serviceContext.getAgrosystCliConfig().getMainActionToToolsCouplingFilePath();
        groundFilePath = serviceContext.getAgrosystCliConfig().getGroundFilePath();

        growingPlanFilePath = serviceContext.getAgrosystCliConfig().getGrowingPlanFilePath();
        growingSystemFilePath = serviceContext.getAgrosystCliConfig().getGrowingSystemFilePath();
        growingSystemNetworksFilePath = serviceContext.getAgrosystCliConfig().getGrowingSystemNetworkFilePath();

        practicedSystemFilePath = serviceContext.getAgrosystCliConfig().getPracticedSystemFilePath();
        practicedPlotFilePath = serviceContext.getAgrosystCliConfig().getPracticedPlotFilePath();
        practicedCropCycleFilePath = serviceContext.getAgrosystCliConfig().getPracticedCropCycleFilePath();

        practicedSeasonalCropCycleFilePath = serviceContext.getAgrosystCliConfig().getPracticedSeasonalCropCycleFilePath();
        practicedCropCycleNodeFilePath = serviceContext.getAgrosystCliConfig().getPracticedCropCycleNodeFilePath();
        practicedCropCycleConnectionFilePath = serviceContext.getAgrosystCliConfig().getPracticedCropCycleConnectionFilePath();

        practicedPerennialCropCycleFilePath = serviceContext.getAgrosystCliConfig().getPracticedPerennialCropCycleFilePathFilePath();
        practicedCropCyclePhaseFilePath = serviceContext.getAgrosystCliConfig().getPracticedCropCyclePhaseFilePathFilePath();
        practicedCropCycleSpeciesFilePath = serviceContext.getAgrosystCliConfig().getEffectiveCropCycleSpeciesFilePath();

        practicedInterventionFilePath = serviceContext.getAgrosystCliConfig().getPracticedInterventionFilePath();
        practiceInterventionToolsCouplingFilePath = serviceContext.getAgrosystCliConfig().getPracticeInterventionToolsCouplingFilePath();

        effectivePerennialCropCycleFilePath = serviceContext.getAgrosystCliConfig().getEffectivePerennialCropCycleFilePath();
        effectiveCropCycleSpeciesFilePath = serviceContext.getAgrosystCliConfig().getEffectiveCropCycleSpeciesFilePath();
        effectivePhaseFilePath = serviceContext.getAgrosystCliConfig().getEffectivePhaseFilePath();

        effectiveSeasonalCropCycleFilePath = serviceContext.getAgrosystCliConfig().getEffectiveSeasonalCropCycleFilePath();
        effectiveCropCycleNodeFilePath = serviceContext.getAgrosystCliConfig().getEffectiveCropCycleNodeFilePath();
        effectiveCropCycleConnectionFilePath = serviceContext.getAgrosystCliConfig().getEffectiveCropCycleConnectionFilePath();

        effectiveInterventionFilePath = serviceContext.getAgrosystCliConfig().getEffectiveInterventionFilePath();
        effectiveSpeciesStadeFilePath = serviceContext.getAgrosystCliConfig().getEffectiveSpeciesStadeFilePath();
        effectiveToolsCouplingFilePath = serviceContext.getAgrosystCliConfig().getEffectiveToolsCouplingFilePath();
        effectiveInvolvedRuleFilePath = serviceContext.getAgrosystCliConfig().getEffectiveInvolvedRuleFilePath();

        abstractActionFilePath = serviceContext.getAgrosystCliConfig().getAbstractActionFilePath();
        seedingActionSpeciesFilePath = serviceContext.getAgrosystCliConfig().getSeedingActionSpeciesFilePath();
        harverstingYealdFilePath = serviceContext.getAgrosystCliConfig().getHarverstingYealdFilePath();
        abstractInputFilePath = serviceContext.getAgrosystCliConfig().getAbstractInputFilePath();
        fertiMinUnifaFilePath = serviceContext.getAgrosystCliConfig().getRefFertiMinUnifaFilePath();
        plotFilePath = serviceContext.getAgrosystCliConfig().getPlotFilePath();
        zoneFilePAth = serviceContext.getAgrosystCliConfig().getZoneFilePath();

        usersRolesFilePath = serviceContext.getAgrosystCliConfig().getUsersRolesFilePath();

        userAuthentication();
    }
}
