package fr.inra.agrosyst.services.pz0import.input;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2013 - 2015 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.action.AbstractAction;
import fr.inra.agrosyst.api.entities.action.AbstractInput;
import fr.inra.agrosyst.api.entities.action.BiologicalControlAction;
import fr.inra.agrosyst.api.entities.action.BiologicalProductInput;
import fr.inra.agrosyst.api.entities.action.BiologicalProductInputImpl;
import fr.inra.agrosyst.api.entities.action.MineralFertilizersSpreadingAction;
import fr.inra.agrosyst.api.entities.action.MineralProductInput;
import fr.inra.agrosyst.api.entities.action.MineralProductInputImpl;
import fr.inra.agrosyst.api.entities.action.OrganicFertilizersSpreadingAction;
import fr.inra.agrosyst.api.entities.action.OrganicProductInput;
import fr.inra.agrosyst.api.entities.action.OrganicProductInputImpl;
import fr.inra.agrosyst.api.entities.action.OrganicProductUnit;
import fr.inra.agrosyst.api.entities.action.OtherAction;
import fr.inra.agrosyst.api.entities.action.OtherProductInput;
import fr.inra.agrosyst.api.entities.action.OtherProductInputImpl;
import fr.inra.agrosyst.api.entities.action.PesticideProductInput;
import fr.inra.agrosyst.api.entities.action.PesticideProductInputImpl;
import fr.inra.agrosyst.api.entities.action.PesticidesSpreadingAction;
import fr.inra.agrosyst.api.entities.action.PhytoProductInput;
import fr.inra.agrosyst.api.entities.action.PhytoProductUnit;
import fr.inra.agrosyst.api.entities.action.SeedingAction;
import fr.inra.agrosyst.api.entities.action.SeedingProductInput;
import fr.inra.agrosyst.api.entities.action.SeedingProductInputImpl;
import fr.inra.agrosyst.api.entities.referential.RefActaTraitementsProduit;
import fr.inra.agrosyst.api.entities.referential.RefActaTraitementsProduitTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefFertiMinUNIFA;
import fr.inra.agrosyst.api.entities.referential.RefFertiMinUNIFATopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefFertiOrga;
import fr.inra.agrosyst.api.entities.referential.RefFertiOrgaTopiaDao;
import fr.inra.agrosyst.api.services.action.ActionService;
import fr.inra.agrosyst.api.services.effective.EffectiveInterventionDto;
import fr.inra.agrosyst.api.services.practiced.PracticedInterventionDto;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.services.pz0import.AbstractCSVImporter;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.InputStream;
import java.util.Map;

/**
 * Created by davidcosse on 24/02/15.
 **/
public class InputImporter extends AbstractCSVImporter {

    private static final Log log = LogFactory.getLog(InputImporter.class);

    protected ActionService actionService;

    protected RefFertiMinUNIFATopiaDao mineralProductDao;
    protected RefActaTraitementsProduitTopiaDao phytoProductDao;
    protected RefFertiOrgaTopiaDao organicProductDao;


    @Override
    public ImportResults importFromStream(InputStream is, Map<String, EntityAndDependencies> entitiesByCsvId) {
        ImportResults importResults = new ImportResults(AbstractInput.class);
        log.debug(String.format("importfromstream"));

        InputModel model = new InputModel();
        // récupère le DTO
        Import<InputImportDto> importer = Import.newImport(model, is);

        // match the first csv line number with data (not header).
        long line = FIRST_LINE_NUMBER;
        for (InputImportDto dto : importer) {
            Boolean error = false;

            // valid required fields
            AbstractInput input = getInput(dto, importResults, line, error);

            error = validAction(importResults, input, line, dto, error);

            if (!error) {
                PracticedInterventionDto practicedInterventionDto = (PracticedInterventionDto) pz0IdToObject.get(AbstractAction.class, PracticedInterventionDto.class, dto.getPz0ActionId());
                EffectiveInterventionDto effectiveInterventionDto = (EffectiveInterventionDto) pz0IdToObject.get(AbstractAction.class, EffectiveInterventionDto.class, dto.getPz0ActionId());

                if (practicedInterventionDto != null) {
                    practicedInterventionDto.addInput(input);
                    importResults.addInfoLine(line, String.format("AJOUT DE L'INTRANT '%s' VALIDÉE", input.getInputType()));
                    importResults.increaseAddedRecords();
                } else if(effectiveInterventionDto != null) {
                    effectiveInterventionDto.addInput(input);
                    importResults.addInfoLine(line, String.format("AJOUT DE L'INTRANT '%s' VALIDÉE", input.getInputType()));
                    importResults.increaseAddedRecords();
                } else {
                    importResults.increaseIgnoredRecords();
                    importResults.addErrorLine(line, "AJOUT DE L'INTRANT IGNORÉE, intervention non retrouvée");
                }
            } else {
                importResults.increaseIgnoredRecords();
                importResults.addErrorLine(line, "AJOUT DE L'INTRANT IGNORÉE");
            }
            line++;
        }
        return importResults;
    }

    protected void validPhytoProductInput(PhytoProductInput input, InputImportDto from, ImportResults importResults, long line, boolean error) {
        if (StringUtils.isNotBlank(from.getPhytoProductId())) {
            RefActaTraitementsProduit phytoProduct = phytoProductDao.forTopiaIdEquals(from.getPhytoProductId()).findUniqueOrNull();
            if (phytoProduct != null) {
                if(input != null) {
                    input.setPhytoProduct(phytoProduct);
                }
            } else {
                error = true;
                importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT '%s' IGNORÉ!, le produit avec comme identifiant %s n'a pas été retrouvée.", from.getInputTypeName(), from.getPhytoProductId()));
            }

        } else {
            error = true;
            importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT '%s' IGNORÉ!, la colonne 'phytoproduct' n'est pas renseignée.",from.getInputTypeName()));
        }

        if (StringUtils.isNotBlank(from.getPhytoProductUnitName())) {
            try {
                PhytoProductUnit phytoProductUnit = PhytoProductUnit.valueOf(from.getPhytoProductUnitName());
                if (input != null) {
                    input.setPhytoProductUnit(phytoProductUnit);
                }
            } catch (IllegalArgumentException e) {
                error = true;
                importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT '%s' IGNORÉ!, unité de produit phytosanitaire non reconnue: %s.",from.getInputTypeName(), from.getPhytoProductUnitName()));
            }
        }

    }

    protected AbstractAction validAndGetAction(String actionId, String inputTypeName, ImportResults importResults, Long line, boolean error) {
        AbstractAction action = null;
        if (StringUtils.isNotBlank(actionId)) {
            action = (AbstractAction) pz0IdToObject.get(AbstractAction.class, actionId);
            if (action == null) {
                error = true;
                importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT '%s' IGNORÉ!, l'action avec comme identifiant %s n'a pas été retrouvée.", inputTypeName, actionId));
            }
        } else {
            error = true;
            importResults.addErrorLine(line, "AJOUT DE L'INTRANT IGNORÉ!, le type d'intrant ne correspond pas à l'action associé.");
        }
        return action;
    }


    protected AbstractInput getInput(InputImportDto from, ImportResults importResults, long line, Boolean error) {
        AbstractInput input = null;
        AgrosystInterventionType type = null;
        if (StringUtils.isNotBlank(from.getInputTypeName())) {
            try {
                type = AgrosystInterventionType.valueOf(from.getInputTypeName());
                switch (type) {
                    case APPLICATION_DE_PRODUITS_FERTILISANTS_MINERAUX:
                        input = new MineralProductInputImpl();
                        break;
                    case EPANDAGES_ORGANIQUES:
                        input = new OrganicProductInputImpl();
                        break;
                    case APPLICATION_DE_PRODUITS_PHYTOSANITAIRES:
                        input = new PesticideProductInputImpl();
                        break;
                    case LUTTE_BIOLOGIQUE:
                        input = new BiologicalProductInputImpl();
                        break;
                    case SEMIS:
                        input = new SeedingProductInputImpl();
                        break;
                    case AUTRE:
                        input = new OtherProductInputImpl();
                        break;
                    default:
                        error = true;
                        importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT IGNORÉ!, type d'intrant non reconnu %s.", from.getInputTypeName()));
                }
            } catch(IllegalArgumentException e){
                error = true;
                importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT IGNORÉ!, type d'intrant non reconnu %s.", from.getInputTypeName()));
            }
        } else {
            error = true;
            importResults.addErrorLine(line, "AJOUT DE L'INTRANT IGNORÉ!, la colonne 'inputtype' n'est pas renseignée.");
        }
        if (input != null) {
            input.setInputType(type);
        }
        return input;
    }

    protected boolean validAction(ImportResults importResults, AbstractInput input, long line, InputImportDto from, Boolean error) {

        AgrosystInterventionType type = input.getInputType();

        if (type != null) {
            if(AgrosystInterventionType.APPLICATION_DE_PRODUITS_FERTILISANTS_MINERAUX.equals(type)) {

                AbstractAction action = validAndGetAction(from.getMineralFertilizersSpreadingActionId(), from.getInputTypeName(), importResults, line, error);
                ((MineralProductInput)input).setMineralFertilizersSpreadingAction((MineralFertilizersSpreadingAction) action);
                if (StringUtils.isNotBlank(from.getMineralProductId())) {
                    RefFertiMinUNIFA mineralProduct = mineralProductDao.forTopiaIdEquals(from.getMineralProductId()).findUniqueOrNull();
                    if (mineralProduct != null) {
                        ((MineralProductInput)input).setMineralProduct(mineralProduct);

                    } else {
                        error = true;
                        importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT 'MINERA'L IGNORÉ!, le produit avec comme identifiant %s n'a pas été retrouvé.", from.getMineralProductId()));
                    }
                } else {
                    error = true;
                    importResults.addErrorLine(line, "AJOUT DE L'INTRANT 'MINERAL' IGNORÉ!, la colonne 'mineralproduct' n'est pas renseignée.");
                }

            } else if (AgrosystInterventionType.AUTRE.equals(type)) {
                AbstractAction action = validAndGetAction(from.getOtherActionId(), from.getInputTypeName(), importResults, line, error);
                ((OtherProductInput)input).setOtherAction((OtherAction) action);
            } else if (AgrosystInterventionType.APPLICATION_DE_PRODUITS_PHYTOSANITAIRES.equals(type)) {
                AbstractAction action = validAndGetAction(from.getPesticidesSpreadingActionId(), from.getInputTypeName(), importResults, line, error);
                ((PesticideProductInput)input).setPesticidesSpreadingAction((PesticidesSpreadingAction) action);

                validPhytoProductInput((PesticideProductInput) input, from, importResults, line, error);

            } else if (AgrosystInterventionType.SEMIS.equals(type)) {
                AbstractAction action = validAndGetAction(from.getSeedingActionId(), from.getInputTypeName(), importResults, line, error);
                ((SeedingProductInput)input).setSeedingAction((SeedingAction) action);

                validPhytoProductInput((SeedingProductInput) input, from, importResults, line, error);
            } else if (AgrosystInterventionType.LUTTE_BIOLOGIQUE.equals(type)) {
                AbstractAction action = validAndGetAction(from.getBiologicalControlActionId(), from.getInputTypeName(), importResults, line, error);
                ((BiologicalProductInput)input).setBiologicalControlAction((BiologicalControlAction) action);

                validPhytoProductInput((BiologicalProductInput) input, from, importResults, line, error);
            } else if (AgrosystInterventionType.EPANDAGES_ORGANIQUES.equals(type)) {
                AbstractAction action = validAndGetAction(from.getOrganicFertilizersSpreadingActionId(), from.getInputTypeName(), importResults, line, error);
                ((OrganicProductInput)input).setOrganicFertilizersSpreadingAction((OrganicFertilizersSpreadingAction) action);

                if (StringUtils.isNotBlank(from.getOrganicProductId())) {
                    RefFertiOrga product = organicProductDao.forTopiaIdEquals(from.getOrganicProductId()).findUniqueOrNull();
                    if (product != null) {
                        ((OrganicProductInput)(input)).setOrganicProduct(product);
                    } else {
                        error = true;
                        importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT '%s' IGNORÉ!, le produit avec comme identifiant %s n'a pas été retrouvée.", from.getInputTypeName(), from.getOrganicProductId()));
                    }
                } else {
                    error = true;
                    importResults.addErrorLine(line, "AJOUT DE L'INTRANT '%s' IGNORÉ!, la colonne 'organicproduct' n'est pas renseignée.");
                }

                if (StringUtils.isNotBlank(from.getOrganicProductUnitName())) {
                    try {
                        OrganicProductUnit organicProductUnit = OrganicProductUnit.valueOf(from.getOrganicProductUnitName());
                        ((OrganicProductInput)input).setOrganicProductUnit(organicProductUnit);
                    } catch (IllegalArgumentException e) {
                        error = true;
                        importResults.addErrorLine(line, String.format("AJOUT DE L'INTRANT '%s' IGNORÉ!, unité de produit organique non reconnues %s.", from.getInputTypeName(), from.getOrganicProductUnitName()));
                    }
                }
            }
        }


        if (!error) {
            Binder<InputImportDto, AbstractInput> binder = BinderFactory.newBinder(InputImportDto.class, AbstractInput.class);
            binder.copy(from, input);
        }

        return error;
    }


    public void init(ServiceContext serviceContext) {
        super.init(serviceContext);
        mineralProductDao = getPersistenceContext().getRefFertiMinUNIFADao();
        phytoProductDao = getPersistenceContext().getRefActaTraitementsProduitDao();
        organicProductDao = getPersistenceContext().getRefFertiOrgaDao();
    }

}
