package fr.inra.agrosyst.services.pz0import.domain;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: DomainImporter.java 4806 2015-02-18 08:55:30Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.4.7/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/domain/DomainImporter.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Strings;

import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.DomainTopiaDao;
import fr.inra.agrosyst.api.entities.DomainType;
import fr.inra.agrosyst.api.entities.WeatherStation;
import fr.inra.agrosyst.api.entities.WeatherStationTopiaDao;
import fr.inra.agrosyst.api.entities.Zoning;
import fr.inra.agrosyst.api.entities.referential.RefLegalStatus;
import fr.inra.agrosyst.api.entities.referential.RefLegalStatusTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefLocation;
import fr.inra.agrosyst.api.entities.referential.RefLocationTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefOTEX;
import fr.inra.agrosyst.api.entities.referential.RefOTEXTopiaDao;
import fr.inra.agrosyst.api.services.domain.DomainService;
import fr.inra.agrosyst.api.services.pz0.domains.DomainAndDependencies;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.services.pz0import.AbstractCSVImporter;

import java.io.InputStream;
import java.util.Map;

import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.services.pz0import.domain.dto.DomainImportDto;
import fr.inra.agrosyst.services.pz0import.domain.model.DomainImportModel;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

/**
 *
 * @author schellen, eancelet
 * 
 * TODO : si besoin ajouter :
 * * * * * * * l'import de speciestoarea
 * 
 */

public class DomainImporter extends AbstractCSVImporter {

    private static final Log log = LogFactory.getLog(DomainImporter.class);

    protected DomainService domainService;
    protected DomainTopiaDao domainDao;
    protected RefLocationTopiaDao locationDao;
    protected RefLegalStatusTopiaDao legalStatusDao;
    protected WeatherStationTopiaDao weatherStationDao;
    protected RefOTEXTopiaDao refOtexDao;


    @Override
    public ImportResults importFromStream(InputStream is,  Map<String, EntityAndDependencies> entitiesByCsvIds) {

        ImportResults importResults = new ImportResults(Domain.class);

        DomainImportModel model = new DomainImportModel();
        Import<DomainImportDto> importer = Import.newImport(model, is);

        // match the csv line number.
        Long line = FIRST_LINE_NUMBER;
        for (DomainImportDto dto : importer) {

            boolean error = false;
            Domain domain = domainService.newDomain();
            Binder<DomainImportDto, Domain> domainBinder = BinderFactory.newBinder(DomainImportDto.class, Domain.class);
            domainBinder.copyExcluding(dto, domain,
                    Domain.PROPERTY_TOPIA_ID
            );

            domain.setCode(csvCodeToAgrosystCode(Domain.class, dto.getCode()));

            // required data
            error = validDomainCode(importResults, line, domain, error);
            error = validateCampaign(domain.getCampaign(), importResults, domain, line, error);
            error = validateMainContact(domain.getMainContact(), importResults, domain, line, error);
            error = validateDomainName(domain.getName(), importResults, line, error);
            error = validLocation(importResults, line, dto, error);
            error = validDomainType(importResults, line, domain, dto.getTypeName(), error);

            error = zoningNameToDomainZone(importResults, line, domain, dto, error);
            error = getLegalStatusId(importResults, line, dto, domain, error);
            error = otex18idToDomainOtex18(importResults, line, dto, domain, error);
            error = otex70IdToDomainOtex70(importResults, line, dto, domain, error);
            error = weatherStationIdToDomainWeatherStation(importResults, line, dto, domain, error);

            if (!error) {
                String legalStatusId = domain.getLegalStatus() != null ? domain.getLegalStatus().getTopiaId() : null;
                Integer otex18 = domain.getOtex18() != null ? domain.getOtex18().getCode_OTEX_18_postes() : null;
                Integer otex70 = domain.getOtex70() != null ? domain.getOtex70().getCode_OTEX_70_postes() : null;
                DomainAndDependencies domainAndDependencies = new DomainAndDependencies(domain, dto.getLocationId(), legalStatusId, otex18, otex70, line);
                importResults.addEntity(dto.getId(), domainAndDependencies);
                importResults.addInfoLine(line, "DOMAINE VALIDÉ, csvid: " + dto.getId());
                pz0IdToObject.put(Domain.class,  dto.getId(), domainAndDependencies);
                importResults.increaseAddedRecords();
            } else {
                importResults.increaseIgnoredRecords();
                importResults.addErrorLine(line, "DOMAINE IGNORÉ csvid:" + dto.getId());
            }

            line++;
        }
        return  importResults;
    }

    protected boolean weatherStationIdToDomainWeatherStation(ImportResults result, long line, DomainImportDto dto, Domain domain, boolean error) {
        if (StringUtils.isNotBlank(dto.getDefaultWeatherStationId())) {
            WeatherStation weatherStation = weatherStationDao.forTopiaIdEquals(dto.getDefaultWeatherStationId()).findUniqueOrNull();
            domain.setDefaultWeatherStation(weatherStation);
            if (weatherStation == null) {
                error = true;
                result.addErrorLine(line, String.format("DOMAINE IGNORÉ! La station météo n'a pas été retrouvée pour domaine %s\" sur la campagne %s!", dto.getName(), dto.getCampaign()));
            }
        }
        return error;
    }

    protected boolean otex70IdToDomainOtex70(ImportResults result, long line, DomainImportDto dto, Domain domain, boolean error) {
        if (!Strings.isNullOrEmpty(dto.getRefOtex70Id())) {
            RefOTEX refOtex70 = refOtexDao.forTopiaIdEquals(dto.getRefOtex70Id()).findUniqueOrNull();
            if (refOtex70 == null) {
                error = true;
                result.addErrorLine(line, String.format("DOMAINE IGNORÉ! le Code OTEX70 de topiaId %s n'existe pas en base de données pour le domaine %s!", dto.getOtex70(), dto.getName()));
            }
            else {
                domain.setOtex70(refOtex70);
            }
        }
        return error;
    }

    protected boolean otex18idToDomainOtex18(ImportResults result, long line, DomainImportDto dto, Domain domain, boolean error) {
        if (!Strings.isNullOrEmpty(dto.getRefOtex18Id())) {
            RefOTEX refOtex18 = refOtexDao.forTopiaIdEquals(dto.getRefOtex18Id()).findUniqueOrNull();
            if (refOtex18 == null) {
                error = true;
                result.addInfoLine(line, String.format("DOMAINE IGNORÉ! le Code OTEX18 de topiaId %s n'existe pas en base de données pour le domaine %s!", dto.getOtex18(), dto.getName()));
            } else {
                domain.setOtex18(refOtex18);
            }
        }
        return error;
    }

    protected boolean getLegalStatusId(ImportResults result, long line, DomainImportDto dto, Domain domain, boolean error) {
        // RefLegalStatus
        if (!Strings.isNullOrEmpty(dto.getLegalStatusId())) {
            RefLegalStatus legalStatus = legalStatusDao.forTopiaIdEquals(dto.getLegalStatusId()).findUniqueOrNull();
            if (legalStatus == null)  {
                error = true;
                result.addInfoLine(line, String.format("ATTENTION! le refLegalStatus de topiaId %s n'existe pas en base de données pour le domaine %s!", dto.getLegalStatus(), dto.getName()));
            }
            else {
                domain.setLegalStatus(legalStatus);
            }
        }
        return error;
    }

    protected boolean zoningNameToDomainZone(ImportResults result, long line, Domain domain, DomainImportDto dto, boolean error) {
        String zoningName = dto.getZoningName();
        if (!Strings.isNullOrEmpty(zoningName)) {
            Zoning zoning;
            try {
                zoning = Zoning.valueOf(zoningName);
                domain.setZoning(zoning);
            } catch (IllegalArgumentException ex) {
                error = true;
                result.addInfoLine(line, "DOMAINE IGNORÉ! Zoning non supportée : " + zoningName);
                result.addErrorLine(line, String.format("COMBINAISON D'OUTILS IGNORÉE ! Unité de rendement non supportée, %s!", zoningName));
            }
        }
        return error;
    }

    protected boolean validDomainCode(ImportResults importResults, long line, Domain domain, boolean error) {
        if (StringUtils.isBlank(domain.getCode())) {
            importResults.addErrorLine(line, String.format("DOMAINE IGNORÉ! La colonne 'code' n'est pas renseignée pour le domaine %s!", domain.getName()));
            error = true;
        }
        return error;
    }

    protected boolean validateCampaign(int campaign, ImportResults results, Domain domain, long line, boolean error) {
        if (campaign == 0) {
            results.addErrorLine(line, String.format("DOMAINE IGNORÉ! Campagne non valide pour le domaine %s!", domain.getName()));
            error = true;
        }
        return error;
    }

    protected boolean validateMainContact(String mainContact, ImportResults results, Domain domain, long line, boolean error) {
        if (StringUtils.isBlank(mainContact)){
            results.addErrorLine(line, String.format("DOMAINE IGNORÉ! Aucun nom de contact principale de spécifié pour le domaine %s!", domain.getName()));
            error = true;
        }
        return error;
    }

    protected boolean validateDomainName(String domainName, ImportResults results, long line, boolean error) {
        if (StringUtils.isBlank(domainName)){
            results.addErrorLine(line, "DOMAINE IGNORÉ! La colonne 'name' n'est pas renseignée!");
            error = true;
        }
        return error;
    }

    protected boolean validLocation(ImportResults result, long line, DomainImportDto dto, boolean error) {
        if (StringUtils.isNotBlank(dto.getLocationId())) {
            RefLocation location = locationDao.forTopiaIdEquals(dto.getLocationId()).findUniqueOrNull();
            if (location == null) {
                result.addInfoLine(line, String.format("DOMAINE IGNORÉ! L'objet RefLocation de topiaId %s n'existe pas en base de données pour le domaine %s.", dto.getLocationId(), dto.getName()));
                error = true;
            }
        } else {
            result.addErrorLine(line, String.format("DOMAINE IGNORÉ! La colonne 'reflocation' n'est pas renseignée pour le domaine %s!", dto.getName()));
            error = true;
        }
        return error;
    }

    protected boolean validDomainType(ImportResults result, long line, Domain domain, String typeName, boolean error) {
        if (StringUtils.isNotBlank(typeName)) {
            try {
                domain.setType(DomainType.valueOf(typeName));
            } catch(IllegalArgumentException e){
                result.addErrorLine(line, String.format("DOMAINE IGNORÉ! le type de domaine n'est pas reconnu %s!", typeName));
            }
        } else {
            result.addErrorLine(line, String.format("DOMAINE IGNORÉ! La colonne 'type' n'est pas renseignée pour le domaine %s!", domain.getName()));
            error = true;
        }
        return error;
    }

    @Override
    public void init(ServiceContext serviceContext) {
        super.init(serviceContext);
        domainDao = getPersistenceContext().getDomainDao();
        locationDao = getPersistenceContext().getRefLocationDao();
        legalStatusDao = getPersistenceContext().getRefLegalStatusDao();
        weatherStationDao = getPersistenceContext().getWeatherStationDao();
        refOtexDao = getPersistenceContext().getRefOTEXDao();
        domainService = getServiceFactory().newService(DomainService.class);
    }

}
