package fr.inra.agrosyst.services.pz0import.action;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: ActionImporter.java 4789 2015-02-12 16:58:15Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/1.4.6/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/action/ActionImporter.java $
 * %%
 * Copyright (C) 2013 - 2015 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import fr.inra.agrosyst.api.entities.action.AbstractAction;
import fr.inra.agrosyst.api.entities.action.BiologicalControlAction;
import fr.inra.agrosyst.api.entities.action.BiologicalControlActionImpl;
import fr.inra.agrosyst.api.entities.action.CarriageAction;
import fr.inra.agrosyst.api.entities.action.CarriageActionImpl;
import fr.inra.agrosyst.api.entities.action.HarvestingAction;
import fr.inra.agrosyst.api.entities.action.HarvestingActionImpl;
import fr.inra.agrosyst.api.entities.action.IrrigationAction;
import fr.inra.agrosyst.api.entities.action.IrrigationActionImpl;
import fr.inra.agrosyst.api.entities.action.MaintenancePruningVinesAction;
import fr.inra.agrosyst.api.entities.action.MaintenancePruningVinesActionImpl;
import fr.inra.agrosyst.api.entities.action.MineralFertilizersSpreadingAction;
import fr.inra.agrosyst.api.entities.action.MineralFertilizersSpreadingActionImpl;
import fr.inra.agrosyst.api.entities.action.OrganicFertilizersSpreadingAction;
import fr.inra.agrosyst.api.entities.action.OrganicFertilizersSpreadingActionImpl;
import fr.inra.agrosyst.api.entities.action.OtherActionImpl;
import fr.inra.agrosyst.api.entities.action.PesticidesSpreadingAction;
import fr.inra.agrosyst.api.entities.action.PesticidesSpreadingActionImpl;
import fr.inra.agrosyst.api.entities.action.SeedType;
import fr.inra.agrosyst.api.entities.action.SeedingAction;
import fr.inra.agrosyst.api.entities.action.SeedingActionImpl;
import fr.inra.agrosyst.api.entities.action.TillageAction;
import fr.inra.agrosyst.api.entities.action.TillageActionImpl;
import fr.inra.agrosyst.api.entities.action.YealdUnit;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCycle;
import fr.inra.agrosyst.api.entities.referential.RefInterventionAgrosystTravailEDI;
import fr.inra.agrosyst.api.entities.referential.RefInterventionAgrosystTravailEDITopiaDao;
import fr.inra.agrosyst.api.services.action.ActionService;
import fr.inra.agrosyst.api.services.effective.EffectiveInterventionDto;
import fr.inra.agrosyst.api.services.practiced.PracticedInterventionDto;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.services.pz0import.AbstractCSVImporter;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;

import java.io.InputStream;
import java.util.Map;

/**
 * Created by davidcosse on 09/02/15.
 */
public class ActionImporter extends AbstractCSVImporter {

    private static final Log log = LogFactory.getLog(ActionImporter.class);

    protected ActionService actionService;

    protected RefInterventionAgrosystTravailEDITopiaDao refInterventionAgrosystTravailEDIDao;

    @Override
    public ImportResults importFromStream(InputStream is, Map<String, EntityAndDependencies> entitiesByCsvId) {
        ImportResults importResults = new ImportResults(ActionImportDto.class);
        log.debug(String.format("importfromstream"));

        ActionModel model = new ActionModel();
        // récupère le DTO
        Import<ActionImportDto> importer = Import.newImport(model, is);

        // match the first csv line number with data (not header).
        long line = FIRST_LINE_NUMBER;
        for (ActionImportDto dto : importer) {
            boolean error = false;

            // valid required fields
            error = validIntervention(importResults, line, dto, error);

            PracticedInterventionDto practicedInterventionDto = null;
            EffectiveInterventionDto effectiveInterventionDto = null;
            if (!error) {
                practicedInterventionDto = getPracticedIntervention(dto);
                effectiveInterventionDto = getEffectiveIntervention(dto);
            }

            AbstractAction abstractAction = getEffectiveAction(dto);
            if (abstractAction != null) {
                error = validToolsCouplingCode(importResults, line, dto, abstractAction, practicedInterventionDto, effectiveInterventionDto, error);

                validMainAction(importResults, line, dto, abstractAction, error);

                validSpecificActionData(importResults, line, dto, abstractAction, error);
            } else {
                error = true;
                importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION IGNORÉE!, Le discriminator %s n'est pas valide.", dto.getTopiaDiscriminator()));
            }


            if (!error) {
                if (practicedInterventionDto != null) {
                    practicedInterventionDto.addAction(abstractAction);
                } else {
                    effectiveInterventionDto.addAction(abstractAction);
                }
                pz0IdToObject.put(AbstractAction.class, dto.getId(), abstractAction);
                importResults.addInfoLine(line, "AJOUT DE L'ACTION VALIDÉE");
                importResults.increaseAddedRecords();
            } else {
                importResults.increaseIgnoredRecords();
                importResults.addErrorLine(line, "AJOUT DE L'ACTION IGNORÉE");
            }
            line++;
        }
        return importResults;
    }

    protected AbstractAction getEffectiveAction(ActionImportDto dto) {
        AbstractAction action = null;
        String discriminator = dto.getTopiaDiscriminator();
        String actionType = StringUtils.remove(discriminator, "fr.inra.agrosyst.api.entities.action.");

        if (actionType.equals("MineralFertilizersSpreadingActionImpl")) {
            action = new MineralFertilizersSpreadingActionImpl();
        } else if(actionType.equals("PesticidesSpreadingActionImpl")) {
            action = new PesticidesSpreadingActionImpl();
        } else if(actionType.equals("OtherActionImpl")) {
            action = new OtherActionImpl();
        } else if(actionType.equals("MaintenancePruningVinesActionImpl")) {
            action = new MaintenancePruningVinesActionImpl();
        } else if(actionType.equals("OrganicFertilizersSpreadingActionImpl")) {
            action = new OrganicFertilizersSpreadingActionImpl();
        } else if(actionType.equals("IrrigationActionImpl")) {
            action = new IrrigationActionImpl();
        } else if(actionType.equals("BiologicalControlActionImpl")) {
            action = new BiologicalControlActionImpl();
        } else if(actionType.equals("HarvestingActionImpl")) {
            action = new HarvestingActionImpl();
        } else if(actionType.equals("SeedingActionImpl")) {
            action = new SeedingActionImpl();
        } else if(actionType.equals("CarriageActionImpl")) {
            action = new CarriageActionImpl();
        } else if(actionType.equals("TillageActionImpl")) {
            action = new TillageActionImpl();
        }

        return action;
    }

    protected boolean validIntervention(ImportResults importResults, long line, ActionImportDto from, boolean error) {
        String practicedInterventionId = from.getPracticedInterventionId();
        String effectiveInterventionId = from.getEffectiveInterventionId();

        if (StringUtils.isNotBlank(practicedInterventionId)) {
            PracticedInterventionDto interventionDto = (PracticedInterventionDto)pz0IdToObject.get(PracticedInterventionDto.class, practicedInterventionId);
            if (interventionDto == null) {
                error = true;
                importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION IGNORÉE!, L'intervention synthétisé %s n'a pas été retrouvé.", practicedInterventionId));
            }
        } else if (StringUtils.isNotBlank(effectiveInterventionId)) {
            // todo import effective is not implemented yet
        } else {
            error = true;
            importResults.addErrorLine(line, "AJOUT DE L'ACTION IGNORÉE!, La colonne 'practicedintervention' ou 'effectiveintervention' doit être renseignée.");
        }

        return error;
    }

    protected EffectiveInterventionDto getEffectiveIntervention(ActionImportDto from) {
        EffectiveInterventionDto interventionDto = null;
        String effectiveInterventionId = from.getEffectiveInterventionId();
        interventionDto = (EffectiveInterventionDto)pz0IdToObject.get(EffectiveInterventionDto.class, effectiveInterventionId);
        return interventionDto;
    }

    protected PracticedInterventionDto getPracticedIntervention(ActionImportDto from) {
        PracticedInterventionDto interventionDto = null;
        String practicedInterventionId = from.getPracticedInterventionId();
        interventionDto = (PracticedInterventionDto)pz0IdToObject.get(PracticedInterventionDto.class, practicedInterventionId);
        return interventionDto;
    }


    protected boolean validToolsCouplingCode(ImportResults importResults,
                                             long line,
                                             ActionImportDto from,
                                             AbstractAction to,
                                             PracticedInterventionDto practicedInterventionDto,
                                             EffectiveInterventionDto effectiveInterventionDto,
                                             boolean error) {

        String toolsCouplingCode = from.getToolsCouplingCode();
        if (StringUtils.isNotBlank(toolsCouplingCode)){
            to.setToolsCouplingCode(toolsCouplingCode);
            if (practicedInterventionDto != null) {
                if (practicedInterventionDto.getToolsCouplingCodes() == null || !practicedInterventionDto.getToolsCouplingCodes().contains(toolsCouplingCode)) {
                    error = true;
                    importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION IGNORÉE!, La combinaison d'outils %s n'a pas été retrouvée.", toolsCouplingCode));
                }
            } else if (effectiveInterventionDto != null) {
                if (effectiveInterventionDto.getToolsCouplingCodes() == null || !effectiveInterventionDto.getToolsCouplingCodes().contains(toolsCouplingCode)) {
                    error = true;
                    importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION IGNORÉE!, La combinaison d'outils %s n'a pas été retrouvée.", toolsCouplingCode));
                }
            }
        }

        return error;
    }

    protected boolean validMainAction(ImportResults importResults,
                                      long line,
                                      ActionImportDto from,
                                      AbstractAction abstractAction,
                                      boolean error) {

        String mainActionId = from.getMainActionId();
        if (StringUtils.isNotBlank(mainActionId)) {
            RefInterventionAgrosystTravailEDI interventionAgrosystTravailEDI = refInterventionAgrosystTravailEDIDao.forTopiaIdEquals(mainActionId).findUniqueOrNull();
            if(interventionAgrosystTravailEDI != null) {
                abstractAction.setMainAction(interventionAgrosystTravailEDI);
            } else {
                error = true;
                importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION IGNORÉE!, La l'action principale avec comme identifiant %s n'a pas été trouvée.", mainActionId));
            }

        } else {
            error = true;
            importResults.addErrorLine(line, "AJOUT DE L'ACTION IGNORÉE!, La colonne 'mainaction' doit être renseignée.");
        }
        return error;
    }

    protected boolean validSpecificActionData(ImportResults importResults,
                                              long line,
                                              ActionImportDto from,
                                              AbstractAction to,
                                              boolean error) {
        if (to instanceof TillageAction) {
            // nothing to validate
            ((TillageAction) to).setTillageDepth(from.getTillageDepth());
            ((TillageAction) to).setOtherSettingTool(from.getOtherSettingTool());
        } else if (to instanceof MineralFertilizersSpreadingAction) {
            if (from.getBurial() == null) {
                error = true;
                importResults.addErrorLine(line, "AJOUT DE L'ACTION IGNORÉE!, la colonne 'burial' n'est pas renseignée pour l'action application de produits fertilisants minéreaux.");
            } else {
                ((MineralFertilizersSpreadingAction) to).setBurial(from.getBurial());
            }
            if (from.getLocalizedSpreading() == null) {
                error = true;
                importResults.addErrorLine(line, "AJOUT DE L'ACTION IGNORÉE!, la colonne 'localizedspreading' n'est pas renseignée pour l'action application de produits fertilisants minéreaux.");
            } else {
                ((MineralFertilizersSpreadingAction) to).setLocalizedSpreading(from.getLocalizedSpreading());
            }
        } else if (to instanceof OrganicFertilizersSpreadingAction) {
            if (from.getLandFilledWaste() == null) {
                error = true;
                importResults.addErrorLine(line, "AJOUT DE L'ACTION IGNORÉE!, la colonne 'landfilledwaste' n'est pas renseignée pour l'action épandage organique.");
            } else {
                ((OrganicFertilizersSpreadingAction) to).setLandfilledWaste(from.getLandFilledWaste());
            }
        } else if (to instanceof BiologicalControlAction) {
            ((BiologicalControlAction) to).setBoiledQuantityPerTrip(from.getBoiledQuantityPerTrip());
            ((BiologicalControlAction) to).setTripFrequency(from.getTripFrequency());
            if (from.getBoiledQuantity() == null) {
                error = true;
                importResults.addErrorLine(line, "AJOUT DE L'ACTION IGNORÉE!, la colonne 'boiledquantity' n'est pas renseignée pour l'action de lutte biologique.");
            } else {
                ((BiologicalControlAction) to).setBoiledQuantity(from.getBoiledQuantity());
            }
            if (from.getProportionOfTreatedSurface() == null) {
                error = true;
                importResults.addErrorLine(line, "AJOUT DE L'ACTION IGNORÉE!, la colonne 'proportionoftreatedsurface' n'est pas renseignée pour l'action de lutte biologique.");
            } else {
                ((BiologicalControlAction) to).setProportionOfTreatedSurface(from.getProportionOfTreatedSurface());
            }
        } else if (to instanceof PesticidesSpreadingAction) {
            ((PesticidesSpreadingAction) to).setBoiledQuantity(from.getBoiledQuantity());
            ((PesticidesSpreadingAction) to).setBoiledQuantityPerTrip(from.getBoiledQuantityPerTrip());
            ((PesticidesSpreadingAction) to).setTripFrequency(from.getTripFrequency());
            if (from.getAntiDriftNozzle() == null) {
                error = true;
                importResults.addErrorLine(line, "AJOUT DE L'ACTION IGNORÉE!, la colonne 'antidriftnozzle' n'est pas renseignée pour l'action d'application de produits phytosanitaires.");
            } else {
                ((PesticidesSpreadingAction) to).setAntiDriftNozzle(from.getAntiDriftNozzle());
            }
            if (from.getProportionOfTreatedSurface() == null) {
                error = true;
                importResults.addErrorLine(line, "AJOUT DE L'ACTION IGNORÉE!, la colonne 'proportionoftreatedsurface' n'est pas renseignée pour l'action d'application de produits phytosanitaires.");
            } else {
                ((PesticidesSpreadingAction) to).setProportionOfTreatedSurface(from.getProportionOfTreatedSurface());
            }
        } else if (to instanceof SeedingAction) {
            ((SeedingAction) to).setYealdTarget(from.getYealdTarget());
            try {
                if (StringUtils.isNotBlank(from.getYealdUnitName())) {
                    ((SeedingAction) to).setYealdUnit(YealdUnit.valueOf(from.getYealdUnitName()));
                }
            } catch (IllegalArgumentException e) {
                error = true;
                importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION IGNORÉE!, unité de rendement non reconnue: %s pour l'action de type semis.", from.getYealdUnitName()));
            }
            try {
                if (StringUtils.isNotBlank(from.getSeedTypeName())) {
                    ((SeedingAction) to).setSeedType(SeedType.valueOf(from.getSeedTypeName()));
                }
            } catch (IllegalArgumentException e) {
                error = true;
                importResults.addErrorLine(line, String.format("AJOUT DE L'ACTION IGNORÉE!, le type de semis n'est pas reconnue: %s pour l'action de type semis.", from.getSeedTypeName()));
            }
        }  else if (to instanceof CarriageAction) {
            ((CarriageAction) to).setLoadCapacity(from.getLoadCapacity());
            ((CarriageAction) to).setTripFrequency(from.getTripFrequency());
        } else if (to instanceof HarvestingAction){
            ((HarvestingAction) to).setMoisturePercent(from.getMoisturePercent());

            if (from.getExportedWaste() == null) {
                error = true;
                importResults.addErrorLine(line, "AJOUT DE L'ACTION IGNORÉE!, la colonne 'exportedwaste' n'est pas renseignée pour l'action de récolte.");
            } else {
                ((HarvestingAction) to).setExportedWaste(from.getExportedWaste());
            }
        }  else if (to instanceof IrrigationAction){
            ((IrrigationAction) to).setWaterQuantityMax(from.getWaterQuantityMin());
            ((IrrigationAction) to).setWaterQuantityMax(from.getWaterQuantityMax());
            ((IrrigationAction) to).setWaterQuantityMedian(from.getWaterQuantityMedian());
            ((IrrigationAction) to).setAzoteQuantity(from.getAzoteQuantity());
            if (from.getWaterQuantityAverage() == null) {
                error = true;
                importResults.addErrorLine(line, "AJOUT DE L'ACTION IGNORÉE!, la colonne 'waterquantityaverage' n'est pas renseignée pour l'action d'irrigation.");
            } else {
                ((IrrigationAction) to).setWaterQuantityAverage(from.getWaterQuantityAverage());
            }
        }
        return error;
    }

    public void init(ServiceContext serviceContext) {
        super.init(serviceContext);
        actionService = getServiceFactory().newService(ActionService.class);
        refInterventionAgrosystTravailEDIDao = getPersistenceContext().getRefInterventionAgrosystTravailEDIDao();
    }

}
