package fr.inra.agrosyst.services.pz0import.practicedPlot;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: PracticedPlotImporter.java 4771 2015-02-06 09:45:33Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.4.5/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/practicedPlot/PracticedPlotImporter.java $
 * %%
 * Copyright (C) 2013 - 2015 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Strings;
import fr.inra.agrosyst.api.entities.BufferStrip;
import fr.inra.agrosyst.api.entities.FrostProtectionType;
import fr.inra.agrosyst.api.entities.HosesPositionning;
import fr.inra.agrosyst.api.entities.IrrigationSystemType;
import fr.inra.agrosyst.api.entities.MaxSlope;
import fr.inra.agrosyst.api.entities.PompEngineType;
import fr.inra.agrosyst.api.entities.SolWaterPh;
import fr.inra.agrosyst.api.entities.WaterFlowDistance;
import fr.inra.agrosyst.api.entities.practiced.PracticedPlot;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystem;
import fr.inra.agrosyst.api.entities.referential.RefLocation;
import fr.inra.agrosyst.api.entities.referential.RefLocationTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefSolProfondeurIndigo;
import fr.inra.agrosyst.api.entities.referential.RefSolProfondeurIndigoTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefSolTextureGeppa;
import fr.inra.agrosyst.api.entities.referential.RefSolTextureGeppaTopiaDao;
import fr.inra.agrosyst.api.services.practiced.PracticedPlotService;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.practicedPlot.PracticedPlotAndDependencies;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.services.pz0import.AbstractCSVImporter;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.InputStream;
import java.util.Map;

/**
 * Created by davidcosse on 29/01/15.
 */
public class PracticedPlotImporter extends AbstractCSVImporter {

    protected PracticedPlotService practicedPlotService;
    protected RefLocationTopiaDao locationDao;
    protected RefSolTextureGeppaTopiaDao refSolTextureGeppaDao;
    protected RefSolProfondeurIndigoTopiaDao refSolProfondeurIndigoDao;

    private static final Log log = LogFactory.getLog(PracticedPlotImporter.class);

    @Override
    public ImportResults importFromStream(InputStream is, Map<String, EntityAndDependencies> practicedSystemsByCsvId) {
        ImportResults importResults = new ImportResults(PracticedSystem.class);
        log.debug(String.format("importfromstream"));

        PracticedPlotImportModel model = new PracticedPlotImportModel();
        // récupère le DTO
        Import<PracticedPlotImportDto> importer = Import.newImport(model, is);

        // match the first csv line number with data (not header).
        long line = FIRST_LINE_NUMBER;
        for (PracticedPlotImportDto dto : importer) {
            boolean error = false;
            PracticedPlot practicedPlot = practicedPlotService.getPracticedPlot(null);

            Binder<PracticedPlotImportDto, PracticedPlot> practicedPlotBinder = BinderFactory.newBinder(PracticedPlotImportDto.class, PracticedPlot.class);
            practicedPlotBinder.copyExcluding(dto, practicedPlot,
                    PracticedPlot.PROPERTY_TOPIA_ID,
                    PracticedPlot.PROPERTY_PRACTICED_SYSTEM,
                    PracticedPlot.PROPERTY_LOCATION,
                    PracticedPlot.PROPERTY_SURFACE_TEXTURE,
                    PracticedPlot.PROPERTY_SUB_SOIL_TEXTURE,
                    PracticedPlot.PROPERTY_SOL_DEPTH,
                    PracticedPlot.PROPERTY_WATER_FLOW_DISTANCE,
                    PracticedPlot.PROPERTY_IRRIGATION_SYSTEM_TYPE,
                    PracticedPlot.PROPERTY_POMP_ENGINE_TYPE,
                    PracticedPlot.PROPERTY_HAIL_PROTECTION,
                    PracticedPlot.PROPERTY_FROST_PROTECTION_TYPE,
                    PracticedPlot.PROPERTY_SOL_WATER_PH,
                    PracticedPlot.PROPERTY_MAX_SLOPE,
                    PracticedPlot.PROPERTY_BUFFER_STRIP);

            // valid required fields
            error = validPracticedPlotName(importResults, line, practicedPlot, error);
            error = validPracticedSystemAffiliation(importResults, line, dto, practicedPlot, practicedSystemsByCsvId, error);
            error = validWaterFlowDistance(importResults, line, dto, practicedPlot, error);
            error = validBufferStrip(importResults, line, dto, practicedPlot, error);
            error = validMaxSlope(importResults, line, dto, practicedPlot, error);

            // valid related entities existence
            error = validLocation(importResults, line, dto, practicedPlot, error);
            error = validSurfaceTexture(importResults, line, dto, practicedPlot, error);
            error = validSubSoilTexture(importResults, line, dto, practicedPlot, error);
            error = validSolDepth(importResults, line, dto, practicedPlot, error);

            // valid enums
            error = validIrrigationSystemType(importResults, line, dto, practicedPlot, error);
            error = validPompEngineType(importResults, line, dto, practicedPlot, error);
            error = validHosesPositionning(importResults, line, dto, practicedPlot, error);
            error = validFrostProtectionType(importResults, line, dto, practicedPlot, error);
            error = validSolWaterPh(importResults, line, dto, practicedPlot, error);

            //saveGrowingSytem(importResults, line, dto, growingSystem, refTypeAgricultureTopiaId);
            if (!error) {
                PracticedPlotAndDependencies practicedPlotAndDependencies = new PracticedPlotAndDependencies(practicedPlot);
                importResults.addEntity(dto.getId(), practicedPlotAndDependencies);
                importResults.addInfoLine(line, "PARCELLE DU SYNTHÉTISÉ VALIDÉE, csvid: " + dto.getId());
            } else {
                importResults.increaseIgnoredRecords();
                importResults.addErrorLine(line, "PARCELLE DU SYNTHÉTISÉ IGNORÉE csvid:" + dto.getId());
            }

            line++;
        }
        return importResults;
    }

    protected boolean validPracticedPlotName(ImportResults importResults, long line, PracticedPlot practicedPlot, boolean error) {

        if (StringUtils.isBlank(practicedPlot.getName())) {
            importResults.addErrorLine(line, "PARCELLE DU SYNTHÉTISÉ IGNORÉE!, la colonne 'name' n'est pas renseignée !");
            error = true;
        }

        return error;
    }

    protected boolean validPracticedSystemAffiliation(ImportResults importResults, long line, PracticedPlotImportDto dto, PracticedPlot practicedPlot, Map<String, EntityAndDependencies> practicedSystemsByCsvIds, boolean error) {
        try {
            if (StringUtils.isNotBlank(dto.getPracticedSystemId())) {
                EntityAndDependencies entityAndDependencies = practicedSystemsByCsvIds.get(dto.getPracticedSystemId());
                if (entityAndDependencies != null) {
                    practicedPlot.setPracticedSystem((PracticedSystem) entityAndDependencies.getEntity());
                } else {
                    importResults.addErrorLine(line, String.format("PARCELLE DU SYNTHÉTISÉ IGNORÉE!, aucun système synthétisé n'est retrouvé avec l'identifiant suivant: %s", dto.getPracticedSystemId()));
                    error = true;
                }
            } else {
                importResults.addErrorLine(line, String.format("PARCELLE DU SYNTHÉTISÉ IGNORÉE!, la colonne 'practicedSystem' n'est pas renseignée pour la parcelle  \"%s\"!", dto.getName()));
                error = true;
            }
        } catch (ClassCastException e) {
            importResults.addErrorLine(line, String.format("PARCELLE DU SYNTHÉTISÉ IGNORÉE!, système synthétisé non retrouvé pour la parcelle \"%s\"!", dto.getName()));
            error = true;
        }
        return error;
    }

    protected boolean validLocation(ImportResults result, long line, PracticedPlotImportDto dto, PracticedPlot practicedPlot, boolean error) {
        if (StringUtils.isNotBlank(dto.getLocationId())) {
            RefLocation location = locationDao.forTopiaIdEquals(dto.getLocationId()).findUniqueOrNull();
            if (location == null) {
                result.addErrorLine(line, String.format("PARCELLE DU SYNTHÉTISÉ IGNORÉE! l'objet RefLocation de topiaId %s n'existe pas en base de données pour la parcelle \"%s\".", dto.getLocationId(), dto.getName()));
                error = true;
            }
            practicedPlot.setLocation(location);
        } else {
            result.addErrorLine(line, String.format("PARCELLE DU SYNTHÉTISÉ IGNORÉE! la colonne 'reflocation' n'est pas renseignée pour la parcelle \"%s\"!", dto.getName()));
            error = true;
        }
        return error;
    }

    protected boolean validSurfaceTexture(ImportResults result, long line, PracticedPlotImportDto dto, PracticedPlot practicedPlot, boolean error) {
        if (StringUtils.isNotBlank(dto.getSurfaceTextureId())) {
            RefSolTextureGeppa surfaceTexture = refSolTextureGeppaDao.forTopiaIdEquals(dto.getSurfaceTextureId()).findUniqueOrNull();
            if (surfaceTexture == null) {
                result.addErrorLine(line, String.format("PARCELLE DU SYNTHÉTISÉ IGNORÉE! l'objet RefSolTextureGeppa de topiaId %s n'existe pas en base de données pour la parcelle \"%s\".", dto.getLocationId(), dto.getName()));
                error = true;
            }
            practicedPlot.setSurfaceTexture(surfaceTexture);
        }
        return error;
    }

    protected boolean validSubSoilTexture(ImportResults result, long line, PracticedPlotImportDto dto, PracticedPlot practicedPlot, boolean error) {
        if (StringUtils.isNotBlank(dto.getSubSoilTextureId())) {
            RefSolTextureGeppa subSolTexture = refSolTextureGeppaDao.forTopiaIdEquals(dto.getSubSoilTextureId()).findUniqueOrNull();
            if (subSolTexture == null) {
                result.addErrorLine(line, String.format("PARCELLE DU SYNTHÉTISÉ IGNORÉE! l'objet RefSolTextureGeppa de topiaId %s n'existe pas en base de données pour la parcelle \"%s\".", dto.getLocationId(), dto.getName()));
                error = true;
            }
            practicedPlot.setSubSoilTexture(subSolTexture);
        }
        return error;
    }

    protected boolean validSolDepth(ImportResults result, long line, PracticedPlotImportDto dto, PracticedPlot practicedPlot, boolean error) {
        if (StringUtils.isNotBlank(dto.getSolDepthId())) {
            RefSolProfondeurIndigo solDepth = refSolProfondeurIndigoDao.forTopiaIdEquals(dto.getSolDepthId()).findUniqueOrNull();
            if (solDepth == null) {
                result.addErrorLine(line, String.format("PARCELLE DU SYNTHÉTISÉ IGNORÉE! l'objet RefSolProfondeurIndigo de topiaId %s n'existe pas en base de données pour la parcelle \"%s\".", dto.getLocationId(), dto.getName()));
                error = true;
            }
            practicedPlot.setSolDepth(solDepth);
        }
        return error;
    }

    protected boolean validWaterFlowDistance(ImportResults result, long line, PracticedPlotImportDto dto, PracticedPlot practicedPlot, boolean error) {

        String value = dto.getWaterFlowDistanceImportedValue();
        WaterFlowDistance waterFlowDistance = null;
        if (StringUtils.isNotBlank(value)) {
            try {
                waterFlowDistance = WaterFlowDistance.valueOf(value);
            } catch (IllegalArgumentException e) {
                result.addErrorLine(line, String.format("PARCELLE DU SYNTHÉTISÉ IGNORÉE! Distance à l'eau, non supportée : %s", value));
                error = true;
            }
        } else {
            result.addErrorLine(line, String.format("PARCELLE DU SYNTHÉTISÉ IGNORÉE! la colonne 'waterflowdistance' n'est pas renseignée pour la parcelle \"%s\"!", dto.getName()));
            error = true;
        }
        practicedPlot.setWaterFlowDistance(waterFlowDistance);
        return error;
    }

    protected boolean validIrrigationSystemType(ImportResults result, long line, PracticedPlotImportDto dto, PracticedPlot practicedPlot, boolean error) {

        String value = dto.getIrrigationSystemTypeImportedValue();
        IrrigationSystemType irrigationSystemType = null;

        if (!Strings.isNullOrEmpty(value)) {
            try {
                irrigationSystemType = IrrigationSystemType.valueOf(value);
            } catch (IllegalArgumentException e) {
                result.addErrorLine(line, String.format("PARCELLE DU SYNTHÉTISÉ IGNORÉE! Système d'irrigation, non supportée : %s", value));
                error = true;
            }
        }
        practicedPlot.setIrrigationSystemType(irrigationSystemType);
        return error;
    }

    protected boolean validPompEngineType(ImportResults result, long line, PracticedPlotImportDto dto, PracticedPlot practicedPlot, boolean error) {
        String value = dto.getPompEngineTypeImportedValue();
        PompEngineType pompEngineType = null;
        if (!Strings.isNullOrEmpty(value)) {
            try {
                pompEngineType = PompEngineType.valueOf(value);
            } catch (IllegalArgumentException e) {
                result.addErrorLine(line, String.format("PARCELLE DU SYNTHÉTISÉ IGNORÉE! Type de moteur de pompe, non supportée : %s", value));
                error = true;
            }
        }
        practicedPlot.setPompEngineType(pompEngineType);
        return error;
    }

    protected boolean validHosesPositionning(ImportResults result, long line, PracticedPlotImportDto dto, PracticedPlot practicedPlot, boolean error) {
        String value = dto.getHosesPositionningImportedValue();
        HosesPositionning hosesPositionning = null;

        if (!Strings.isNullOrEmpty(value)) {
            try {
                hosesPositionning = HosesPositionning.valueOf(value);
            } catch (IllegalArgumentException e) {
                result.addErrorLine(line, String.format("PARCELLE DU SYNTHÉTISÉ IGNORÉE! Positionnement des tuyaux d'arrosage, non supportée : %s", value));
                error = true;
            }
        }
        practicedPlot.setHosesPositionning(hosesPositionning);

        return error;
    }

    protected boolean validFrostProtectionType(ImportResults result, long line, PracticedPlotImportDto dto, PracticedPlot practicedPlot, boolean error) {
        String value = dto.getFrostProtectionTypeImportedValue();
        FrostProtectionType frostProtectionType = null;

        if (!Strings.isNullOrEmpty(value)) {
            try {
                frostProtectionType = FrostProtectionType.valueOf(value);
            } catch (IllegalArgumentException e) {
                result.addErrorLine(line, String.format("PARCELLE DU SYNTHÉTISÉ IGNORÉE! Type de moteur de pompe, non supportée : %s", value));
                error = true;
            }
        }
        practicedPlot.setFrostProtectionType(frostProtectionType);
        return error;
    }

    protected boolean validSolWaterPh(ImportResults result, long line, PracticedPlotImportDto dto, PracticedPlot practicedPlot, boolean error) {
        String value = dto.getSolWaterPhImportedValue();
        SolWaterPh solWaterPh = null;

        if (!Strings.isNullOrEmpty(value)) {
            value = StringUtils.replaceChars(value, ',', '.');
            try {
                Double dVal = Double.parseDouble(value);
                if (dVal < 5.5) {
                    solWaterPh = SolWaterPh.TRES_ACIDE;
                } else if (dVal < 6.5) {
                    solWaterPh = SolWaterPh.ACIDE;
                } else if (dVal < 7.5) {
                    solWaterPh = SolWaterPh.NEUTRE;
                } else {
                    solWaterPh = SolWaterPh.BASIQUE;
                }
            } catch (NumberFormatException e) {
                result.addErrorLine(line, String.format("PARCELLE DU SYNTHÉTISÉ IGNORÉE! PH de l'eau, non supportée : %s", value));
                error = true;
            }
        }
        practicedPlot.setSolWaterPh(solWaterPh);
        return error;

    }

    protected boolean validMaxSlope(ImportResults result, long line, PracticedPlotImportDto dto, PracticedPlot practicedPlot, boolean error) {

        String value = dto.getMaxSlopeImportedValue();
        MaxSlope maxSlope = null;

        if (!Strings.isNullOrEmpty(value)) {
            try {
                maxSlope = MaxSlope.valueOf(value);
            } catch (IllegalArgumentException e) {
                result.addErrorLine(line, String.format("PARCELLE DU SYNTHÉTISÉ IGNORÉE! Pente maximum, non supportée : %s", value));
                error = true;
            }
        } else {
            result.addErrorLine(line, String.format("PARCELLE DU SYNTHÉTISÉ IGNORÉE! la colonne 'maxslope' n'est pas renseignée pour la parcelle \"%s\"!", dto.getName()));
            error = true;
        }
        practicedPlot.setMaxSlope(maxSlope);
        return error;

    }

    protected boolean validBufferStrip(ImportResults result, long line, PracticedPlotImportDto dto, PracticedPlot practicedPlot, boolean error) {
        String value = dto.getBufferStripImportedValue();
            BufferStrip bufferStrip = null;

            if (!Strings.isNullOrEmpty(value)) {
                try {
                    bufferStrip = BufferStrip.valueOf(value);
                } catch (IllegalArgumentException e) {
                    result.addErrorLine(line, String.format("PARCELLE DU SYNTHÉTISÉ IGNORÉE! Bande tampon, non supportée : %s", value));
                    error = true;
                }
            } else {
                result.addErrorLine(line, String.format("PARCELLE DU SYNTHÉTISÉ IGNORÉE! la colonne 'bufferstrip' n'est pas renseignée pour la parcelle \"%s\"!", dto.getName()));
                error = true;
            }
        practicedPlot.setBufferStrip(bufferStrip);
            return error;
    }

    @Override
    public void init(ServiceContext serviceContext) {
        super.init(serviceContext);
        practicedPlotService = getServiceFactory().newService(PracticedPlotService.class);
        locationDao = getPersistenceContext().getRefLocationDao();
        refSolTextureGeppaDao = getPersistenceContext().getRefSolTextureGeppaDao();
        refSolProfondeurIndigoDao= getPersistenceContext().getRefSolProfondeurIndigoDao();
    }


}
