package fr.inra.agrosyst.services.pz0import;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: CliStarter.java 4775 2015-02-10 15:13:34Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.4.4/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/CliStarter.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import fr.inra.agrosyst.api.entities.AgrosystTopiaPersistenceContext;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.Equipment;
import fr.inra.agrosyst.api.entities.GrowingPlan;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.ToolsCoupling;
import fr.inra.agrosyst.api.entities.action.AbstractAction;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCycle;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCycleConnection;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCycleNode;
import fr.inra.agrosyst.api.entities.practiced.PracticedIntervention;
import fr.inra.agrosyst.api.entities.practiced.PracticedPlot;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystem;
import fr.inra.agrosyst.api.entities.security.AgrosystUser;
import fr.inra.agrosyst.api.entities.security.AgrosystUserTopiaDao;
import fr.inra.agrosyst.api.entities.security.RoleType;
import fr.inra.agrosyst.api.entities.security.UserRole;
import fr.inra.agrosyst.api.entities.security.UserRoleTopiaDao;
import fr.inra.agrosyst.api.services.ServiceFactory;
import fr.inra.agrosyst.api.services.domain.DomainService;
import fr.inra.agrosyst.api.services.growingplan.GrowingPlanService;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemService;
import fr.inra.agrosyst.api.services.practiced.PracticedPlotService;
import fr.inra.agrosyst.api.services.practiced.PracticedSystemService;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.security.AuthenticationService;
import fr.inra.agrosyst.api.services.users.UserDto;
import fr.inra.agrosyst.api.services.users.UserService;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.services.pz0import.action.ActionImporter;
import fr.inra.agrosyst.services.pz0import.domain.CroppingPanEntryImporter;
import fr.inra.agrosyst.services.pz0import.domain.CroppingPanEntrySpeciesImporter;
import fr.inra.agrosyst.services.pz0import.domain.DomainImporter;
import fr.inra.agrosyst.services.pz0import.domain.EquipmentForToolsCouplingImporter;
import fr.inra.agrosyst.services.pz0import.domain.EquipmentImporter;
import fr.inra.agrosyst.services.pz0import.domain.MainActionForToolsCouplingImporter;
import fr.inra.agrosyst.services.pz0import.domain.ToolsCouplingImporter;
import fr.inra.agrosyst.services.pz0import.domain.dto.CroppingPlanEntryImportDto;
import fr.inra.agrosyst.services.pz0import.domain.dto.CroppingPlanSpeciesImportDto;
import fr.inra.agrosyst.services.pz0import.growingplan.GrowingPlanImporter;
import fr.inra.agrosyst.services.pz0import.growingsystem.GrowingSystemImporter;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.logging.Level;
import java.util.logging.Logger;

import fr.inra.agrosyst.services.pz0import.practicedPlot.PracticedPlotImporter;
import fr.inra.agrosyst.services.pz0import.practicedSystem.PracticedSystemImporter;
import fr.inra.agrosyst.services.pz0import.practicedSystem.practicedCropCycle.PracticedCropCycleImporter;
import fr.inra.agrosyst.services.pz0import.practicedSystem.practicedCropCycleConnection.PracticedCropCycleConnectionImporter;
import fr.inra.agrosyst.services.pz0import.practicedSystem.practicedCropCycleNode.PracticedCropCycleNodeImporter;
import fr.inra.agrosyst.services.pz0import.practicedSystem.practicedIntervention.PracticedInterventionImporter;
import fr.inra.agrosyst.services.pz0import.practicedSystem.practicedInterventionToolsCoupling.PracticedToolsCouplingImportDto;
import fr.inra.agrosyst.services.pz0import.practicedSystem.practicedInterventionToolsCoupling.PracticedToolsCouplingImporter;
import fr.inra.agrosyst.services.pz0import.practicedSystem.practicedSeasonalCropCycle.PracticedSeasonalCropCycleImporter;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 *
 * @author antoine Schellenberger
 */
public class CliStarter {

    private static final Log log = LogFactory.getLog(CliStarter.class);

    protected final static String LAUNCH_INFO = "Un seul argument est authorisé, il sagit du chemin vers le fichier de configuration, si aucun n'est spécifié le fichier 'agrosyst-import.properties' sera utilisé s'il existe sinon les paramêtres par défaut s'appliqueront.";

    protected String dbConfigFilePath;
    protected ServiceFactory serviceFactory;
    protected AgrosystCliServiceContext serviceContext;
    protected DomainService domainService;
    protected GrowingSystemService growingSystemService;
    protected GrowingPlanService growingPlanService;
    protected PracticedSystemService practicedSystemService;
    protected PracticedPlotService practicedPlotService;

    protected String domainFilePath;
    protected String croppingPlanFilePath;
    protected String croppingPlanSpeciesFilePath;

    protected String equipmentFilePath;
    protected String toolCouplingFilePath;
    protected String equipmentToToolsCouplingFilePath;
    protected String mainActionToToolsCouplingFilePath;

    protected String growingPlanFilePath;
    protected String growingSystemFilePath;
    protected String practicedSystemFilePath;
    protected String practicedPlotFilePath;
    protected String practicedCropCycleFilePath;
    protected String practicedSeasonalCropCycleFilePath;;
    protected String practicedCropCycleNodeFilePath;
    protected String practicedCropCycleConnectionFilePath;
    protected String practicedInterventionFilePath;
    protected String practiceInterventionToolsCouplingFilePath;
    protected String abstractActionFilePath;
    protected String seedingActionSpeciesFilePath;
    protected String harverstingYealdFilePath;

    public CliStarter(String dbConfigFilePath) {
        this.dbConfigFilePath = dbConfigFilePath;
    }

    protected void init() {
        serviceContext = new AgrosystCliServiceContext(dbConfigFilePath);
        serviceFactory = serviceContext.getServiceFactory();
        domainService = serviceFactory.newService(DomainService.class);
        growingPlanService = serviceFactory.newService(GrowingPlanService.class);
        growingSystemService = serviceFactory.newService(GrowingSystemService.class);
        practicedSystemService = serviceFactory.newService(PracticedSystemService.class);
        practicedPlotService = serviceFactory.newService(PracticedPlotService.class);

        domainFilePath  = serviceContext.getAgrosystCliConfig().getDomainFilePath();
        croppingPlanFilePath = serviceContext.getAgrosystCliConfig().getCroppingPanFilePath();
        croppingPlanSpeciesFilePath = serviceContext.getAgrosystCliConfig().getCroppingPlanSpeciesFilePath();

        equipmentFilePath = serviceContext.getAgrosystCliConfig().getEquipmentFilePath();
        toolCouplingFilePath = serviceContext.getAgrosystCliConfig().getToolCouplingFilePath();
        equipmentToToolsCouplingFilePath = serviceContext.getAgrosystCliConfig().getEquipmentToToolsCouplingFilePath();
        mainActionToToolsCouplingFilePath = serviceContext.getAgrosystCliConfig().getMainActionToToolsCouplingFilePath();

        growingPlanFilePath = serviceContext.getAgrosystCliConfig().getGrowingPlanFilePath();
        growingSystemFilePath = serviceContext.getAgrosystCliConfig().getGrowingSystemFilePath();
        practicedSystemFilePath = serviceContext.getAgrosystCliConfig().getPracticedSystemFilePath();
        practicedPlotFilePath = serviceContext.getAgrosystCliConfig().getPracticedPlotFilePath();
        practicedCropCycleFilePath = serviceContext.getAgrosystCliConfig().getPracticedCropCycleFilePath();
        practicedSeasonalCropCycleFilePath = serviceContext.getAgrosystCliConfig().getPracticedSeasonalCropCycleFilePath();
        practicedCropCycleNodeFilePath = serviceContext.getAgrosystCliConfig().getPracticedCropCycleNodeFilePath();
        practicedCropCycleConnectionFilePath = serviceContext.getAgrosystCliConfig().getPracticedCropCycleConnectionFilePath();
        practicedInterventionFilePath = serviceContext.getAgrosystCliConfig().getPracticedInterventionFilePath();
        practiceInterventionToolsCouplingFilePath = serviceContext.getAgrosystCliConfig().getPracticeInterventionToolsCouplingFilePath();
        abstractActionFilePath = serviceContext.getAgrosystCliConfig().getAbstractActionFilePath();
        seedingActionSpeciesFilePath = serviceContext.getAgrosystCliConfig().getSeedingActionSpeciesFilePath();
        harverstingYealdFilePath = serviceContext.getAgrosystCliConfig().getHarverstingYealdFilePath();

        userAuthentication();
    }

    public AgrosystTopiaPersistenceContext getPersistenceContext() {
        return serviceContext.getPersistenceContext();
    }

    protected Map<Class, ImportResults> processImport() {
        Map<Class, ImportResults> allResults = Maps.newHashMap();

        try {
            Preconditions.checkNotNull(domainFilePath);
            Preconditions.checkNotNull(growingPlanFilePath);
            Preconditions.checkNotNull(growingSystemFilePath);

            // process file parsing, create entity and save result
            processPzOCSVFilesParsing(allResults);

            // if no errors, entities will be saved
            processEntitiesPersistence(allResults);

        } catch (FileNotFoundException ex) {
            getPersistenceContext().rollback();
            Logger.getLogger(CliStarter.class.getName()).log(Level.SEVERE, null, ex);
        } catch (Exception e) {
            getPersistenceContext().rollback();
            System.out.println("rollback\n");
            e.printStackTrace();
        }

        return allResults;
    }

    protected void processPzOCSVFilesParsing(Map<Class, ImportResults> allResults) throws FileNotFoundException {
        importDomains(domainFilePath, allResults);
        importCroppingPlans(croppingPlanFilePath, allResults);
        importCroppingPlanSpecies(croppingPlanSpeciesFilePath, allResults);
        importEquipments(equipmentFilePath, allResults);
        importToolsCouplings(toolCouplingFilePath, equipmentToToolsCouplingFilePath, mainActionToToolsCouplingFilePath, allResults);
        // TODO DCossé 09/12/14 test if  at least one equipment exist on tools coupling if tractor type is not 'auto moteur'
        importGrowingPlans(growingPlanFilePath, allResults);
        importGrowingSystems(growingSystemFilePath, allResults);
        importPracticedSystems(practicedSystemFilePath, allResults);
        importPracticedPlots(practicedPlotFilePath, allResults);
        importPracticedCropCycle(practicedCropCycleFilePath, allResults);
        importPracticedSeasonalCropCycleFilePath(practicedSeasonalCropCycleFilePath, allResults);
        importPracticedCropCycleNodeFilePath(practicedCropCycleNodeFilePath, allResults);
        importPracticedCropCycleConnectionFilePath(practicedCropCycleConnectionFilePath, allResults);
        importParcticedInterventionFilePath(practicedInterventionFilePath, allResults);
        importPracticeInterventionToolsCouplingFilePath(practiceInterventionToolsCouplingFilePath, allResults);
        //importAbstractActionFilePath(abstractActionFilePath, allResults);

        // TODO DCossé 03/02/15 the folowing ones are not implemented


        importSeedingActionSpeciesFilePath(seedingActionSpeciesFilePath);
        importHarverstingYealdFilePath(harverstingYealdFilePath);
    }

    protected void processEntitiesPersistence(Map<Class, ImportResults> allResults) {
        boolean importErrors = isImportErrors(allResults);
        if (!importErrors) {
            domainService.importPZ0Domains(allResults);
            growingPlanService.importPZ0GrowingPlans(allResults);
            growingSystemService.importPZ0GrowingSystems(allResults);
            practicedSystemService.importPZ0PracticedSystems(allResults);
            practicedPlotService.importPZ0PracticedPlots(allResults);

            getPersistenceContext().commit();
            log.info("Persistance achevée avec succées");
        } else {
            if (log.isErrorEnabled()) {
                log.error("LES FICHIERS COMPORTANT DES ERREURS ET DOIVENT ÊTRE CORRIGÉS AVANT DE PROCEDER À LA SAUVEGARDE DES ENTITÉS !\nPROCESSUS INTERROMPU");
            }
        }
    }

    protected boolean isImportErrors(Map<Class, ImportResults> allResults) {
        boolean errors = false;
        for (ImportResults result : allResults.values()) {
            if (result.getIgnoredRecords() > 0) {
                errors = true;
                break;
            }
        }
        return errors;
    }

    protected void importDomains(String domainFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        DomainImporter domainImporter = new DomainImporter();
        domainImporter.init(getServiceContext());
        ImportResults domainImportResults = domainImporter.importFromStream(new FileInputStream(domainFilePath), null);
        displayImportResult(domainImportResults);
        allResults.put(Domain.class, domainImportResults);
    }

    protected void importCroppingPlans(String croppingPlanFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        CroppingPanEntryImporter croppingPanEntryImporter = new CroppingPanEntryImporter();
        croppingPanEntryImporter.init(getServiceContext());
        if (StringUtils.isNotBlank(croppingPlanFilePath)) {
            ImportResults croppingPlanEntryImportResults = croppingPanEntryImporter.importFromStream(new FileInputStream(croppingPlanFilePath), null);
            displayImportResult(croppingPlanEntryImportResults);
            allResults.put(CroppingPlanEntryImportDto.class, croppingPlanEntryImportResults);
        }
    }

    protected void importCroppingPlanSpecies(String croppingPlanSpeciesFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        CroppingPanEntrySpeciesImporter croppingPanEntrySpeciesImporter = new CroppingPanEntrySpeciesImporter();
        croppingPanEntrySpeciesImporter.init(getServiceContext());
        if (StringUtils.isNotBlank(croppingPlanSpeciesFilePath)) {
            ImportResults croppingPlanSpeciesImportResults = croppingPanEntrySpeciesImporter.importFromStream(new FileInputStream(croppingPlanSpeciesFilePath), null);
            displayImportResult(croppingPlanSpeciesImportResults);
            allResults.put(CroppingPlanSpeciesImportDto.class, croppingPlanSpeciesImportResults);
        }
    }

    protected void importEquipments(String equipmentFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        EquipmentImporter equipmentsImporter = new EquipmentImporter();
        equipmentsImporter.init(getServiceContext());
        if (StringUtils.isNotBlank(equipmentFilePath)) {
            ImportResults domainsImportResults = allResults.get(Domain.class);
            ImportResults equipmentsImportResults = equipmentsImporter.importFromStream(new FileInputStream(equipmentFilePath), domainsImportResults.getEntityAndDepsByCsvIds());
            displayImportResult(equipmentsImportResults);
            allResults.put(Equipment.class, equipmentsImportResults);
        }
    }

    protected void importToolsCouplings(String toolCouplingFilePath, String equipmentToToolCouplingFilePath, String mainActionToToolCouplingFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        ToolsCouplingImporter toolsCouplingImporter = new ToolsCouplingImporter();
        toolsCouplingImporter.init(getServiceContext());
        if (StringUtils.isNotBlank(toolCouplingFilePath)) {
            ImportResults domainsImportResults = allResults.get(Domain.class);
            ImportResults toolsCouplingsImportResults = toolsCouplingImporter.importFromStream(new FileInputStream(toolCouplingFilePath), domainsImportResults.getEntityAndDepsByCsvIds());

            allResults.put(ToolsCoupling.class, toolsCouplingsImportResults);

            importEquipmentToToolCoupling(equipmentToToolCouplingFilePath, allResults, toolsCouplingsImportResults);
            importToolsCouplingMainsActions(mainActionToToolCouplingFilePath, allResults, toolsCouplingsImportResults);

            displayImportResult(toolsCouplingsImportResults);
        }
    }

    protected void importEquipmentToToolCoupling(String equipmentToToolCouplingFilePath, Map<Class, ImportResults> allResults, ImportResults toolsCouplingsImportResults) throws FileNotFoundException {
        if (StringUtils.isNotBlank(equipmentToToolCouplingFilePath)) {
            EquipmentForToolsCouplingImporter equipmentForToolCouplingImporter = new EquipmentForToolsCouplingImporter();
            equipmentForToolCouplingImporter.init(getServiceContext());
            ImportResults equipmentForToolCoupling = equipmentForToolCouplingImporter.importFromStream(new FileInputStream(equipmentToToolCouplingFilePath), null);
            displayImportResult(equipmentForToolCoupling);
            allResults.put(ToolsCoupling.class, equipmentForToolCoupling);
        } else {
            toolsCouplingsImportResults.addErrorLine(0L, "FICHIER DES ÉQUIPEMENT NON TROUVÉ, une combinaison d'outils doit obligatoirement comporter un tracteur ou automoteur.");
        }
    }

    protected void importToolsCouplingMainsActions(String mainsActionsToolCouplingFilePath, Map<Class, ImportResults> allResults, ImportResults toolsCouplingsImportResults) throws FileNotFoundException {
        if (StringUtils.isNotBlank(mainsActionsToolCouplingFilePath)) {
            MainActionForToolsCouplingImporter mainActionForToolsCouplingImporter = new MainActionForToolsCouplingImporter();
            mainActionForToolsCouplingImporter.init(getServiceContext());
            ImportResults mainsActionsForToolsCouplings = mainActionForToolsCouplingImporter.importFromStream(new FileInputStream(mainsActionsToolCouplingFilePath), null);
            displayImportResult(mainsActionsForToolsCouplings);
            allResults.put(ToolsCoupling.class, mainsActionsForToolsCouplings);
        } else {
            toolsCouplingsImportResults.addErrorLine(0L, "FICHIER DES ACTIONS PRINCIPALES NON TROUVÉ, une combinaison d'outils doit obligatoirement être associée avec au moins une action principale.");
        }
    }

    protected void importGrowingPlans(String growingPlanFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        GrowingPlanImporter growingPlanImporter = new GrowingPlanImporter();
        growingPlanImporter.init(getServiceContext());
        ImportResults domainImportResults = allResults.get(Domain.class);
        ImportResults growingPlanImportResults = growingPlanImporter.importFromStream(new FileInputStream(growingPlanFilePath), domainImportResults.getEntityAndDepsByCsvIds());
        displayImportResult(growingPlanImportResults);
        allResults.put(GrowingPlan.class, growingPlanImportResults);
    }

    protected void importGrowingSystems(String growingSystemFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        GrowingSystemImporter growingSystemImporter = new GrowingSystemImporter();
        growingSystemImporter.init(getServiceContext());
        ImportResults growingPlanImportResults = allResults.get(GrowingPlan.class);
        ImportResults growingSystemImportResults = growingSystemImporter.importFromStream(new FileInputStream(growingSystemFilePath), growingPlanImportResults.getEntityAndDepsByCsvIds());
        displayImportResult(growingSystemImportResults);
        allResults.put(GrowingSystem.class, growingSystemImportResults);
    }

    protected void importPracticedSystems(String practicedSystemFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        PracticedSystemImporter importer = new PracticedSystemImporter();
        importer.init(getServiceContext());
        ImportResults growingSystemImportResults = allResults.get(GrowingSystem.class);
        ImportResults practicedSystemImportResults = importer.importFromStream(new FileInputStream(practicedSystemFilePath), growingSystemImportResults.getEntityAndDepsByCsvIds());
        displayImportResult(practicedSystemImportResults);
        allResults.put(PracticedSystem.class, practicedSystemImportResults);
    }

    protected void importPracticedPlots(String practicedPlotFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        PracticedPlotImporter importer = new PracticedPlotImporter();
        importer.init(getServiceContext());
        ImportResults practicedSystemImportResults = allResults.get(PracticedSystem.class);
        ImportResults practicedPlotImportResults = importer.importFromStream(new FileInputStream(practicedPlotFilePath), practicedSystemImportResults.getEntityAndDepsByCsvIds());
        displayImportResult(practicedPlotImportResults);
        allResults.put(PracticedPlot.class, practicedPlotImportResults);
    }

    private void importPracticedCropCycle(String practicedCropCycleFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        PracticedCropCycleImporter importer = new PracticedCropCycleImporter();
        ImportResults practicedSystemImportResults = allResults.get(PracticedSystem.class);
        ImportResults importResults = importer.importFromStream(new FileInputStream(practicedCropCycleFilePath), practicedSystemImportResults.getEntityAndDepsByCsvIds());
        displayImportResult(importResults);
        allResults.put(PracticedCropCycle.class, importResults);
    }

    private void importPracticedSeasonalCropCycleFilePath(String practicedSeasonalCropCycleFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        PracticedSeasonalCropCycleImporter importer = new PracticedSeasonalCropCycleImporter();
        ImportResults importResults = importer.importFromStream(new FileInputStream(practicedSeasonalCropCycleFilePath), null);
        displayImportResult(importResults);
        allResults.put(PracticedCropCycle.class, importResults);
    }

    private void importPracticedCropCycleNodeFilePath(String practicedCropCycleNodeFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        PracticedCropCycleNodeImporter importer = new PracticedCropCycleNodeImporter();
        ImportResults importResults = importer.importFromStream(new FileInputStream(practicedCropCycleNodeFilePath), null);
        displayImportResult(importResults);
        allResults.put(PracticedCropCycleNode.class, importResults);
    }

    private void importPracticedCropCycleConnectionFilePath(String practicedCropCycleConnectionFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        PracticedCropCycleConnectionImporter importer = new PracticedCropCycleConnectionImporter();
        ImportResults importResults = importer.importFromStream(new FileInputStream(practicedCropCycleConnectionFilePath), null);
        displayImportResult(importResults);
        allResults.put(PracticedCropCycleConnection.class, importResults);
    }

    private void importParcticedInterventionFilePath(String practicedInterventionFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        PracticedInterventionImporter importer = new PracticedInterventionImporter();
        ImportResults importResults = importer.importFromStream(new FileInputStream(practicedInterventionFilePath), null);
        displayImportResult(importResults);
        allResults.put(PracticedIntervention.class, importResults);
    }

    private void importPracticeInterventionToolsCouplingFilePath(String practiceInterventionToolsCouplingFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        PracticedToolsCouplingImporter importer = new PracticedToolsCouplingImporter();
        ImportResults importResults = importer.importFromStream(new FileInputStream(practiceInterventionToolsCouplingFilePath), null);
        displayImportResult(importResults);
        allResults.put(PracticedToolsCouplingImportDto.class, importResults);
    }

    private void importAbstractActionFilePath(String abstractActionFilePath, Map<Class, ImportResults> allResults) throws FileNotFoundException {
        ActionImporter importer = new ActionImporter();
        ImportResults importResults = importer.importFromStream(new FileInputStream(abstractActionFilePath), null);
        displayImportResult(importResults);
        allResults.put(AbstractAction.class, importResults);
    }

    private void importHarverstingYealdFilePath(String harverstingYealdFilePath) {
        // TODO DCossé 03/02/15
    }

    private void importSeedingActionSpeciesFilePath(String seedingActionSpeciesFilePath) {
        // TODO DCossé 03/02/15
    }














    private void displayImportResult(ImportResults importResults) {
        if (log.isInfoEnabled()){
            log.info(String.format("Import des %s", importResults.getImportedEntityClass().getSimpleName()));
        }

        if (log.isErrorEnabled()) {
            Map<Long,List<String>> errorMessages = importResults.getErrorLineNumbersWithMessage();
            for (Map.Entry<Long,List<String>> longStringEntry : errorMessages.entrySet()) {
                long key = longStringEntry.getKey();
                for (String message :longStringEntry.getValue()) {
                    log.error(String.format("ligne %d, %s", key, message));
                }
            }
        }
        if (log.isInfoEnabled()){
            SortedMap<Long,List<String>> infoMessages = importResults.getInfoLineNumbersWithMessage();
            for (Map.Entry<Long,List<String>> longStringEntry : infoMessages.entrySet()) {
                long key = longStringEntry.getKey();
                for (String message :longStringEntry.getValue()) {
                    log.info(String.format("ligne %d, %s", key, message));
                }
            }

            // summary
            log.info(String.format("%d %s importés.", importResults.getAddedRecords(), importResults.getImportedEntityClass().getSimpleName()));
            log.info((String.format("%d %s ignorés.", importResults.getIgnoredRecords(), importResults.getImportedEntityClass().getSimpleName())));

        }
    }

    /**
     *
     * @param args Optional Parameter: config file path (if null 'agrosyst-import.properties' will be used)
     */
    public static void main(String[] args) {

        String dbConfigFilePath = null;

        if(args.length == 1) {
            dbConfigFilePath = args[0];
        } else if (args.length > 1) {
            System.out.println(LAUNCH_INFO);
        }

        CliStarter importer = new CliStarter(dbConfigFilePath);
        importer.init();
        importer.processImport();
    }


    protected void userAuthentication() {
        String email = serviceContext.getAgrosystCliConfig().getUserEmail();
        String password = serviceContext.getAgrosystCliConfig().getUserPassword();
        String firstName = serviceContext.getAgrosystCliConfig().getUserFirstName();
        String lastName = serviceContext.getAgrosystCliConfig().getUserLastName();

        AuthenticationService authenticationService = serviceFactory.newService(AuthenticationService.class);

        AgrosystUserTopiaDao userDao = getPersistenceContext().getAgrosystUserDao();

        // if user does not exist a new Administrateur one is created
        if (!userDao.forEmailEquals(email).tryFindAny().isPresent()) {
            createAdminUser(email, password, firstName, lastName, userDao);
        }

        UserDto result = authenticationService.login(email, password);
        serviceContext.setAuthenticationToken(result.getAuthenticationToken());

        log.info("connecté avec l'utilisateur:" + result.getFirstName() + " " + result.getLastName() + " " + result.getEmail());
    }

    private void createAdminUser(String email, String password, String firstName, String lastName, AgrosystUserTopiaDao userDao) {
        UserService userService = serviceFactory.newService(UserService.class);
        UserDto user = new UserDto();
        user.setActive(true);
        user.setEmail(email);
        user.setFirstName(firstName);
        user.setLastName(lastName);
        user = userService.createUser(user, password);

        AgrosystUser userEntity = userDao.forTopiaIdEquals(user.getTopiaId()).findUnique();

        UserRoleTopiaDao userRoleTopiaDao = getPersistenceContext().getUserRoleDao();
        userRoleTopiaDao.create(UserRole.PROPERTY_AGROSYST_USER, userEntity, UserRole.PROPERTY_TYPE, RoleType.ADMIN);
    }

    public void setDomainFilePath(String domainFilePath) {
        this.domainFilePath = domainFilePath;
    }

    public void setGrowingPlanFilePath(String growingPlanFilePath) {
        this.growingPlanFilePath = growingPlanFilePath;
    }

    public void setGrowingSystemFilePath(String growingSystemFilePath) {
        this.growingSystemFilePath = growingSystemFilePath;
    }

    protected ServiceContext getServiceContext() {
        return serviceContext;
    }
}
