package fr.inra.agrosyst.services.pz0import;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: AgrosystCliServiceContext.java 4612 2014-12-06 18:43:48Z echatellier $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.4.4/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/AgrosystCliServiceContext.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.gson.Gson;
import fr.inra.agrosyst.api.entities.AgrosystTopiaApplicationContext;
import fr.inra.agrosyst.api.entities.AgrosystTopiaDaoSupplier;
import fr.inra.agrosyst.api.entities.AgrosystTopiaPersistenceContext;
import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import fr.inra.agrosyst.api.services.AgrosystService;
import fr.inra.agrosyst.api.services.ServiceFactory;
import fr.inra.agrosyst.commons.gson.AgrosystGsonSupplier;
import fr.inra.agrosyst.services.AgrosystConfigurationHelper;
import fr.inra.agrosyst.services.DefaultServiceFactory;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.services.security.SecurityContext;
import java.util.Date;
import java.util.Properties;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.persistence.TopiaApplicationContextCache;
import org.nuiton.topia.persistence.TopiaTransaction;

/**
 *
 * @author Antoine Schellenberger
 */
public class AgrosystCliServiceContext implements ServiceContext {

    private static final Log log = LogFactory.getLog(AgrosystCliServiceContext.class);

    protected AgrosystTopiaPersistenceContext transaction;
    protected AgrosystTopiaApplicationContext rootContext;

    protected ServiceFactory serviceFactory;
    protected AgrosystCliServiceConfig config;

    protected String authenticationToken;
    protected SecurityContext securityContext;

    protected String configFileName;

    /**
     *
     * @param configFileName configuration file path, if null 'agrosyst-import.properties' will be used
     */
    public AgrosystCliServiceContext(String configFileName) {

        try {
            this.configFileName = configFileName;
            this.serviceFactory = new DefaultServiceFactory(this);

            config = new AgrosystCliServiceConfig(configFileName);

            Properties contextProperties = AgrosystConfigurationHelper.getRootContextProperties(config);
            rootContext = TopiaApplicationContextCache.getContext(contextProperties, AgrosystConfigurationHelper.getCreateTopiaContextFunction(config));
            rootContext.applicationInit();

        } catch (Exception eee) {
            throw new AgrosystTechnicalException("An exception occurred", eee);
        }
    }


    public void setAuthenticationToken(String authenticationToken) {
        this.authenticationToken = authenticationToken;
        if (!Strings.isNullOrEmpty(authenticationToken)) {
            securityContext = null;
        }
    }

    public ServiceFactory getServiceFactory() {
        return serviceFactory;
    }

    @Override
    public ServiceContext newServiceContext() {
        return this;
    }

    @Override
    public void close() {
        // do nothing for test, done by #afterEachTest()
    }

    @Override
    public Date getCurrentDate() {
        return new Date();
    }

    @Override
    public TopiaTransaction getTransaction() {
        TopiaTransaction result = getTransaction(true);
        return result;
    }

    @Override
    public TopiaTransaction getTransaction(boolean create) {
        AgrosystTopiaPersistenceContext result = getTransaction0(create);
        return result;
    }

    @Override
    public AgrosystTopiaDaoSupplier getDaoSupplier() {
        AgrosystTopiaDaoSupplier result = getTransaction0(true);
        return result;
    }

    @Override
    public AgrosystTopiaPersistenceContext getPersistenceContext() {
        // must stay to false for test, test should not open new transactions
        AgrosystTopiaPersistenceContext result = getTransaction0(false);
        return result;
    }

    @Override
    public AgrosystTopiaPersistenceContext getPersistenceContext(boolean create) {
        AgrosystTopiaPersistenceContext result = getTransaction0(create);
        return result;
    }

    private AgrosystTopiaPersistenceContext getTransaction0(boolean create) {
        Preconditions.checkState(rootContext != null);

        if (transaction == null && create) {
            transaction = rootContext.newPersistenceContext();
        }
        return transaction;
    }

    @Override
    public <E extends AgrosystService> E newService(Class<E> clazz) {
        return serviceFactory.newService(clazz);
    }

    @Override
    public <I> I newInstance(Class<I> clazz) {
        return serviceFactory.newInstance(clazz);
    }

    @Override
    public AgrosystCliServiceConfig getConfig() {
        if (config == null) {
            config = new AgrosystCliServiceConfig(configFileName);
        }
        return config;
    }

    public AgrosystCliServiceConfig getAgrosystCliConfig() {
        if (config == null) {
            config = new AgrosystCliServiceConfig(configFileName);
        }
        return config;
    }


    public void cleanup() {
        if (transaction != null && !transaction.isClosed()) {
            try {
                transaction.rollback();
//                transaction.getHibernateSupport().getHibernateSession().clear();
            } finally {
                transaction.close();
            }
        }
        transaction = null;

        if (!rootContext.isClosed()) {
            rootContext.close();
        }
        serviceFactory = null;
        securityContext = null;
    }
  

    @Override
    public SecurityContext getSecurityContext() {
        if (securityContext == null) {
            securityContext = new SecurityContext(getServiceFactory(), authenticationToken);
        }
        return securityContext;
    }

    @Override
    public SecurityContext getSecurityContextAsUser(String userId) {
        SecurityContext securityContext = new SecurityContext(getServiceFactory(), authenticationToken, userId);
        return securityContext;
    }

    @Override
    public Gson getGson() {
        return new AgrosystGsonSupplier().get();
    }

}
