package fr.inra.agrosyst.services.pz0import.domain;

/*
 * #%L
 * Agrosyst :: Command Line Interface
 * $Id: DomainImporter.java 4612 2014-12-06 18:43:48Z echatellier $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.4.1/agrosyst-cli/src/main/java/fr/inra/agrosyst/services/pz0import/domain/DomainImporter.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Strings;

import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.DomainTopiaDao;
import fr.inra.agrosyst.api.entities.DomainType;
import fr.inra.agrosyst.api.entities.WeatherStation;
import fr.inra.agrosyst.api.entities.WeatherStationTopiaDao;
import fr.inra.agrosyst.api.entities.Zoning;
import fr.inra.agrosyst.api.entities.referential.RefLegalStatus;
import fr.inra.agrosyst.api.entities.referential.RefLegalStatusTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefLocation;
import fr.inra.agrosyst.api.entities.referential.RefLocationTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefOTEX;
import fr.inra.agrosyst.api.entities.referential.RefOTEXTopiaDao;
import fr.inra.agrosyst.api.services.domain.DomainService;
import fr.inra.agrosyst.api.services.pz0.domains.DomainAndDependencies;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.services.pz0import.AbstractCSVImporter;

import java.io.InputStream;
import java.util.Map;

import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.services.pz0import.domain.dto.DomainImportDto;
import fr.inra.agrosyst.services.pz0import.domain.model.DomainImportModel;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

/**
 *
 * @author schellen, eancelet
 * 
 * TODO : si besoin ajouter :
 * * * * * * * l'import de speciestoarea
 * 
 */

public class DomainImporter extends AbstractCSVImporter {

    private static final Log log = LogFactory.getLog(DomainImporter.class);

    protected DomainService domainService;
    protected DomainTopiaDao domainDao;
    protected RefLocationTopiaDao locationDao;
    protected RefLegalStatusTopiaDao legalStatusDao;
    protected WeatherStationTopiaDao weatherStationDao;
    protected RefOTEXTopiaDao refOtexDao;


    @Override
    public ImportResults importFromStream(InputStream is,  Map<String, EntityAndDependencies> entitiesByCsvIds) {

        ImportResults importResults = new ImportResults(Domain.class);

        DomainImportModel model = new DomainImportModel();
        Import<DomainImportDto> importer = Import.newImport(model, is);

        // match the csv line number.
        Long line = FIRST_LINE_NUMBER;
        for (DomainImportDto dto : importer) {

            boolean error = false;
            Domain domain = domainService.newDomain();
            Binder<DomainImportDto, Domain> domainBinder = BinderFactory.newBinder(DomainImportDto.class, Domain.class);
            domainBinder.copyExcluding(dto, domain,
                    Domain.PROPERTY_TOPIA_ID
            );

            domain.setCode(csvCodeToAgrosystCode(Domain.class, dto.getCode()));

            // required datas
            error = validDomainCode(importResults, line, domain, error);
            error = validateCampaign(domain.getCampaign(), importResults, line, error);
            error = validateMainContact(domain.getMainContact(), importResults, line, error);
            error = validateDomainName(domain.getName(), importResults, line, error);
            error = validLocation(importResults, line, dto, error);
            error = validDomainType(importResults, line, domain, dto.getTypeName(), error);


            zoningNameToDomainZone(importResults, line, domain, dto);
            String legalStatusTopiaId = getLegalStatusId(importResults, line, dto, domain);
            Integer otex18 = otex18idToDomainOtex18(importResults, line, dto, domain);
            Integer otex70 = otex70IdToDomainOtex70(importResults, line, dto, domain);
            weatherStationIdToDomainWeatherStation(importResults, line, dto);


            // TODO DCossé 03/12/14 check it, don't know what to do
            SpeciesToArea(importResults, line, dto);

            if (!error) {
                DomainAndDependencies domainAndDependencies = new DomainAndDependencies(domain, dto.getLocationId(), legalStatusTopiaId, otex18, otex70);
                importResults.addEntity(dto.getId(), domainAndDependencies);
                importResults.addInfoLine(line, "DOMAINE VALIDÉ, csvid: " + dto.getId());
                pz0IdToObject.put(Domain.class,  dto.getId(), domainAndDependencies);
            } else {
                importResults.increaseIgnoredRecords();
                importResults.addErrorLine(line, "DOMAINE IGNORÉ csvid:" + dto.getId());
            }

            line++;
        }
        return  importResults;
    }

    protected void SpeciesToArea(ImportResults result, long line, DomainImportDto dto) {
        if (StringUtils.isNotBlank(dto.getSpeciesToArea())) {
            result.addInfoLine(line, String.format("DOMAINE IGNORÉ! Gérer speciestoarea pour domaine \"%s\", campagne %s!", dto.getName(), dto.getCampaign()));
        }
    }

    protected void weatherStationIdToDomainWeatherStation(ImportResults result, long line, DomainImportDto dto) {
        if (StringUtils.isNotBlank(dto.getDefaultWeatherStationId())) {
            WeatherStation weatherStation = weatherStationDao.forTopiaIdEquals(dto.getDefaultWeatherStationId()).findUniqueOrNull();
            if (weatherStation == null) {
                result.addInfoLine(line, String.format("La station météo n'a pas été retrouvée pour domaine \"%s\" sur la campagne %s !", dto.getName(), dto.getCampaign()));
            }
        }
    }

    protected Integer otex70IdToDomainOtex70(ImportResults result, long line, DomainImportDto dto, Domain domain) {
        Integer otex70 = null;
        if (!Strings.isNullOrEmpty(dto.getRefOtex70Id())) {
            RefOTEX refOtex70 = refOtexDao.forTopiaIdEquals(dto.getRefOtex70Id()).findUniqueOrNull();
            if (refOtex70 == null) {
                result.addInfoLine(line, String.format("ATTENTION! le Code OTEX70 de topiaId %s n'existe pas en base de données pour le domaine \"%s\".", dto.getOtex70(), dto.getName()));
            }
            else {
                otex70 = refOtex70.getCode_OTEX_70_postes();
                domain.setOtex70(refOtex70);
            }
        }
        return otex70;
    }

    protected Integer otex18idToDomainOtex18(ImportResults result, long line, DomainImportDto dto, Domain domain) {
        Integer otex18 = null;
        if (!Strings.isNullOrEmpty(dto.getRefOtex18Id())) {
            RefOTEX refOtex18 = refOtexDao.forTopiaIdEquals(dto.getRefOtex18Id()).findUniqueOrNull();
            if (refOtex18 == null) {
                result.addInfoLine(line, String.format("ATTENTION! le Code OTEX18 de topiaId %s n'existe pas en base de données pour le domaine \"%s\".", dto.getOtex18(), dto.getName()));
            } else {
                otex18 = refOtex18.getCode_OTEX_18_postes();
                domain.setOtex18(refOtex18);
            }
        }
        return otex18;
    }

    protected String getLegalStatusId(ImportResults result, long line, DomainImportDto dto, Domain domain) {
        String legalStatusTopiaId = null;
        // RefLegalStatus
        if (!Strings.isNullOrEmpty(dto.getLegalStatusId())) {
            RefLegalStatus legalStatus = legalStatusDao.forTopiaIdEquals(dto.getLegalStatusId()).findUniqueOrNull();
            if (legalStatus == null)  {
                legalStatusTopiaId = null;
                result.addInfoLine(line, String.format("ATTENTION! le legalStatus de topiaId %s n'existe pas en base de données pour le domaine \"%s\".", dto.getLegalStatus(), dto.getName()));
            }
            else {
                domain.setLegalStatus(legalStatus);
                legalStatusTopiaId = legalStatus.getTopiaId();
            }
        }
        return legalStatusTopiaId;
    }

    protected boolean validLocation(ImportResults result, long line, DomainImportDto dto, boolean error) {
        if (StringUtils.isNotBlank(dto.getLocationId())) {
            RefLocation location = locationDao.forTopiaIdEquals(dto.getLocationId()).findUniqueOrNull();
            if (location == null) {
                result.addInfoLine(line, String.format("DOMAINE IGNORÉ! l'objet RefLocation de topiaId %s n'existe pas en base de données pour le domaine \"%s\".", dto.getLocationId(), dto.getName()));
                error = true;
            }
        } else {
            result.addErrorLine(line, String.format("DOMAINE IGNORÉ! la colonne reflocation n'est pas renseignée pour le domaine \"%s\"!", dto.getName()));
            error = true;
        }
        return error;
    }

    protected void zoningNameToDomainZone(ImportResults result, long line, Domain domain, DomainImportDto dto) {
        String zoningName = dto.getZoningName();
        if (!Strings.isNullOrEmpty(zoningName)) {
            if (zoningName.equalsIgnoreCase("ZONE_DE_PLAINE")) {
                domain.setZoning(Zoning.ZONE_DE_PLAINE);
            } else if (zoningName.equalsIgnoreCase("ZONE_DEFAVORISEE_SIMPLE")) {
                domain.setZoning(Zoning.ZONE_DEFAVORISEE_SIMPLE);
            } else if (zoningName.equalsIgnoreCase("ZONE_DE_MONTAGNE")) {
                domain.setZoning(Zoning.ZONE_DE_MONTAGNE);
            } else if (zoningName.equalsIgnoreCase("ZONE_DE_PIEMONT")) {
                domain.setZoning(Zoning.ZONE_DE_PIEMONT);
            } else {
                result.addInfoLine(line, "Zoning non supportée : " + zoningName);
            }
        }
    }

    protected boolean validDomainCode(ImportResults importResults, long line, Domain domain, boolean error) {
        if (StringUtils.isBlank(domain.getCode())) {
            importResults.addErrorLine(line, "DOMAINE IGNORÉ!, Le code du domaine n'est pas valide.");
            error = true;
        }
        return error;
    }

    protected boolean validateCampaign(int campaign, ImportResults results, long line, boolean error) {
        if (campaign == 0) {
            results.addErrorLine(line, "Campagne non valide !");
            error = true;
        }
        return error;
    }

    protected boolean validateMainContact(String mainContact, ImportResults results, long line, boolean error) {
        if (StringUtils.isBlank(mainContact)){
            results.addErrorLine(line, "aucun nom de contact principale de spécifié !");
            error = true;
        }
        return error;
    }

    protected boolean validateDomainName(String domainName, ImportResults results, long line, boolean error) {
        if (StringUtils.isBlank(domainName)){
            results.addErrorLine(line, "Le nom de domaine n'est pas renseigné !");
            error = true;
        }
        return error;
    }

    protected boolean validDomainType(ImportResults result, long line, Domain domain, String typeName, boolean error) {
        if (StringUtils.isNotBlank(typeName)) {
            if (typeName.equalsIgnoreCase("exploitation_agricole")) {
                domain.setType(DomainType.EXPLOITATION_AGRICOLE);
            } else if (typeName.equalsIgnoreCase("domaine_experimental")) {
                domain.setType(DomainType.DOMAINE_EXPERIMENTAL);
            } else if (typeName.equalsIgnoreCase("ferme_de_lycee_agricole")) {
                domain.setType(DomainType.FERME_DE_LYCEE_AGRICOLE);
            } else {
                result.addErrorLine(line, "Type de domaine non supporté : " + typeName);
                error = true;
            }
        } else {
            result.addErrorLine(line, "Type de domaine non renseigné : " + typeName);
            error = true;
        }
        return error;
    }

    @Override
    public void init(ServiceContext serviceContext) {
        super.init(serviceContext);
        domainDao = getPersistenceContext().getDomainDao();
        locationDao = getPersistenceContext().getRefLocationDao();
        legalStatusDao = getPersistenceContext().getRefLegalStatusDao();
        weatherStationDao = getPersistenceContext().getWeatherStationDao();
        refOtexDao = getPersistenceContext().getRefOTEXDao();
        domainService = getServiceFactory().newService(DomainService.class);
    }

}
