package fr.inra.agrosyst.api.services.referential;

/*
 * #%L
 * Agrosyst :: API
 * $Id: ReferentialService.java 5023 2015-07-06 12:21:51Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-api/src/main/java/fr/inra/agrosyst/api/services/referential/ReferentialService.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.BioAgressorType;
import fr.inra.agrosyst.api.entities.CroppingPlanSpecies;
import fr.inra.agrosyst.api.entities.MaterielType;
import fr.inra.agrosyst.api.entities.referential.RefActaDosageSPC;
import fr.inra.agrosyst.api.entities.referential.RefActaTraitementsProduit;
import fr.inra.agrosyst.api.entities.referential.RefBioAgressor;
import fr.inra.agrosyst.api.entities.referential.RefClonePlantGrape;
import fr.inra.agrosyst.api.entities.referential.RefDepartmentShape;
import fr.inra.agrosyst.api.entities.referential.RefElementVoisinage;
import fr.inra.agrosyst.api.entities.referential.RefEspece;
import fr.inra.agrosyst.api.entities.referential.RefFertiEngraisorg;
import fr.inra.agrosyst.api.entities.referential.RefFertiMinUNIFA;
import fr.inra.agrosyst.api.entities.referential.RefFertiOrga;
import fr.inra.agrosyst.api.entities.referential.RefInterventionAgrosystTravailEDI;
import fr.inra.agrosyst.api.entities.referential.RefLocation;
import fr.inra.agrosyst.api.entities.referential.RefMateriel;
import fr.inra.agrosyst.api.entities.referential.RefOrientationEDI;
import fr.inra.agrosyst.api.entities.referential.RefParcelleZonageEDI;
import fr.inra.agrosyst.api.entities.referential.RefSolArvalis;
import fr.inra.agrosyst.api.entities.referential.RefSolCaracteristiqueIndigo;
import fr.inra.agrosyst.api.entities.referential.RefSolProfondeurIndigo;
import fr.inra.agrosyst.api.entities.referential.RefSolTextureGeppa;
import fr.inra.agrosyst.api.entities.referential.RefTraitSdC;
import fr.inra.agrosyst.api.entities.referential.RefUniteEDI;
import fr.inra.agrosyst.api.entities.referential.RefVariete;
import fr.inra.agrosyst.api.services.AgrosystService;
import fr.inra.agrosyst.api.services.practiced.RefStadeEdiDto;
import fr.inra.agrosyst.api.services.pz0.ImportResults;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Referential service.
 * 
 * @author Eric Chatellier
 */
public interface ReferentialService extends AgrosystService {

    /**
     * Retourne les valeurs des champs 'typeMateriel1' utilisé par les materiels groupé par
     * categorie de materiel.
     * 
     * @see RefMateriel#PROPERTY_TYPE_MATERIEL1
     * @return values
     */
    Map<MaterielType, List<String>> getTypeMateriel1List();

    /**
     * Retourne les valeurs des champs 'typeMateriel2' utilisé par les materiels.
     * 
     * @see RefMateriel#PROPERTY_TYPE_MATERIEL2
     * @param filter filter
     * @return values
     */
    List<String> getTypeMateriel2List(TypeMaterielFilter filter);

    /**
     * Retourne les valeurs des champs 'typeMateriel3' utilisé par les materiels.
     * 
     * @see RefMateriel#PROPERTY_TYPE_MATERIEL3
     * @param filter filter
     * @return values
     */
    List<String> getTypeMateriel3List(TypeMaterielFilter filter);

    /**
     * Retourne les valeurs des champs 'typeMateriel4' utilisé par les materiels.
     * 
     * @see RefMateriel#PROPERTY_TYPE_MATERIEL4
     * @param filter filter
     * @return values
     */
    List<String> getTypeMateriel4List(TypeMaterielFilter filter);

    /**
     * Retourne les valeurs des champs 'typeMateriel4' utilisé par les materiels.
     * 
     * @see RefMateriel#PROPERTY_UNITE
     * @see RefMateriel#PROPERTY_UNITE_PAR_AN
     * @param filter filter
     * @return values
     */
    Map<String, String[]> getMaterielUniteMap(TypeMaterielFilter filter);

    /**
     * Find materiel from topia id.
     * 
     * @param materielTopiaId materiel topia id
     * @return
     */
    RefMateriel findMateriel(String materielTopiaId);

    /**
     * Return l'ensemble des régions utilisées par les sols arvalis.
     * 
     * @return ensemble des regions
     */
    Map<Integer, String> getSolArvalisRegions();

    /**
     * Get ref sol arvalis filtered by domain region.
     * 
     * @param regionCode region
     * @return sol arvalis for region
     */
    List<RefSolArvalis> getSolArvalis(Integer regionCode);

    /**
     * Find sol arvalis from topia id.
     * 
     * @param solArvalisTopiaId sol arvalis topia id
     * @return
     */
    RefSolArvalis findSolArvalis(String solArvalisTopiaId);

    List<RefEspece> findSpecies(String filter);

    /**
     * Valid that species with same species botanic code as variety's edi species code exists within GEVES or PLANT_GRAP referential
     * @param variete variety to valid
     * @param speciesId species to look for botanic code
     * @return true if valid
     */
    boolean validVarietesFromSpeciesId(RefVariete variete, String speciesId);

    /**
     * Valid that variety with same spece botanique code as the given one exists within GEVES or PLANT_GRAP referential
     * @param variete variete to valid
     * @param code_espece_botanique code to look for
     * @return true if valid
     */
    boolean validVarietesFromCodeEspeceEdi(RefVariete variete, String code_espece_botanique);

    RefEspece getSpecies(String speciesId);

    List<RefVariete> findVarietes(String speciesId, String filter);

    RefVariete getVariete(String varieteId);

    RefClonePlantGrape getClonePlantGrape(String clonePlantGrapeId);

    /**
     * Get a specific location from its identifier.
     * @param refLocationId
     * @return RefLocationArvalis
     */
    RefLocation getRefLocation(String refLocationId);

    /**
     * Find all RefCommuneValues according to the filter
     * @param filter
     * @return all RefCommuneValues
     */
    List<RefLocation> findActiveCommunes(String filter);
    /**
     * Get all Otex18 codes and wording.
     * @return all Otex18 codes and wording.
     */
    Map<Integer, String> findAllActiveOtex18Code();
    
    /**
     * Get all Otex 70 codes filtered from the otex18code code.
     * @param otex18code the otex 18 code to filter on.
     * @return all RefOtex filtered from the given filter.
     */
    Map<Integer, String> findAllActiveCodeOtex70ByOtex18code(Integer otex18code);

    /**
     * Find all RefOrientationEDI.
     * 
     * @return all orientation EDI
     */
    List<RefOrientationEDI> findAllReferentielEDI();

    /**
     * Find orientation entity from topia id.
     * 
     * @param orientationTopiaId topai id
     * @return orientation edi entity
     */
    RefOrientationEDI findOrientation(String orientationTopiaId);

    /**
     * Find all species with :
     * - colonne D (nom Section), valeur "Porte-greffes"
     * - "varietes PlantGrape" : colonne C (Utilisation), valeur "PG"
     * and name starting with 'term'.
     * 
     * @param term search term
     * @return all matching varieties
     */
    List<RefVariete> findGraftSupports(String term);

    /**
     * Find all varieties with :
     * - "variétes Geves" : colonne P (Type Variétal), valeur "CLO" 
     * - "Clones PlanGrape" : afficher à l'utilisateur le code clone AnnéeAgrement Origne
     * and name starting with 'term'.
     * 
     * @param speciesId species id to search plant grape for
     * @param varietyId species's variety id
     * @param term search term
     * @return all matching varieties
     */
    List<RefClonePlantGrape> findGraftClones(String speciesId, String varietyId, String term);

    /**
     * Find all active RefTypeTravailEDIs
     * @return all active RefTypeTravailEDIs
     */
    List<RefInterventionAgrosystTravailEDI> findAllActiveAgrosystActions();

    /**
     * Find all active RefTypeTravailEDIs according the interventionType given as parameter
     * @param interventionType The action type to filter on.
     * @return All active RefTypeTravailEDIs according the interventionType given as parameter.
     */
    List<RefInterventionAgrosystTravailEDI> findAllActiveAgrosystActions(AgrosystInterventionType interventionType);

    /**
     * Find the RefActionAgrosystTravailEDI from it's topiaId
     * @param topiaId The RefActionAgrosystTravailEDI topiaId's
     * @return the RefActionAgrosystTravailEDI
     */
    RefInterventionAgrosystTravailEDI getRefInterventionAgrosystTravailEDI(String topiaId);

    /**
     * Recuperation de tous les parcelles zonages actif.
     * 
     * @return all active RefParcelleZonageEDI
     */
    List<RefParcelleZonageEDI> getAllActiveParcelleZonage();

    /**
     * Get all active sol textures.
     * 
     * @return all active textures
     */
    List<RefSolTextureGeppa> getAllActiveSolTextures();

    /**
     * Get all active sol profondeurs.
     * 
     * @return all active profondeurs
     */
    List<RefSolProfondeurIndigo> getAllActiveSolProfondeurs();

    /**
     * Get all active sol caracteristiques.
     * 
     * @return all active sol caracteristiques
     */
    List<RefSolCaracteristiqueIndigo> getAllActiveSolCaracteristiques();

    /**
     * return all categories and ProductTypes from the actives RefFertiMinUNIFA
     * @return all categories and ProductTypes from the actives RefFertiMinUNIFA
     */
    List<MineralProductType> findAllActiveMineralProductTypes();

    /**
     * Return all fertilizerShape corresponding to the actives fertiMin category given has parameter.
     * @param categ the category to apply filter on.
     * @return All fertilizerShape corresponding to the actives fertiMin category given has parameter.
     */
    List<String> findAllActiveFertiMinShape(Integer categ);

    /**
     * Find all active RefFertiMinUnifa filter by category and shape.
     * @param categ The Fertilizer category
     * @param fertilizerShape the fertelizer shape
     * @param productId product topiaId
     * @return all RefFertiMinUnifa matching the condition.
     */
    List<RefFertiMinUNIFA> findAllActiveRefFertiMinUnifaByCategAndShape(Integer categ, String fertilizerShape, String productId);

    /**
     * Find all Actives RefUnitesEDI
     * @return All Actives RefUnitesEDI
     */
    List<RefUniteEDI> findAllActiveRefUnitesEDI();

    /**
     * Find all Actives RefFertiOrga
     * @return All Actives RefFertiOrga
     */
    List<RefFertiOrga> findAllActiveOrganicProductTypes();

    /**
     * Find all Actives RefFertiorgs
     * @return All Actives RefFertiorgs
     */
    List<RefFertiEngraisorg> findAllActiveRefFertiorgs();

//    /**
//     * Find ProductPrices form RefFertiMin products for the given campaigns if not null or all campaigns if campaigns is null
//     * @param refFertiMinUnifaTopiaId Topia Id of the product
//     * @param campaigns All the campaigns to get product prices. If null the prices are find on all campaigns
//     * @return All ProductPrices according the campaigns filter.
//     */
//    ProductPrices getFertiMinProductPrices(String refFertiMinUnifaTopiaId, String campaigns);

//    /**
//     * Find ProductPrices form RefFertiEngraisorg products for the given campaigns if not null or all campaigns if campaigns is null
//     * @param refFertiEngraiorgTopiaId Topia Id of the product
//     * @param campaigns All the campaigns to get product prices. If null the prices are find on all campaigns
//     * @return All ProductPrices according the campaigns filter.
//     */
//    ProductPrices getFertiEngraisorgProductPrices(String refFertiEngraiorgTopiaId, String campaigns);

    /**
     * Return all actives RefStadeEDI with vegetativeProfile matching the one given in parameters.
     * @param vegetativeProfile the vegetativeProfile to filter on
     * @return All actives RefStadeEDI with vegetativeProfile matching the one given in parameters.
     */
    List<RefStadeEdiDto> getRefStadesEdi(Integer vegetativeProfile);


    /**
     * return the treatment code and name from the Acta treatment referential.
     * @return the treatment code and name
     */
    Map<String, String> getActaTreatementCodesAndNames();

    /**
     * return the list of ACTA treatment product type ordered by AgrosystInterventionType
     * @return the list of ACTA treatment product type by AgrosystInterventionType
     */
    Map<AgrosystInterventionType, List<String>> getAllActiveActaTreatmentProductTypes();

    /**
     * Return all active element de voisinage.
     * 
     * @return all active element de voisinage
     */
    List<RefElementVoisinage> getAllActiveElementVoisinages();

    List<RefBioAgressor> getTreatmentTargets(BioAgressorType category);

    /**
     * If several species are specified, the minimum value is returned
     *
     * @param phytoId_Produit is the ID from RefActaTraitementsProduits#id_produit
     * @param speciesIds is a list of ID from RefEspece#topiaId
     * @return the computed dose
     */
    RefActaDosageSPC computeActaReferenceDose(String phytoId_Produit, Set<String> speciesIds);

    /**
     * Return all active System Growing Characteristics including the characteristic types.
     * 
     * @return all {@link RefTraitSdC}
     */
    List<RefTraitSdC> getAllActiveGrowingSystemCharacteristics();

    /**
     * Return all Active ActaTraitementsProduit according the given product type
     *
     * @param interventionType
     * @param productType the product type to filter on
     * @return
     */
    List<RefActaTraitementsProduit> getActaTraitementsProduits(AgrosystInterventionType interventionType, String productType);

    /**
     * Get RefBioAgressor with id the given id
     * @param bioAgressorId RefBioAgressor id
     * @return the RefBioAgressor  with id the given id
     */
    RefBioAgressor getBioAgressor(String bioAgressorId);

    /**
     * Return RefDepartmentShape from the given departmentCode if found.
     * @param departmentCode departmentCode where country shape can be found
     * @return RefDepartmentShape if found
     */
    RefDepartmentShape getDepartmentShape(String departmentCode);

    /**
     *
     * @return Map of communeSite indexed by station Id
     */
    Map<String, String> getAllRefStationMeteoMap();

    /**
     * Create new RefFertiMinUNIFA instance, not persisted yet
     * @return the RefFertiMinUNIFA
     */
    RefFertiMinUNIFA getNewRefFertiMinUNIFA();
    /**
     * Valide RefFertiMin preconditions
     * @param product product to valid
     * @return true if valid false if not
     */
    boolean isValidRefFertiMinProduct(RefFertiMinUNIFA product);

    /**
     * Create or update the given product
     * @param product product to save
     * @return persisted product
     */
    RefFertiMinUNIFA createOrUpdateRefMineralProductToInput(RefFertiMinUNIFA product);

    /**
     * Try to Import RefFertiMinUnifa created from from PZ0 csv
     * @param allResult All RefFertiMinUnifas.
     */
    void importRefFertiMinUnifaPz0(Map<Class, ImportResults> allResult);

    /**
     * get CroppingPlanSpecies code by key.
     * The key is made of two parts (part 2 is optional)
     * part 1: the RefEspece id
     * part 2: (if there is a RefVariety) "_" + RefVariety Id
     * @param allCroppingPlanSpecies the croppingPalnEntry where species come from
     * @return map of key to croppingPlanSpecies code
     */
    Map<String, String> getCroppingPlanSpeciesCodeByRefEspeceAndVarietyKey(Collection<CroppingPlanSpecies> allCroppingPlanSpecies);

    /**
     *
     * @param allCroppingPlanSpecies
     * @return
     */
    Map<String, String> getRefEspeceAndVarietyKeyByCroppingPlanSpeciesCode(Collection<CroppingPlanSpecies> allCroppingPlanSpecies);
}
