package fr.inra.agrosyst.api.services.referential;

/*
 * #%L
 * Agrosyst :: API
 * $Id: ImportService.java 3270 2014-02-17 11:08:03Z echatellier $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-api/src/main/java/fr/inra/agrosyst/api/services/referential/ImportService.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.io.InputStream;

import fr.inra.agrosyst.api.services.AgrosystService;

/**
 * Referentiel service.
 *
 * @author Eric Chatellier
 */
public interface ImportService extends AgrosystService {

    /**
     * Import des materiels tracteurs au format csv.
     *
     * @param contentStream content stream
     * @return import result (counts)
     */
    ImportResult importMaterielTracteursCSV(InputStream contentStream);

    /**
     * Import des materiels automoteurs au format csv.
     *
     * @param contentStream content stream
     * @return import result (counts)
     */
    ImportResult importMaterielAutomoteursCSV(InputStream contentStream);

    /**
     * Import des materiels outils au format csv.
     *
     * @param contentStream content stream
     * @return import result (counts)
     */
    ImportResult importMaterielOutilsCSV(InputStream contentStream);

    /**
     * Import des materiels migration au format csv.
     *
     * @param contentStream content stream
     * @return import result (counts)
     */
    ImportResult importMaterielIrrigationCSV(InputStream contentStream);

    /**
     * Import des communes
     *
     * @param communesStream  stream contenant le CSV des communes (source INSEE)
     * @param postCodesStream stream contenant le CSV des codes postaux
     * @param raStream        stream contenant le CSV des régions agricoles
     * @return import result (counts)
     */
    ImportResult importCommuneInseeCSV(InputStream communesStream, InputStream postCodesStream, InputStream raStream);

    /**
     * Import des communes
     *
     * @param communesAndPostCodesStream  stream contenant le CSV des communes et dles ocdes postaux (source OSM)
     * @param raStream        stream contenant le CSV des régions agricoles
     * @return import result (counts)
     */
    ImportResult importCommuneOsmCSV(InputStream communesAndPostCodesStream, InputStream raStream);


    ImportResult importDepartmentShapes(InputStream stream);

    /**
     * Import des sols arvalis.
     *
     * @param solsStream sols arvalis stream
     * @param regionsStream region name to region code stream
     * @return import result (counts)
     */
    ImportResult importSolArvalisCSV(InputStream solsStream, InputStream regionsStream);

    /**
     * Import des statuts juridiques.
     *
     * @param statusStream fichier des status
     * @return import result (counts)
     */
    ImportResult importLegalStatusCSV(InputStream statusStream);

    ImportResult importEspeces(InputStream stream);

    ImportResult importVarietesGeves(InputStream stream);

    ImportResult importVarietesPlantGrape(InputStream stream);

    ImportResult importClonesPlantGrape(InputStream stream);

    ImportResult importEspecesToVarietes(InputStream stream);

    /**
     * Import du référentiel OTEX au format csv.
     *
     * @param contentStream content stream
     * @return import result (counts)
     */
    ImportResult importOtexCSV(InputStream contentStream);

    /**
     * Import du référentiel
     *
     * @param contentStream content stream
     * @return import result (counts)
     */
    ImportResult importOrientationEdiCSV(InputStream contentStream);


    ImportResult importInterventionAgrosystTravailEdiCSV(InputStream contentStream);

    ImportResult importTypeTravailEdiCSV(InputStream contentStream);

    ImportResult importStadesEdiCSV(InputStream contentStream);

    /**
     * Import du référentiel sol texture geppa.
     *
     * @param stream flux de lecture
     * @return import result
     */
    ImportResult importSolTextureGeppa(InputStream stream);

    /**
     * Import du référentiel zonage parcelle edi.
     *
     * @param stream flux de lecture
     * @return import result
     */
    ImportResult importZonageParcelleEdi(InputStream stream);

    /**
     * Import du référentiel sol profondeur indigo.
     *
     * @param stream flux de lecture
     * @return import result
     */
    ImportResult importSolProfondeurIndigo(InputStream stream);

    /**
     * Import du référentiel sol caracteristiques indigo.
     *
     * @param stream flux de lecture
     * @return import result
     */
    ImportResult importSolCarateristiquesIndigo(InputStream stream);

    /**
     * Import du référentiel UnitesEDI
     * @param stream flux de lecture
     * @return import result
     */
    ImportResult importUniteEDI(InputStream stream);

    /**
     * Import du référentiel FertiMinUNIFA
     * @param contentStream flux de lecture
     * @return import result
     */
    ImportResult importFertiMinUNIFA(InputStream contentStream);

    /**
     * Import du référentiel Adventices.
     * @param contentStream flux de lecture
     * @return import result
     */
    ImportResult importAdventices(InputStream contentStream);

    /**
     * Import du référentiel NuisiblesEDI.
     * @param contentStream flux de lecture
     * @return import result
     */
    ImportResult importNuisiblesEDI(InputStream contentStream);

    /**
     * Import du référentiel ferti Orga.
     * @param contentStream flux de lecture
     * @return import result
     */
    ImportResult importFertiOrga(InputStream contentStream);

    /**
     * Import du référentiel Ferti_engraisorg.
     * @param contentStream contentStream flux de lecture
     * @return import result
     */
    ImportResult importFertiEngraisOrg(InputStream contentStream);

    /**
     * Import du référentiel station meteo.
     * @param contentStream contentStream flux de lecture
     * @return import result
     */
    ImportResult importStationMeteo(InputStream contentStream);

    ImportResult importGesCarburants(InputStream stream);

    ImportResult importGesEngrais(InputStream stream);

    ImportResult importGesPhyto(InputStream stream);

    ImportResult importGesSemences(InputStream stream);

    ImportResult importNrjCarburants(InputStream stream);

    ImportResult importNrjEngrais(InputStream stream);

    ImportResult importNrjPhyto(InputStream stream);

    ImportResult importNrjSemences(InputStream stream);

    ImportResult importNrjGesOutils(InputStream stream);

    ImportResult importMesure(InputStream stream);

    ImportResult importSupportOrganeEDI(InputStream stream);
    
    ImportResult importStadeNuisibleEDI(InputStream stream);

    ImportResult importTypeNotationEDI(InputStream stream);

    ImportResult importValeurQualitativeEDI(InputStream stream);

    ImportResult importUnitesQualifiantEDI(InputStream stream);

    ImportResult importActaTraitementsProduits(InputStream stream);

    ImportResult importActaTraitementsProduitsCateg(InputStream stream);

    ImportResult importActaSubstanceActive(InputStream stream);

    ImportResult importProtocoleVgObs(InputStream stream);

    ImportResult importElementVoisinage(InputStream stream);

    ImportResult importRcesoRulesGroundWater(InputStream stream);

    ImportResult importRcesoFuzzySetGroundWater(InputStream stream);

    ImportResult importRcesoCaseGroundWater(InputStream stream);

    ImportResult importPhytoSubstanceActiveIphy(InputStream stream);

    ImportResult importRcesuRunoffPotRulesParc(InputStream stream);

    ImportResult importTypeAgriculture(InputStream stream);
 
    ImportResult importActaDosageSpc(InputStream stream);

    ImportResult importActaGroupeCultures(InputStream stream);

    ImportResult importLienCulturesEdiActa(InputStream stream);

    ImportResult importSaActaIphy(InputStream stream);
    
    ImportResult importTraitSdC(InputStream stream);

    ImportResult importCouvSolAnnuelle(InputStream stream);

    ImportResult importCultureEdiGroupeCouvSol(InputStream stream);

    ImportResult importCouvSolPerenne(InputStream stream);
    
    ImportResult importZoneClimatiqueIphy(InputStream stream);
}
