package fr.inra.agrosyst.api.services.practiced;

/*
 * #%L
 * Agrosyst :: API
 * $Id: PracticedSystemService.java 4972 2015-06-09 13:14:05Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-api/src/main/java/fr/inra/agrosyst/api/services/practiced/PracticedSystemService.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.Price;
import fr.inra.agrosyst.api.entities.practiced.PracticedPerennialCropCycle;
import fr.inra.agrosyst.api.entities.practiced.PracticedSeasonalCropCycle;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystem;
import fr.inra.agrosyst.api.services.AgrosystService;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.domain.CroppingPlanSpeciesDto;
import fr.inra.agrosyst.api.services.domain.ToolsCouplingDto;
import fr.inra.agrosyst.api.services.pz0.ImportResults;

import java.io.InputStream;
import java.util.List;
import java.util.Map;

/**
 * Practiced system service.
 *
 * @author Eric Chatellier
 */
public interface PracticedSystemService extends AgrosystService {

    /**
     * Return the practicedSystem matching the given TopiaId key if null a new empty instance is created.
     *
     * @param practicedSystemTopiaId the PracticedSystem's TopiaId can be null
     * @return the entity
     */
    PracticedSystem getPracticedSystem(String practicedSystemTopiaId);

    /**
     * Return the list of Plots filtered according the given filter.
     *
     * @param filter the Plot filter
     * @return the filtered plots
     */
    ResultList<PracticedSystem> getFilteredPracticedSystems(PracticedSystemFilter filter);

    /**
     * Return a list of Plot's DTOs filtered according the given filter.
     * @param filter the Plot filter
     * @return the filtered dto plots
     */
    ResultList<PracticedSystemDto> getFilteredPracticedSystemsDto(PracticedSystemFilter filter);

    //boolean areCampaignsValid(String campaigns);

    List<String> getToolsCouplingsFromGrowingSystemAndCampaigns(String growingSystemId, String campaigns);

    /**
     *
     * @param domainCode
     * @param campaigns
     * @return
     */
    List<String> getToolsCouplingsCodesFromDomainAndCampaigns(String domainCode, String campaigns);

    /**
     * Return all Cropping Plan Entry code present on domain related to the growing system that match the given growing system id and for the given campaigns.
     * @param growingSystemId the growing system's id
     * @param campaigns the domain campaigns
     * @return all Cropping Plan Entry code for the growing system's domain and for the given campaigns
     */
    List<String> getCropCodesFromGrowingSystemIdForCampaigns(String growingSystemId, String campaigns);

    /**
     * Return all Cropping Plan Entry code present on domains with same code as the given domainCode and for the given campaigns.
     * @param domainCode domain's code
     * @param campaigns domain's campaigns
     * @return all Cropping Plan Entry code for the domain and for the given campaigns
     */
    List<String> getCropCodesFromDomainCodeForCampaigns(String domainCode, String campaigns);

    /**
     * Retourne les modeles des cultures.
     *
     * @param growingSystemId
     * @param campaigns
     * @param includeIntermediate
     * @param includeCropsFromInactiveDomains
     * @return
     */
    Map<CropCycleModelDto, List<CroppingPlanSpeciesDto>> getCropCycleModelMap(String growingSystemId, String campaigns, boolean includeIntermediate, boolean includeCropsFromInactiveDomains);

    /**
     * Retourne les modeles des cultures principales (sans les cultures intermédiaires) du système de culture passé en paramètre
     * et pour les campagnes passée en paramètres.
     *
     * @param growingSystemId     Growing sytem identifiant's
     * @param campaigns           The growing system campain's.
     * @param includeIntermediate Shall intermediate cropping entry be included ?
     * @return all crops models
     */
    List<CropCycleModelDto> getCropCycleModel(String growingSystemId, String campaigns, boolean includeIntermediate, boolean includeCropsFromInactiveDomains);

    /**
     *
     * @param cycleId The cycle id where node come from
     * @return all node from the cycle matching the given id. Nodes are mapped into DTOs
     */
    Map<PracticedCropCycleNodeDto, List<CroppingPlanSpeciesDto>> getNodes(String cycleId);

    /**
     * Create or update practicedPerennialCropCycle.
     *
     * @param practicedSystem                 The practicedPerennialCropCycle's PracticedSystem
     * @param practicedPerennialCropCycleDtos All the PracticedPerennialCropCycleDtos
     * @param practicedSeasonalCropCycleDtos  All the PracticedSeasonalCropCycleDto
     * @param prices                          The list of prices associated with this practicedSystem
     * @return practiced system
     */
    PracticedSystem createOrUpdatePracticedSystem(PracticedSystem practicedSystem,
                                                  List<PracticedPerennialCropCycleDto> practicedPerennialCropCycleDtos,
                                                  List<PracticedSeasonalCropCycleDto> practicedSeasonalCropCycleDtos,
                                                  List<Price> prices);

    /**
     * Create a list of toolsCouplingDto from the domain associated to the growing system and the selected campains that have the same interventionType.
     *
     * @param growingSystemId  Growing system id use to find tools coupling
     * @param campaigns        The growing system campain's.
     * @param interventionType The tools coupling required InterventionType
     * @return All tools coupling mapped into DTo found from frowing system's domain matching the given interventionType
     */
    List<ToolsCouplingDto> getToolsCouplingModel(String growingSystemId, String campaigns, AgrosystInterventionType interventionType);

    /**
     *
     * @param cycleId practicedSeasonalCropCycle id, if null a new instance is created
     * @return the PracticedSeasonalCropCycle matching the given cycleId
     */
    PracticedSeasonalCropCycle getPracticedSeasonalCropCycle(String cycleId);

    /**
     * Return All practicedPerennialCropCycle related to the PracticedSystem
     *
     * @param practicedSystemId the PracticedSystem's topiaId where to find the practicedPerennialCropCycle
     * @return All practicedPerennialCropCycle
     */
    List<PracticedPerennialCropCycleDto> getAllPracticedPerennialCropCycles(String practicedSystemId);

    /**
     * Return All PracticedSeasonalCropCycle related to the PracticedSystem
     *
     * @param practicedSystemId the PracticedSystem's topiaId where to find the PracticedSeasonalCropCycle
     * @return All PracticedSeasonalCropCycle
     */
    List<PracticedSeasonalCropCycleDto> getAllPracticedSeasonalCropCycles(String practicedSystemId);

    /**
     * Méthode qui charge la liste des espèces disponibles pour le cycle passé en paramètre et remplit les Dto avec les
     * informations déjà saisie dans ce cycle
     *
     * @param croppingPlanEntryCode l'identifiant de la culture à traiter
     * @param cycle                 le cycle pérenne
     * @param campaigns             la liste des campagnes concernées (issues du système synthétisé)
     * @return la liste complète des espèces complétée des informations liées au cycle
     */
    List<PracticedCropCycleSpeciesDto> getCropCyclePerennialSpecies(String croppingPlanEntryCode, PracticedPerennialCropCycle cycle, String campaigns);

    /**
     * return the name of the cropping plan
     *
     * @param croppingPlanEntryCode the cropping plan's code, if null a new instance is created but not persisted yed
     * @return the cropping plan's name
     */
    String getCroppingPlanEntryName(String croppingPlanEntryCode);

    /**
     * Return practicedPerennialCropCycle with the given cycleId
     * @param cycleId if of cycle
     * @return the practicedPerennialCropCycle found
     */
    PracticedPerennialCropCycle getPracticedPerennialCropCycle(String cycleId);

    /**
     * Loads the prices involved in the given practiced system ITK
     *
     * @param practicedSystemId the identifier of the practiced system it is about
     * @return the list of prices. This list is entirely separated from the on in domain's as it relies on several campaigns
     */
    List<Price> getPracticedPrices(String practicedSystemId);

    /**
     * Duplicate practiced system with id practicedSystemId on the target growing system with id the growingSystemId given has parameter.
     *
     * @param practicedSystemId practiced system id
     * @param growingSystemId
     * @return duplicated practiced system
     */
    PracticedSystem duplicatePracticedSystem(String practicedSystemId, String growingSystemId);

    /**
     * Unactivate or reactivate the given practiced System
     *
     * @param practicedSystemIds    :   Practiced System topiaId to unactivate/reactivate
     * @param activate              :   if "true", reactivate practiced systems, else unactivate.
     */
    void unactivatePracticedSystem(List<String> practicedSystemIds, boolean activate);

    /**
     * return the Domain code from the domain related to the growing system with id as growingSystemId
     * @param growingSystemId growing system Id
     * @return the DomainKeys
     */
    String getdomainCode(String growingSystemId);

    /**
     * Set valid true to practiced system
     * @param practicedSystemId The practiced system id to valid
     * @return the validate practiced system
     */
    PracticedSystem validate(String practicedSystemId);

    /**
     * Export cycle
     * @param effectiveCropCycleIds cycle id's to export
     * @return exported cycles input stream
     */
    InputStream exportPracticedSystemsAsXlsStream(List<String> effectiveCropCycleIds);

    /**
     * Import practiced System from PZ0
     * @param allResults imported data from PZ0 witch include practiced systems
     */
    void importPZ0PracticedSystems(Map<Class, ImportResults> allResults);
}
