package fr.inra.agrosyst.api.services.plot;

/*
 * #%L
 * Agrosyst :: API
 * $Id: PlotService.java 5111 2015-10-26 11:24:32Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-api/src/main/java/fr/inra/agrosyst/api/services/plot/PlotService.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.Plot;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.measure.MeasurementSession;
import fr.inra.agrosyst.api.services.AgrosystService;
import fr.inra.agrosyst.api.services.common.UsageList;
import fr.inra.agrosyst.api.services.domain.ExtendContext;
import fr.inra.agrosyst.api.services.domain.PlotDto;
import fr.inra.agrosyst.api.services.pz0.ImportResults;

import java.io.InputStream;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * @author David Cossé : cosse@codelutin.com
 */
public interface PlotService extends AgrosystService {

    /**
     * Return the plot with reference, the given TopiaId key.
     * If plot does not exists a new one is created, not persisted yet
     * @param plotTopiaId the Plots's TopiaId
     * @return the topiaID
     */
    Plot getPlot(String plotTopiaId);

    /**
     * Create or update the plot
     * @param plot the plot to save
     * @param domainId the plot domain's id
     * @param locationId RefLocation id, may be null
     * @param growingSystemId Growing System id, may be null
     * @param selectedPlotZoningIds all plotZonings, may be null
     * @param selectedSolId Ground id
     * @param selectedSurfaceTextureId surface Texture id, may be null
     * @param selectedSubSoilTextureId Sub soil texture id, may be null
     * @param selectedSolDepthId sol Depth id from RefSolProfondeurIndigo, may be null
     * @param solHorizons plot all sol Horizon, may be null
     * @param zones All plot'szones, at least one is required
     * @param adjacentElementIds all adjacent element ids, may be null
     * @return the persisted plot
     */
    Plot createOrUpdatePlot(Plot plot, String domainId, String locationId, String growingSystemId,
                            Collection<String> selectedPlotZoningIds, String selectedSolId,
                            String selectedSurfaceTextureId, String selectedSubSoilTextureId,
                            String selectedSolDepthId, List<SolHorizonDto> solHorizons,
                            Collection<Zone> zones, List<String> adjacentElementIds);

    /**
     * Find all plots for a specific growing system.
     * 
     * @param growingSystem growing plan
     * @return growing sytem's plot
     */
    List<Plot> findAllByGrowingSystem(GrowingSystem growingSystem);

    /**
     * Find all free plot and plots linked to current system.
     * 
     * @param growingSystem growing system
     * @param growingPlanId growing plan id
     * @return all available plots for current growing system
     */
    List<Plot> findAllFreeAndGrowingSystemPlots(GrowingSystem growingSystem, String growingPlanId);

    /**
     * Retourne les parcelles libres (non affectée à un système de culture) liées au dispositif
     * choisit.
     * 
     * @param growingPlanTopiaId growing plan id
     * @return plots list
     */
    List<Plot> getFreePlotForGrowingPlan(String growingPlanTopiaId);

    /**
     * Duplicate plot for extend domain process.
     * Do not commit transaction.
     * 
     * @param extendContext extend context
     * @param clonedDomain domain to attache plot to
     * @param plot plot
     * @return cloned plot
     */
    Plot extendPlot(ExtendContext extendContext, Domain clonedDomain, Plot plot);

    /**
     * Update Plots relationship with a growingSystem
     * @param growingSystem the growing system
     * @param plotIds All plots TopiaIds related with the growingSystem.
     */
    void updatePlotsGrowingSystemRelationship(GrowingSystem growingSystem,
                                              Collection<String> plotIds);

    /**
     * Find all plot for a specific domain.
     * 
     * @param domain domain
     * @return all domain's plots
     */
    List<Plot> findAllByDomain(Domain domain);

    /**
     * Find all plot's zone.
     * 
     * @param plot plot
     * @return zones
     */
    List<Zone> getPlotZones(Plot plot);

    /**
     * Get zones with map of plot's zones topiaIds with boolean set to true, if zone is used or not into the soft.
     *
     * @param plotTopiaId plot topia id
     * @return
     */
    UsageList<Zone> getZonesAndUsages(String plotTopiaId);

    /**
     * Get all plots related to given code.
     * 
     * @param plotCode plot code
     * @return related plots
     */
    LinkedHashMap<Integer, String> getRelatedPlots(String plotCode);

    /**
     * Get all zones related to given code.
     * 
     * @param zoneCode zone code
     * @return related zones
     */
    LinkedHashMap<Integer, String> getRelatedZones(String zoneCode);

    /**
     * Duplicate given plot topia id and return duplicated plot topia id.
     * 
     * @param topiaId plot topia id to duplicate
     * @return duplicated plot topia id
     */
    Plot duplicatePlot(String topiaId);

    /**
     * Unactivate plot and return unactivated plot.
     * 
     * @param plotTopiaId plot id to unactivate
     * @param activate activate or unactivate plot
     * @return unactivated plot
     */
    Plot unactivatePlot(String plotTopiaId, boolean activate);

    boolean validMergingPlots(List<String> plotTopiaIds);

    /**
     * Merge given plot id into a single plot.
     * 
     * Remaining plot is plot with bigger surface.
     * 
     * @param plotTopiaIds
     * @return
     */
    Plot mergePlots(List<String> plotTopiaIds);

    /**
     * Retourne la liste des observations et mesures associées à la parcelle.
     * 
     * @param plotTopiaId plot topia id
     * @return plot's measurement
     */
    List<MeasurementSession> getPlotMeasurementSessions(String plotTopiaId);

    /**
     * Retourne la liste des autres zones attachées au domaine de la zone dont l'identifiant
     * et l'identifiant de zone passée en paramètre.
     *
     * @param zoneId identifiant de la zone
     * @return les autres zones du domaine de la zone dont l'identifiant est passé en paramètre.
     */
    List<Zone> getZonesWithoutCycle(String zoneId);

    List<PlotDto> getPlots(Collection<String> plotIds);

    /**
     * Export selected effective plots id as excel sheet.
     * 
     * @param plotIds zone ids
     * @return stream
     */
    InputStream exportPlotsAsXlsStream(List<String> plotIds);

    /**
     * Import effective plots from stream.
     * 
     * @param is input stream
     */
    void importPlotsForXlsStream(InputStream is);

    /**
     * return plots that target the given growingSystem
     * @param growingSystem The growingSystem targeted bay the plots
     * @return All plots targeted the given growingSystem.
     */
    List<Plot> getAllGrowingSystemPlot(GrowingSystem growingSystem);

    /**
     * Plot PZ0 import
     * @param allResults import parsing result
     */
    void importPZ0Plots(Map<Class, ImportResults> allResults);
}
