package fr.inra.agrosyst.api.services.network;

/*
 * #%L
 * Agrosyst :: API
 * $Id: NetworkService.java 4009 2014-04-16 12:11:09Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-api/src/main/java/fr/inra/agrosyst/api/services/network/NetworkService.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.io.InputStream;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Set;

import fr.inra.agrosyst.api.entities.Network;
import fr.inra.agrosyst.api.entities.NetworkManager;
import fr.inra.agrosyst.api.services.AgrosystService;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.referential.ImportResult;

/**
 * @author cosse
 */
public interface NetworkService extends AgrosystService {

    /**
     * Return the Networks according the given filter.
     *
     * @param filter The Network filter
     * @return the networks
     */
    ResultList<Network> getFilteredNetworks(NetworkFilter filter);

    /**
     * Return the Network matching with the network topiaId given has parameter.
     *
     * @param networkTopiaId the network topiaId
     * @return the network
     */
    Network getNetwork(String networkTopiaId);

    /**
     * Return a new not persisted Network.
     *
     * @return the network
     */
    Network newNetwork();

    /**
     * Return a new not persisted Network Manager.
     *
     * @return the network Manager
     */
    NetworkManager newNetworkManager();

    /**
     * Update the given network
     *
     * @param network            the network to update
     * @param networkManagerDtos the network managers DTO
     * @param parentsIds         List of network's parent's TopiaIds
     * @return the updated network
     */
    Network createOrUpdateNetwork(Network network, Collection<NetworkManagerDto> networkManagerDtos, List<String> parentsIds);

    /**
     * Return all the network with there name looking as the researched term.
     *
     * @param research         The researched term
     * @param nbResult         The maximum number if return result
     * @param exclusions       The identifiers of the networks to exclude
     * @param selfNetworkId    The identifier of the current network (in case editing a network)
     * @param onNetworkEdition If the current request should consider only networks without SdC children or not
     * @return All the network with there name looking as the researched term.
     */
    LinkedHashMap<String, String> searchNameFilteredActiveNetworks(String research, Integer nbResult,
                                                                   Set<String> exclusions, String selfNetworkId,
                                                                   boolean onNetworkEdition);

    Set<String> findNetworksByName(String name, String excludeNetworkId);

    NetworkGraph buildFullNetworkGraph();

    NetworkGraph buildNetworkGraph(String fromNetworkId);

    NetworkGraph buildGrowingSystemAndNetworkGraph(String growingSystemName, Set<String> parentNetworkIds);

    /**
     * Unactivate or reactivate the given networks.
     *
     * @param networkIds networkIds topiaId to unactivate
     * @param activate   activate insteadof unactivate
     */
    void unactivateNetworks(List<String> networkIds, boolean activate);

    /**
     * return the number of networks
     *
     * @return The number of networks
     * @param active optional active filter (may be null)
     */
    long getNetworksCount(Boolean active);

    /**
     * Import de réseaux par lot.
     * 
     * @param netStream
     * @return
     */
    ImportResult importNetworks(InputStream netStream);

    NetworkIndicators getIndicators(String networkId);

    List<Network> getNteworkWithName(String networkName);
}
