package fr.inra.agrosyst.api.services.network;

/*
 * #%L
 * Agrosyst :: API
 * $Id: NetworkGraph.java 4009 2014-04-16 12:11:09Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-api/src/main/java/fr/inra/agrosyst/api/services/network/NetworkGraph.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import fr.inra.agrosyst.api.entities.Network;

/**
 * @author Arnaud Thimel : thimel@codelutin.com
 */
public class NetworkGraph implements Serializable {

    private static final long serialVersionUID = 734183498915994008L;
    protected static final String GROWING_SYSTEM_AS_A_NETWORK_ID = "growing-system_" + UUID.randomUUID().toString();

    protected List<List<SmallNetworkDto>> networks;
    protected Set<NetworkConnectionDto> connections;

    public NetworkGraph() {
        networks = Lists.newArrayList();
        connections = Sets.newHashSet();
    }

    protected int baseRow = 0;

    public List<List<SmallNetworkDto>> getNetworks() {
        return networks;
    }

    public void setNetworks(List<List<SmallNetworkDto>> networks) {
        this.networks = networks;
    }

    public Set<NetworkConnectionDto> getConnections() {
        return connections;
    }

    public void setConnections(Set<NetworkConnectionDto> connections) {
        this.connections = connections;
    }

    public void addNetwork(Network network) {
        SmallNetworkDto dto = new SmallNetworkDto(network);
        int level = findLevel(dto);
        if (level == -1) {
            level = baseRow;
            addNetworkToLevel(level, dto);
        }
        Collection<Network> parents = network.getParents();
        if (parents != null) {
            for (Network parent : parents) {
                SmallNetworkDto parentDto = new SmallNetworkDto(parent);
                int parentLevel = findLevel(parentDto);
                if (parentLevel == -1) { // Non présent
                    parentLevel = level - 1;
                    addNetworkToLevel(parentLevel, parentDto);
                } else { // Déjà présent
                    if (parentLevel < level) {
                        // Nothing to do, parent is already before in hierarchy
                    } else {
                        parentLevel = level - 1;
                        moveNetworkToLevel(parentLevel, parentDto.getId());
                    }
                }
                addConnection(network.getTopiaId(), parent.getTopiaId());
            }
        }
    }

    public void addGrowingSystem(String name, Set<Network> parents) {
        SmallNetworkDto dto = new SmallNetworkDto(GROWING_SYSTEM_AS_A_NETWORK_ID, name);
        int level = findLevel(dto);
        if (level == -1) {
            level = baseRow;
            addNetworkToLevel(level, dto);
        }
        if (parents != null) {
            for (Network parent : parents) {
                SmallNetworkDto parentDto = new SmallNetworkDto(parent);
                int parentLevel = findLevel(parentDto);
                if (parentLevel == -1) { // Non présent
                    parentLevel = level - 1;
                    addNetworkToLevel(parentLevel, parentDto);
                } else { // Déjà présent
                    if (parentLevel < level) {
                        // Nothing to do, parent is already before in hierarchy
                    } else {
                        parentLevel = level - 1;
                        moveNetworkToLevel(parentLevel, parentDto.getId());
                    }
                }
                addConnection(GROWING_SYSTEM_AS_A_NETWORK_ID, parent.getTopiaId());
            }
        }
    }

    protected void addNetworkToLevel(int level, SmallNetworkDto network) {
        if (networks.isEmpty()) {
            networks.add(new ArrayList<SmallNetworkDto>());
        }
        if (level <= -1) {
            level = 0;
            baseRow++;
            networks.add(0, new ArrayList<SmallNetworkDto>());
        }
        while (level >= networks.size()) {
            baseRow++;
            networks.add(0, new ArrayList<SmallNetworkDto>());
        }
        networks.get(level).add(network);
    }

    protected void moveNetworkToLevel(int level, final String networkId) {
        int wasAtLevel = findLevel(networkId);
        List<SmallNetworkDto> networksAtLevel = networks.get(wasAtLevel);
        SmallNetworkDto foundNetwork = Iterables.find(networksAtLevel, Predicates.equalTo(new SmallNetworkDto(networkId, null)));
        networksAtLevel.remove(foundNetwork);
        addNetworkToLevel(level, foundNetwork);

        Iterable<NetworkConnectionDto> allParents = Iterables.filter(connections, new Predicate<NetworkConnectionDto>() {
            @Override
            public boolean apply(NetworkConnectionDto input) {
                return networkId.equals(input.getSourceId());
            }
        });
        for (NetworkConnectionDto connectionDto : allParents) {
            moveNetworkToLevel(level - 1, connectionDto.getTargetId());
        }
    }

    public void addConnection(String sourceId, String targetId) {
        NetworkConnectionDto connection = new NetworkConnectionDto(sourceId, targetId);
        this.connections.add(connection);
    }

    protected int findLevel(String networkId) {
        SmallNetworkDto network = new SmallNetworkDto(networkId, null);
        return findLevel(network);
    }

    protected int findLevel(final SmallNetworkDto network) {
        int level = 0;
        for (List<SmallNetworkDto> networkLevel : networks) {
            Optional<SmallNetworkDto> optional = Iterables.tryFind(networkLevel, Predicates.equalTo(network));
            if (optional.isPresent()) {
                return level;
            }
            level++;
        }
        return -1;
    }

}
