package fr.inra.agrosyst.api.services.managementmode;

/*
 * #%L
 * Agrosyst :: API
 * $Id: ManagementModeService.java 4009 2014-04-16 12:11:09Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-api/src/main/java/fr/inra/agrosyst/api/services/managementmode/ManagementModeService.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.BioAgressorType;
import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.managementmode.DecisionRule;
import fr.inra.agrosyst.api.entities.managementmode.ManagementMode;
import fr.inra.agrosyst.api.entities.managementmode.ManagementModeCategory;
import fr.inra.agrosyst.api.entities.referential.RefBioAgressor;
import fr.inra.agrosyst.api.services.AgrosystService;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.domain.ExtendContext;

import java.io.InputStream;
import java.util.Collection;
import java.util.List;

public interface ManagementModeService extends AgrosystService {

    /**
     * Return the DecisionRules according the given filter.
     *
     * @param filter The DecisionRule filter
     * @return the decisionRules
     */
    ResultList<DecisionRulesDto> getFilteredDecisionRules(DecisionRuleFilter filter);

    /**
     * New decision rule instance.
     *
     * @return new decision rule instance
     */
    DecisionRule newDecisionRule();

    /**
     * Find decision rule instance by id.
     *
     * @param decisionRuleTopiaId topia id
     * @return decision rule
     */
    DecisionRule getDecisionRule(String decisionRuleTopiaId);

    /**
     * Return an exact copie of the rule
     *
     * @param decisionRuleCode the code of the rule
     * @return the copy
     */
    DecisionRule getLastDecisionRuleVersion(String decisionRuleCode);

    /**
     * Get bio agressor types.
     *
     * @return bio agressor types
     */
    List<BioAgressorType> getBioAgressorTypes();

    /**
     * Find all agressor from specified type.
     *
     * @param bioAgressorType bio agressor type
     * @return bio agressors
     */
    <E extends RefBioAgressor> List<E> getBioAgressors(BioAgressorType bioAgressorType);

    /**
     * Find all cropping plan entries for domains with code as given in parameter
     * @param domainCode
     * @return
     */
    List<CroppingPlanEntry> getDomainCodeCroppingPlanEntries(String domainCode);

    /**
     * Find all cropping plan entries corresponding to
     *
     * @param growingSystemTopiaId growing system topia id
     * @return cropping plan entries
     */
    List<CroppingPlanEntry> getGrowingSystemCroppingPlanEntries(String growingSystemTopiaId);

    CroppingPlanEntry getCroppingPlanEntries(String croppingPlanEntryTopiaId);

    /**
     * Update decision rule.
     *
     * @param decisionRule
     * @param domainCode
     * @param croppingPlanEntryCode
     * @param bioAgressorTopiaId
     * @return updated decision rule
     */
    DecisionRule createOrUpdateDecisionRule(DecisionRule decisionRule, String domainCode, String croppingPlanEntryCode, String bioAgressorTopiaId);

    /**
     * Create decision rule with params given has parameters

     * @param interventionType
     * @param growingSystemTopiaId
     * @param bioAgressorType
     * @param croppingPlanEntryId
     * @param name
     * @return
     */
    DecisionRule createNewDecisionRule(
            AgrosystInterventionType interventionType,
            String growingSystemTopiaId,
            BioAgressorType bioAgressorType,
            String bioAgressorTopiaId,
            String croppingPlanEntryId,
            String name);
    /**
     * Return the ManagementModes according the given filter.
     *
     * @param managementModeFilter The ManagementMode filter
     * @return the managementModes
     */
    ResultList<ManagementModeDto> getFilteredManagementModeDtos(ManagementModeFilter managementModeFilter);

    /**
     * Create new management mode instance for creation.
     *
     * @return management mode instance
     */
    ManagementMode newManagementMode();

    /**
     * Find management mode instance from id.
     *
     * @param managementModeTopiaId topia id
     * @return management mode instance
     */
    ManagementMode getManagementMode(String managementModeTopiaId);

    /**
     * Update management mode.
     *
     * @param managementMode       management mode to update
     * @param growingSystemTopiaId growing system id
     * @param sections             section dto
     * @return updated management mode
     */
    ManagementMode createOrUpdateManagementMode(ManagementMode managementMode, String growingSystemTopiaId, Collection<SectionDto> sections);

    /**
     * Get growing system's decision rules.
     *
     * @param growingSystemTopiaId growing system id
     * @return decision rules
     */
    Collection<DecisionRule> getGrowingSystemDecisionRules(String growingSystemTopiaId);

    /**
     * Find management mode by growing system.
     *
     * @param growingSystem growing system
     * @return management mode
     */
    ManagementMode getManagementModeByGrowingSystem(GrowingSystem growingSystem);

    /**
     * Extend management mode.
     *
     * @param extendContext  extend context
     * @param managementMode management mode to clone
     * @param clonedGrowingSystem clonedGrowingSystem
     * @return cloned management mode
     */
    ManagementMode extendManagementMode(ExtendContext extendContext,
                                        ManagementMode managementMode,
                                        GrowingSystem clonedGrowingSystem);

    /**
     * Get decision rules version with his related rule's topidId liked to the given rule code
     *
     * @param code the rule code
     * @return the list of version for the rules related to the code.
     */
    List<DecisionRule> getRelatedDecisionRules(String code);

    /**
     * @param decisionRuleCode the code of the source decisionRule
     * @param versionReason    the reason to create this new version
     * @return the created decisionRule
     */
    DecisionRule createNewDecisonRuleVersion(String decisionRuleCode, String versionReason);

    /**
     * Return the categories associated with the GrowingSystem with same topiaId as given in parameter.
     *
     * @param growingSystemId GrowingSystem topiaId
     * @return
     */
    List<ManagementModeCategory> getAvailableManagementModeCategories(String growingSystemId);

    /**
     * Return ManagementMode related to growingSystem given as parameter
     * @param growingSystem the GrowingSystem to witch ManagementModes or attached
     * @return all related ManagementMode
     */
    List<ManagementMode> getRelatedManagementModes(GrowingSystem growingSystem);

    /**
     *
     * @param growingSystemId
     * @param category
     * @return
     */
    ManagementMode copyManagementMode(String growingSystemId, ManagementModeCategory category, String mainChangesFromPlanned, String changeReasonFromPlanned);

    List<GrowingSystem> getGrowingSystemsForManagementMode(NavigationContext navigationContext);

    List<GrowingSystem> getAvailableGsForDuplication(String growingSystemId, NavigationContext navigationContext);

    void unactivateDecisionRules(List<String> decisionRuleIds, boolean activate);

    DecisionRule duplicateDecisionRule(String decisionRuleIds);

    ManagementMode duplicateManagementModes(String plannedManagementModeTopiaId, String observedManagementMode, String duplicateManagementModeGrowingSystemId);

    /**
     * return all decision Rules from topiaIds given as parameters
     * @param decisionRulesIds
     * @return
     */
    List<DecisionRule> getAllDecisionRules(List<String> decisionRulesIds);

    InputStream exportManagementModesAsXlsStream(Collection<ManagementModeDto> managementModeDtos);

    InputStream exportDecisionRulesAsXlsStream(Collection<String> decisionRuleIds);
}
