package fr.inra.agrosyst.api.services.growingsystem;

/*
 * #%L
 * Agrosyst :: API
 * $Id: GrowingSystemService.java 4833 2015-03-13 09:41:48Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-api/src/main/java/fr/inra/agrosyst/api/services/growingsystem/GrowingSystemService.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.GrowingPlan;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.Network;
import fr.inra.agrosyst.api.entities.managementmode.ManagementMode;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystem;
import fr.inra.agrosyst.api.entities.referential.RefTypeAgriculture;
import fr.inra.agrosyst.api.services.AgrosystService;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.domain.ExtendContext;
import fr.inra.agrosyst.api.services.pz0.ImportResults;

import java.io.InputStream;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * @author Arnaud Thimel : thimel@codelutin.com
 */
public interface GrowingSystemService extends AgrosystService {

    /**
     * Return the Growing Systems according the given filter.
     *
     * @param filter The Growing System filter
     * @return The growing systems
     */
    ResultList<GrowingSystem> getFilteredGrowingSystems(GrowingSystemFilter filter);

    /**
     * Return the Growing Systems according the given filter.
     *
     * @param filter The Growing System filter
     * @return The growing systems
     */
    ResultList<GrowingSystemDto> getFilteredGrowingSystemsDto(GrowingSystemFilter filter);

    /**
     * Return Growing Systems according it's id
     * @param growingSystemIds
     * @return
     */
    List<GrowingSystemDto> getGrowingSystems(Collection<String> growingSystemIds);

    /**
     * Find growing system from topia id.
     *
     * @param growingSystemId growing system id
     * @return growing system
     */
    GrowingSystem getGrowingSystem(String growingSystemId);

    /**
     * Create a new (empty) GrowingSystem entity. The returned instance is not persisted yet.
     *
     * @return the newly created instance
     */
    GrowingSystem newGrowingSystem();

    /**
     * Create or Update a GrowingSystem
     *
     * @param growingSystem           The GrowingSystem to create or update.
     * @param growingSystemNetworkIds The networks's growing system.
     * @param typeAgricultureTopiaId type agriculture topia id
     * @return the GrowingSystem
     */
    GrowingSystem createOrUpdateGrowingSystem(GrowingSystem growingSystem, List<String> growingSystemNetworkIds,
            Collection<String> selectedPlotsIds, String typeAgricultureTopiaId, List<String> growingSystemCharacteristicIds);

    /**
     * Find all growing system for a specific growing plan.
     *
     * @param growingPlan growing plan
     * @return growing plan's growing systems
     */
    List<GrowingSystem> findAllByGrowingPlan(GrowingPlan growingPlan);

    /**
     * Find all growing system for a specific network.
     *
     * @param network the network
     * @return network's growing systems
     */
    List<GrowingSystem> findAllByNetwork(Network network);

    /**
     * Find all growing system for a specific domain.
     *
     * @param domain the domain
     * @return the growing systems's
     */
    List<GrowingSystem> findAllActiveByDomainForPlotEdit(Domain domain);

    /**
     * Duplicate a growing system without committing transaction.
     *
     * @param extendContext     extend context
     * @param clonedGrowingPlan growing plan to attach growing system to
     * @param growingSystem     growing system to clone
     * @return duplicated growing system
     */
    GrowingSystem duplicateGrowingSystem(ExtendContext extendContext, GrowingPlan clonedGrowingPlan, GrowingSystem growingSystem);

    /**
     * Unactivate the given growingSystems.
     *
     * @param growingSystemsIds growingSystems ids to unactivate
     * @param activate          activate
     */
    void unactivateGrowingSystems(List<String> growingSystemsIds, boolean activate);

    /**
     * Get growingSystem related to current growingSystem.
     * Related growingSystem got same duplication code.
     *
     * @param growingSystemCode the code identifying the growingSystem
     * @return related growingSystems
     */
    LinkedHashMap<Integer, String> getRelatedGrowingSystems(String growingSystemCode);

    /**
     * Return the number of growingSystems
     *
     * @return The number of growing systems
     * @param active optional active filter (may be null)
     */
    long getGrowingSystemsCount(Boolean active);


    /**
     * Do validate the current growing system state
     *
     * @param growingSystemId the identifier of the growing system to validate
     * @return the validated growing system
     */
    GrowingSystem validate(String growingSystemId);

    /**
     * Retournes tous les types d'agricultures.
     * 
     * @return les types d'agriculture.
     */
    List<RefTypeAgriculture> findAllTypeAgricultures();
    

    /**
     * Recupère la liste des systèmes pratiqués lié à systeme de culture via son code de prolongation.
     * 
     * @param growingSystemCode growing system code
     * @return practiced systems list
     */
    List<PracticedSystem> getPracticedSystems(String growingSystemCode);

    /**
     * Retourne la liste des modes de gestion associés au système de culture.
     * 
     * @param growingSystemId growing system id 
     * @return management modes
     */
    List<ManagementMode> getManagementModes(String growingSystemId);

    /**
     * Return campaigns for growing systems matching the given code
     * @param growingSystemCode Growing system's code
     * @return List of campaigns
     */
    List<Integer> getGrowingSystemCampaigns(String growingSystemCode);

    /**
     * Return all growing systems related to the practiced system's domain withing optional navigationContext criteria, if null no filter is done
     * @param practicedSystemId practiced system id
     * @param navigationContext navigation context may be null,
     * @return all growing systems according filter
     */
    List<GrowingSystem> getAvailableGsForDuplication(String practicedSystemId, NavigationContext navigationContext);

    /**
     * Return growing system matching same name and campaign
     * @param growingSystemName growing system name
     * @param campaign campaign
     * @return All growing system matching filter
     */
    List<GrowingSystem> getGrowingSystemWithNameAndCampaign(String growingSystemName, Integer campaign);

    /**
     * Growing system's xls stream for the given growing system id
     * @param growingSystemIds growing system id
     * @return growing system data export.
     */
    InputStream exportGrowingSystemsAsXlsStream(List<String> growingSystemIds);

    /**
     * return all growing system codes, growing plan codes, domains code matching the same dephy id as the given one, if dephy id is null, null is return.
     * @param dephy dephy id
     * @return list of codes by campaigns,
     *         code[0]: Growing system code
     *         code[1]: Growing plan code
     *         code[2]: Domain code
     */
    LinkedHashMap<Integer, List<String>> getGSCodeGPCodeDomainCodeByCampaignForGrowingSystemDephyId(String dephy);

    /**
     * Growing system PZ0 import
     * @param allResults import parsing result
     */
    void importPZ0GrowingSystems(Map<Class, ImportResults> allResults);
}
