package fr.inra.agrosyst.api.services.growingplan;

/*
 * #%L
 * Agrosyst :: API
 * $Id: GrowingPlanService.java 4795 2015-02-17 06:32:47Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-api/src/main/java/fr/inra/agrosyst/api/services/growingplan/GrowingPlanService.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.io.InputStream;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.GrowingPlan;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.services.AgrosystService;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.domain.ExtendContext;
import fr.inra.agrosyst.api.services.pz0.ImportResults;

public interface GrowingPlanService extends AgrosystService {

    /**
     * Create a new growing plan not persisted yet
     * @return the newly created Growing Plan
     */
    GrowingPlan newGrowingPlan();

    /**
     * Find all growing plan for a domain.
     *
     * @param domain domain
     * @return domain's gronwing plans
     */
    List<GrowingPlan> findAllByDomain(Domain domain);

    /**
     * Find a specific growingPlan from identifier.
     *
     * @param growningPlanId growingPlan id
     * @return growingPlan
     */
    GrowingPlan getGrowingPlan(String growningPlanId);

    /**
     * Update or create a GrowingPlan.
     *
     * @param dispositif growingPlan to update
     * @return growingPlan
     */
    GrowingPlan createOrUpdateGrowingPlan(GrowingPlan dispositif);

    /**
     * Find growingPlans matching user filter and navigation context.
     *
     * @param filter custom user filter
     * @return
     */
    ResultList<GrowingPlan> getFilteredGrowingPlans(GrowingPlanFilter filter);

    /**
     * Find growingPlans matching user filter and navigation context.
     *
     * @param filter custom user filter
     * @return
     */
    ResultList<GrowingPlanDto> getFilteredGrowingPlansDto(GrowingPlanFilter filter);

    /**
     * Unactivate the given growingPlans.
     *
     * @param growingPlanIds growingPlans ids to unactivate
     * @param activate       activate
     */
    void unactivateGrowingPlans(List<String> growingPlanIds, boolean activate);

    /**
     * Duplicate growing plan from id and associate if with given domain id.
     * Copy associated growing systems if needed.
     *
     * @param growingPlanId           growing plan to duplicate
     * @param duplicateDomainId       associate duplicated growing plan to domain
     * @param duplicateGrowingSystems duplicate growing systems option
     */
    GrowingPlan duplicateGrowingPlan(String growingPlanId, String duplicateDomainId, boolean duplicateGrowingSystems);

    /**
     * Duplicate growing plan without committing transaction.
     *
     * @param extendContext           domain extends context
     * @param clonedDomain            cloned domain to attach growing plan to
     * @param growingPlan             plan to clone
     * @param duplicateGrowingSystems also duplicate growing plan's growing systems
     * @return duplicated growing plan
     */
    GrowingPlan duplicateGrowingPlan(ExtendContext extendContext, Domain clonedDomain, GrowingPlan growingPlan, boolean duplicateGrowingSystems);

    /**
     * Get growingPlan related to current growingPlan.
     * Related growingPlan got same duplication code.
     *
     * @param growingPlanCode the code identifying this growingPlan
     * @return related growingPlans
     */
    LinkedHashMap<Integer, String> getRelatedGrowingPlans(String growingPlanCode);

    /**
     * Return the number of growingPlans
     *
     * @return The number of growingPlan
     * @param active optional active filter (may be null)
     */
    long getGrowingPlansCount(Boolean active);

    /**
     * Do validate the current growing plan state
     *
     * @param growingPlanId the identifier of the growing plan to validate
     * @return the validated growing plan
     */
    GrowingPlan validateAndCommit(String growingPlanId);

    /**
     * Retourne les systemes de cultures associé au dispositifs.
     * 
     * @param growingPlanId growing plan id
     * @return growing plan's growing systems
     */
    List<GrowingSystem> getGrowingPlanGrowingSystems(String growingPlanId);

    List<GrowingPlan> getGrowingPlanWithName(String growingPlanName);

    InputStream exportGrowingPlanAsXlsStream(List<String> growingPlanIds);

    /**
     * Import growing plans from stream.
     * 
     * @param is input stream
     */
    void importGrowingPlanForXlsStream(InputStream is);

    void importPZ0GrowingPlans(Map<Class, ImportResults> allResults);
}
