package fr.inra.agrosyst.api.services.effective;

/*
 * #%L
 * Agrosyst :: API
 * $Id: EffectiveCropCycleService.java 5004 2015-06-23 14:07:24Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-api/src/main/java/fr/inra/agrosyst/api/services/effective/EffectiveCropCycleService.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Function;
import com.google.common.base.Strings;
import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.Price;
import fr.inra.agrosyst.api.entities.ToolsCoupling;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.services.AgrosystService;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.domain.ToolsCouplingDto;
import fr.inra.agrosyst.api.services.domain.ZoneDto;
import fr.inra.agrosyst.api.services.pz0.effective.EffectiveCropCycleAndDependencies;

import java.io.InputStream;
import java.util.Collection;
import java.util.List;

/**
 * Service de gestion des interventions culturales (et des cycles liés à une parcelle).
 * 
 * @author Eric Chatellier
 */
public interface EffectiveCropCycleService extends AgrosystService {


    Function<ToolsCoupling, String> GET_TOOLS_COUPLING_BY_CODE = new Function<ToolsCoupling, String>() {
        @Override
        public String apply(ToolsCoupling input) {
            return Strings.nullToEmpty(input.getCode());
        }
    };

    /**
     * Recherche paginée de la liste des zones des parcelles.
     * 
     * @param filter filtre de pagination
     * @return plot list
     */
    ResultList<Zone> getFilteredZones(EffectiveZoneFilter filter);

    ResultList<ZoneDto> getFilteredZonesDto(EffectiveZoneFilter filter);

    /**
     * Recherche paginée de la liste des zones des parcelles avec les infos sur cultures et interventions.
     * 
     * @param filter filtre de pagination
     * @return plot list
     */

    ResultList<ZoneDto> getFilteredZonesAndCroppingPlanInfosDto(EffectiveZoneFilter filter);
        
    /**
     * Find zone by topia id.
     * 
     * @param zoneTopiaId zone topia id
     * @return zone instance
     */
    Zone getZone(String zoneTopiaId);

    /**
     * Find zone's domain cropping plan entries.
     * 
     * @param zone zone
     * @return zone's domain cropping plan entries
     */
    List<CroppingPlanEntry> getZoneCroppingPlanEntries(Zone zone);

    List<CroppingPlanEntry> getZoneCroppingPlanEntriesWithoutDomain(Zone zone);

    /**
     * Find cropping plan entry of last node defined for previous campaign.
     * 
     * @param zoneTopiaId zone topia id
     * @return previous campaign last cropping plan
     */
    CroppingPlanEntry getPreviousCampaignCroppingPlanEntry(String zoneTopiaId);

    /**
     * Get all zone's perennial crop cycles (with phases, species, and interventions).
     * 
     * @param zoneTopiaId zoneTopiaId
     * @return zone's perennial crop cycles
     */
    List<EffectivePerennialCropCycleDto> getAllEffectivePerennialCropCyclesDtos(String zoneTopiaId);

    /**
     * Get all plot's seasonal crop cycles (with nodes, connections, and interventions).
     * 
     * @param zoneTopiaId zoneTopiaId
     * @return zone's seasonal crop cycles
     */
    List<EffectiveSeasonalCropCycleDto> getAllEffectiveSeasonalCropCyclesDtos(String zoneTopiaId);

    /**
     * Update zone's crop cycles.
     *
     * @param zoneId the identifier of the zone on which the crop cycle is about
     * @param effectiveSeasonalCropCycles seasonal crop cycles
     * @param effectivePerennialCropCycles perennial crop cycles
     * @param prices
     */
    void updateEffectiveCropCycles(String zoneId,
                                   List<EffectiveSeasonalCropCycleDto> effectiveSeasonalCropCycles,
                                   List<EffectivePerennialCropCycleDto> effectivePerennialCropCycles,
                                   List<Price> prices);

    /**
     * Loads the prices involved in the given zone's effective crop cycle
     *
     * @param zoneId the identifier of the zone on which the crop cycle is about
     * @return the list of prices. This is a sublist of the domain's prices with only the prices used in this cycle
     */
    List<Price> getEffectivePrices(String zoneId, List<String> objectIds);

    List<ToolsCouplingDto> getToolsCouplingModel(String zoneTopiaId, AgrosystInterventionType interventionType);

    void duplicateEffectiveCropCycles(String fromZoneId, String toZoneId);

    List<ZoneDto> getZones(Collection<String> zoneIds);

    InputStream exportEffectiveCropCyclesAsXlsStream(List<String> effectiveCropCycleIds);

    /**
     *
     * @param zoneTopiaId the zone from where the copy is done
     * @return an object containing all zone where interventions can be copied.
     */
    CopyPasteZoneByCampaigns getAvailableZonesForCopy(String zoneTopiaId);

    /**
     * Copy interventions to the given zone crops
     * @param zones define the intervention targeted crop
     * @param interventionDtos interventions to copy
     */
    void copyInterventions(List<TargetedZones> zones, List<EffectiveInterventionDto> interventionDtos);

    /**
     * Import effective crop cycles from PZ0
     * @param effectiveCropCycleAndDependencies imported data from PZ0 witch include effective crop cycles
     */
    void importPZ0EffectiveCropCycles(Collection<EffectiveCropCycleAndDependencies> effectiveCropCycleAndDependencies);
}
