package fr.inra.agrosyst.api.services.domain;

/*
 * #%L
 * Agrosyst :: API
 * $Id: DomainService.java 4977 2015-06-11 04:13:34Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-api/src/main/java/fr/inra/agrosyst/api/services/domain/DomainService.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.CroppingPlanSpecies;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.Equipment;
import fr.inra.agrosyst.api.entities.GeoPoint;
import fr.inra.agrosyst.api.entities.Ground;
import fr.inra.agrosyst.api.entities.Price;
import fr.inra.agrosyst.api.entities.ToolsCoupling;
import fr.inra.agrosyst.api.entities.WeatherStation;
import fr.inra.agrosyst.api.entities.referential.RefInterventionAgrosystTravailEDI;
import fr.inra.agrosyst.api.entities.referential.RefLegalStatus;
import fr.inra.agrosyst.api.entities.referential.RefStationMeteo;
import fr.inra.agrosyst.api.services.AgrosystService;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.common.UsageList;
import fr.inra.agrosyst.api.services.pz0.ImportResults;

import org.apache.commons.lang3.tuple.Pair;

import java.io.InputStream;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Domain service.
 *
 * @author Eric Chatellier
 */
public interface DomainService extends AgrosystService {

    String NEW_EQUIPMENT = "NEW-EQUIPMENT-";

    /**
     * Find all domain.
     *
     * @return all domains
     */
    List<Domain> getAllDomains();

    /**
     * Create a new (empty) Domain entity. the returned instance is not persisted yet.
     *
     * @return the newly created instance
     */
    Domain newDomain();

    /**
     * Create a new (empty) GPSData entity. the return instance is not persited yed.
     *
     * @return the newly created instance
     */
    GeoPoint newGpsData();

    /**
     * Creates a new (empty) {@link WeatherStation}.
     * <p>
     * This returned instance is not persisted yet.
     * </p>
     *
     * @return the newly created {@link WeatherStation} instance
     */
    WeatherStation newWeatherStation();

    /**
     * Find a specific domain from identifier.
     *
     * @param domainId domain id
     * @return domain
     */
    Domain getDomain(String domainId);

    /**
     * Find random domain by code.
     *
     * @param domainCode domain code
     * @return random domain for code
     */
    DomainDto getDomainByCode(String domainCode);

    /**
     *
     * @param domain                domain
         * @param locationId        RefLocation's topiaID  associated to the domain
         * @param legalStatusId     RefLegalStatus topiaId
         * @param gpsData           Domain's geoPoints
         * @param croppingPlan      Domain's crops
         * @param otex18            otext 18 code
         * @param otex70            otex 70 code
         * @param grounds           Domain's ground
         * @param equipments        Domain"s Equipment
         * @param toolsCouplings    Domain's tools couplings
         * @param prices            Domain's crop's and fuel prices
         * @return persisted domain
     */
    Domain createOrUpdateDomain(Domain domain, String locationId, String legalStatusId, List<GeoPoint> gpsData,
                                List<CroppingPlanEntryDto> croppingPlan, Integer otex18, Integer otex70,
                                List<Ground> grounds, List<Equipment> equipments, List<ToolsCoupling> toolsCouplings, List<Price> prices);

    /**
     * Find domain matching user filter and navigation context.
     *
     * @param filter custom user filter
     * @return matching domains
     */
    ResultList<Domain> getFilteredDomains(DomainFilter filter);

    List<Domain> getDomainWithName(String name);

    /**
     * Find domain matching user filter and navigation context.
     *
     * @param filter custom user filter
     * @return matching domains
     */
    ResultList<DomainDto> getFilteredDomainsDto(DomainFilter filter);

    List<DomainDto> getDomains(Collection<String> domainIds);

    /**
     * Unactivate or reactivate the given domains.
     *
     * @param domainIds domains topiaId to unactivate
     */
    void unactivateDomains(List<String> domainIds, boolean activate);

    /**
     * Find all LegalStaus
     *
     * @return all LegalStaus
     */
    List<RefLegalStatus> getAllRefLegalStatus();

    /**
     * Find the {@link RefStationMeteo} instance by a given {@link RefStationMeteo#getTopiaId()}.
     *
     * @param topiaId {@link RefStationMeteo} ID ({@link RefStationMeteo#getTopiaId()})
     * @return the {@link RefStationMeteo} instance
     */
    RefStationMeteo findRefStationMeteoByTopiaId(String topiaId);

    /**
     * Create new instance of domain materiel.
     *
     * @return new domain materiel instance
     */
    Equipment newMateriel();

    /**
     * Create sol instance.
     *
     * @return new sol instance
     */
    Ground newSol();

    /**
     * Extend domain for specified campaign.
     *
     * @param domainTopiaId  domain to duplicate
     * @param extendCampaign extend domain campaign
     * @return duplicated domain
     */
    Domain extendDomain(String domainTopiaId, int extendCampaign) throws DomainExtendException;

    /**
     * Get domain related to current domain.
     * Related domain got same duplication code.
     *
     * @param domainCode the code identifying the domain
     * @return related domains
     */
    LinkedHashMap<Integer, String> getRelatedDomains(String domainCode);

    /**
     * Create ToolsCoupling instance.
     *
     * @return new ToolsCoupling instance
     */
    ToolsCoupling newToolsCoupling();

    /**
     * return GeoPoint for domain with id the given domainId
     * Data are anonymize if user has no right to see them
     * @param domainId domain'id
     * @return all geopoints for the given domainId
     */
    List<GeoPoint> getGeoPoints(String domainId);

    /**
     *
     * @param domainId domain'id
     * @return All equipements for domain with id the given domainId
     */
    List<Equipment> getEquipments(String domainId);

    /**
     *
     * @param domainId domain'id
     * @return All Ground for domain with id the given domainId
     */
    List<Ground> getGrounds(String domainId);

    /**
     *
     * @param domainId domain'id
     * @return All ToolsCoupling for domain with id the given domainId
     */
    List<ToolsCoupling> getToolsCouplings(String domainId);

    /**
     * The tools coupling UsageList from domain with id the given domainId.
     * @param domainId domain'id
     * @return All ToolsCoupling and there usage status for domain with id the given domainId
     */
    UsageList<ToolsCoupling> getToolsCouplingAndUsage(String domainId);

    /**
     * The CroppingPlanEntryDto UsageList from domain with id the given domainId.
     * @param domainId domain'id
     * @return All CroppingPlanEntryDto and there usage status for domain with id the given domainId
     */
    UsageList<CroppingPlanEntryDto> getCroppingPlanEntryDtoAndUsage(String domainId);

    /**
     * List of CroppingPlan's Codes for domains with code the given code and for given campaigns.
     *
     * @param code domain's code
     * @param campaigns required campaigns for domain, can not be null or empty
     * @return list of CroppingPlan's Codes
     */
    List<String> getCroppingPlanCodeForDomainsAndCampaigns(String code, Set<Integer> campaigns);

    List<String> getCroppingPlanSpeciesCodeForCropCodeAndCampaigns(String cropCode, Set<Integer> campaigns);

    /**
     * list of ToolsCoupling's Codes for domains with code the given code and for given campaigns.
     * @param code domain's code
     * @param campaigns required campaigns for domain, can not be null or empty
     * @return List of ToolsCoupling's Codes
     */
    List<String> getToolsCouplingCodeForDomainsAndCampaigns(String code, Set<Integer> campaigns);

    /**
     * List of ToolsCouplings for domains with code the given code and for given campaigns.
     * @param code domain's code
     * @param campaigns required campaigns for domain, can not be null or empty
     * @return List of ToolsCouplings
     */
    List<ToolsCoupling> getToolsCouplingsForDomainCodeAndCampaigns(String code, Set<Integer> campaigns);

    /**
     * Get the list of cropping plan (assolement) of the given domain containing only main
     * crops (not intermediate). This list is read only and no element can be added to it.
     *
     * @param domainId domain's id
     * @return All cropping plans (assolement) of the given domain containing only main
     * crops (not intermediate).
     */
    List<CroppingPlanEntryDto> getCroppingPlanDtos(String domainId);

    /**
     * All CroppingPlanSpecies Ids and there usage status
     * @param domainId domain's id
     * @return
     */
    Map<String, Boolean> getCroppingPlanSpeciesUsage(String domainId);

    /**
     * Charge les couples cultures/especes à partir du code de l'espèce et d'un Set de campagnes
     *
     * @param croppingPlanEntryCode code de la culture
     * @param campaignsSet          liste des campagnes
     * @return une Pair&lt;Culture, Map&lt;code espece, Espece&gt;&gt;
     */
    Pair<CroppingPlanEntry, Map<String, CroppingPlanSpecies>> getEntryAndSpeciesFromCode(
            String croppingPlanEntryCode, Set<Integer> campaignsSet);

    /**
     * Export selected domains id as excel sheet.
     */
    InputStream exportDomainAsXlsStream(List<String> domainIds);

    /**
     * Import domain from input stream
     * @param is domains as InputStream
     */
    void importDomainForXlsStream(InputStream is);

    /**
     * Check if a domain already exists if specified name.
     *
     * @param domainName domain name to check
     * @return {@code true} if a domain already exists with that name
     */
    boolean checkDomainExistence(String domainName);

    /**
     * Return the number of domains
     *
     * @return The number of domains
     * @param active optional active filter (may be null)
     */
    long getDomainsCount(Boolean active);

    /**
     * @param domainTopiaId the domain's topiaId, can be null, if null an empty list is returned.
     * @return the list of prices involved within the given domain or a list with default prices
     */
    List<Price> getDomainPrices(String domainTopiaId);

    /**
     * delete Tools couplings and the associated object
     *
     * @param toolsCouplings
     */
    void deleteToolsCouplings(Collection<ToolsCoupling> toolsCouplings);

    /**
     * delete related object to the actions that are no more used
     *
     * @param toolsCouplingCode    tools coupling code
     * @param originalMainsActions collection of previous mains actions
     * @param actualMainsActions   collection of actuel mains actions
     */
    void deleteUnusedMainAction(String toolsCouplingCode, Collection<RefInterventionAgrosystTravailEDI> originalMainsActions, Collection<RefInterventionAgrosystTravailEDI> actualMainsActions);

    /**
     * Do validate the current domain state
     *
     * @param domainId the identifier of the domain to validate
     * @return the validated domain
     */
    Domain validateAndCommit(String domainId);

    Map<String, Pair<CroppingPlanEntry, Map<String, CroppingPlanSpecies>>> findSpeciesCodeByCropCodeForCampaigns(List<String> croppingPlanEntryCodes, Set<Integer> campaignsSet);

    /**
     * return the domain code associated to the growing system with topiaId the growingSystemId parameters.
     * @param growingSystemId The id of the growing system
     * @return the domain related to the Growing System
     */
    String getDomainCodeForGrowingSystem(String growingSystemId);

    /**
     * Do not use this method to get writable domains, this is only usable on DecisionRule creation !
     * See https://forge.codelutin.com/issues/4439#note-6
     * @return get active writable domains for DecisionRule
     */
    List<Domain> getActiveWritableDomainsForDecisionRuleCreation();

    /**
     * return all domains TopiaId that match the domain code given as parameter for the given campaigns range.
     * @param code the domains code
     * @param campaigns the required campaigns
     * @return all domains TopiaId
     */
    List<String> findDomainIdsForCodeAndCampaigns(String code, Set<Integer> campaigns, boolean includeCropsFromInactiveDomains);

    /**
     * Copy equipments and optionally tools couplings and manuals interventions from domain the given fromDomain to domain the given toDomains list
     * @param fromDomain source domain Id
     * @param toDomains target domains Ids
     * @param includeToolCouplings true if tools coupling have to be include
     * @param equipmentsToCopy list of source domain equipments
     * @param toolsCouplingsToCopy list of tools couplings
     * @return the result message
     */
    List<String> copyTools(String fromDomain, List<String> toDomains, Boolean includeToolCouplings, List<Equipment> equipmentsToCopy, List<ToolsCoupling> toolsCouplingsToCopy);

    /**
     * Get domain'plots total area.
     *
     * @param domainId domain id
     * @return total area
     */
    double getDomainSAUArea(String domainId);


    /**
     * Try to Import domains created from from PZ0 csv
     * @param allResult All domains and there dependencies required to save it.
     */
    void importPZ0Domains(Map<Class, ImportResults> allResult);

}
