package fr.inra.agrosyst.api.services.domain;

/*
 * #%L
 * Agrosyst :: API
 * $Id: CroppingPlans.java 4612 2014-12-06 18:43:48Z echatellier $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-api/src/main/java/fr/inra/agrosyst/api/services/domain/CroppingPlans.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.ArrayList;
import java.util.List;

import org.nuiton.util.StringUtil;

import com.google.common.base.Function;
import com.google.common.base.MoreObjects;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.Lists;

import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.CroppingPlanSpecies;
import fr.inra.agrosyst.api.entities.referential.RefEspece;
import fr.inra.agrosyst.api.entities.referential.RefVariete;
import fr.inra.agrosyst.api.services.practiced.CropCycleModelDto;

/**
 * @author Arnaud Thimel : thimel@codelutin.com
 */
public class CroppingPlans {

    public static final Function<CroppingPlanSpecies, CroppingPlanSpeciesDto> CROPPING_PLAN_SPECIES_TO_DTO = new Function<CroppingPlanSpecies, CroppingPlanSpeciesDto>() {
        @Override
        public CroppingPlanSpeciesDto apply(CroppingPlanSpecies input) {
            CroppingPlanSpeciesDto result = new CroppingPlanSpeciesDto();
            result.setTopiaId(input.getTopiaId());
            result.setCode(input.getCode());
            RefEspece species = input.getSpecies();
            result.setSpeciesId(species.getTopiaId());
            result.setSpeciesEspece(species.getLibelle_espece_botanique());
            result.setSpeciesQualifiant(species.getLibelle_qualifiant_AEE());
            result.setSpeciesTypeSaisonnier(species.getLibelle_type_saisonnier_AEE());
            result.setSpeciesDestination(species.getLibelle_destination_AEE());
            result.setProfil_vegetatif_BBCH(species.getProfil_vegetatif_BBCH());
            RefVariete variety = input.getVariety();
            if (variety != null) {
                result.setVarietyId(variety.getTopiaId());
                result.setVarietyLibelle(variety.getLabel());
            }
            result.setValidated(input.isValidated());
            return result;
        }
    };

    public static final Function<CroppingPlanEntry, CroppingPlanEntryDto> CROPPING_PLAN_ENTRY_TO_DTO = new Function<CroppingPlanEntry, CroppingPlanEntryDto>() {
        @Override
        public CroppingPlanEntryDto apply(CroppingPlanEntry input) {
            CroppingPlanEntryDto result = new CroppingPlanEntryDto();
            result.setTopiaId(input.getTopiaId());
            result.setCode(input.getCode());
            String name = input.getName();
            result.setName(name);
            result.setSellingPrice(input.getSellingPrice());
            result.setType(input.getType());
            List<CroppingPlanSpecies> species = MoreObjects.firstNonNull(input.getCroppingPlanSpecies(), new ArrayList<CroppingPlanSpecies>());
            result.setSpecies(Lists.transform(species, CROPPING_PLAN_SPECIES_TO_DTO));
            result.setColor(STRING_TO_COLOR.apply(name));
            result.setValidated(input.isValidated());
            return result;
        }
    };

    public static final Function<String, String> STRING_TO_COLOR = new Function<String, String>() {
        @Override
        public String apply(String input) {
            String hash = StringUtil.encodeSHA1(input);
            String result = "#" + hash.substring(7, 13);
            return result;
        }
    };

    public static final Function<CroppingPlanSpecies,String> GET_SPECIES_CODE = new Function<CroppingPlanSpecies, String>() {
        @Override
        public String apply(CroppingPlanSpecies input) {
            return input.getCode();
        }
    };

    public static final Function<CroppingPlanSpeciesDto,String> GET_SPECIES_DTO_CODE = new Function<CroppingPlanSpeciesDto, String>() {
        @Override
        public String apply(CroppingPlanSpeciesDto input) {
            return input.getCode();
        }
    };

    public static final Predicate<CroppingPlanEntryDto> IS_ENTRY_INTERMEDIATE = new Predicate<CroppingPlanEntryDto>() {
        public boolean apply(CroppingPlanEntryDto input) {
            return input.isIntermediate();
        }
    };

    public static final Predicate<CropCycleModelDto> IS_INTERMEDIATE = new Predicate<CropCycleModelDto>() {
        public boolean apply(CropCycleModelDto input) {
            return input.isIntermediate();
        }
    };

    public static final Predicate<CropCycleModelDto> IS_NOT_INTERMEDIATE = Predicates.not(IS_INTERMEDIATE);

}
