package com.cybelia.sandra.services.ejb3;

import com.cybelia.sandra.SandraConfigHelper;
import java.io.IOException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import javax.interceptor.AroundInvoke;
import javax.interceptor.InvocationContext;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;

/**
 * Interceptor to manage transaction
 *
 * @author sletellier
 */
public class TransactionManager {

    /**
     * Logger.
     */
    static Log log = LogFactory.getLog(TransactionManager.class);

    protected TopiaContext rootContext;

    @AroundInvoke
    public Object manageTrasaction(InvocationContext context) throws Exception {

        // Check if transaction is needed (with annotation @Transaction)
        Method method = context.getMethod();
        if (!method.isAnnotationPresent(Transaction.class)) {

            // Proced
            return context.proceed();
        }

        BaseServiceImpl baseService = (BaseServiceImpl) context.getTarget();

        // Get annotation
        Transaction transactionAnnotation = method.getAnnotation(Transaction.class);
        TopiaContext transaction;
        try {

            // Create new root context if not exist for this principal
            if (rootContext == null) {

                rootContext = SandraConfigHelper.getServiceRootContext();
            }
            transaction = rootContext.beginTransaction();
        } catch (TopiaException eee) {
            log.error("Failed to begin transaction", eee);

            throw eee;
        }

        try {

            Object[] parameters = context.getParameters();

            // Inject transaction
            List<Object> parameterList = new ArrayList<Object>();
            parameterList.add(transaction);

            // Proced
            Class<?>[] parameterTypes = method.getParameterTypes();
            if (parameterTypes != null && parameterTypes.length > 0 && parameterTypes[0].equals(TopiaContext.class)) {

                // Remove first parameter (transaction is overide if exist)
                List<Object> list = new ArrayList<Object>(Arrays.asList(parameters));
                list.remove(0);
                parameterList.addAll(list);

            } else {

                List<Class<?>> parameterTypesList = new ArrayList<Class<?>>();
                parameterTypesList.add(TopiaContext.class);
                parameterTypesList.addAll(Arrays.asList(parameterTypes));
                method = method.getDeclaringClass().getMethod(method.getName(), parameterTypesList.toArray(new Class[parameterTypesList.size()]));

                parameterList.addAll(Arrays.asList(parameters));
            }
            if (log.isDebugEnabled()) {
                log.debug("Invoke " + method.getName() + " with params : " + Arrays.toString(parameterList.toArray()));
            }
            Object result = method.invoke(baseService, parameterList.toArray());
            return result;
        } catch (Exception eee) {

            // Get logger
            Log logger = LogFactory.getLog(context.getTarget().getClass());

            // Display log error
            String errorMsg = transactionAnnotation.errorMsg();
            if (StringUtils.isEmpty(errorMsg)) {
                errorMsg = "Failled to execute method '" + method.getName() + "'";
            }
            transactionAnnotation.errorLevel().logError(logger, errorMsg, eee);

            throw eee;
        } finally {

            if (transactionAnnotation.close()) {
                try {
                    transaction = closeTransaction(transaction, transactionAnnotation.commit());
                } catch (TopiaException eee) {
                    log.error("Failed to commit transaction", eee);
                }
            }
        }
    }

    public static TopiaContext beginTransaction() throws TopiaException, IOException {
        TopiaContext rootContext = SandraConfigHelper.getServiceRootContext();
        return rootContext.beginTransaction();
    }

    public static TopiaContext closeTransaction(TopiaContext transaction, boolean commit) throws TopiaException {
        // Commit
        if (transaction != null && !transaction.isClosed()) {

            // If we want commit
            if (commit) {
                transaction.commitTransaction();
            }
            transaction.closeContext();
        } else {
            log.warn("Context is already closed");
        }
        return transaction;
    }
}
