package com.cybelia.sandra.services.ejb3;

import com.cybelia.sandra.SandraDAOHelper;
import com.cybelia.sandra.entities.Eleveur;
import com.cybelia.sandra.entities.InfoAccess;
import com.cybelia.sandra.entities.Label;
import com.cybelia.sandra.entities.TypeConnectionEnum;
import com.cybelia.sandra.entities.UserIndicateurs;
import com.cybelia.sandra.entities.UserIndicateursDAO;
import com.cybelia.sandra.security.SecurityHelper;
import com.cybelia.sandra.services.ServiceHelper;
import com.cybelia.sandra.services.local.ServiceCommonLocal;
import com.cybelia.sandra.services.local.ServiceNotifierLocal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import javax.annotation.security.PermitAll;
import javax.ejb.EJB;
import javax.ejb.Stateless;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.ejb3.annotation.SecurityDomain;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.taas.TaasService;
import org.nuiton.topia.taas.TaasUtil;
import org.nuiton.topia.taas.entities.TaasUser;

/**
 * @author sletellier
 */
@Stateless
@SecurityDomain("sandra")
@PermitAll
public class ServiceCommonImpl extends BaseServiceImpl implements ServiceCommonLocal {

    protected static final Log log = LogFactory.getLog(ServiceCommonImpl.class);

    @EJB
    protected ServiceNotifierLocal serviceNotifier;

    /**
     * Verifie que l'utilisateur a le droit en lecture sur l'entite ayant ce topiaId
     *
     * @param topiaId
     * @return
     */
    @Override
    public boolean canReadEntity(TopiaContext transaction, String topiaId) {
        return canDoOnEntity(transaction, topiaId, TaasUtil.LOAD);
    }

    /**
     * Verifie que l'utilisateur a le droit en ecriture sur l'entite ayant ce topiaId
     *
     * @param topiaId
     * @return
     */
    @Override
    public boolean canUpdateEntity(TopiaContext transaction, String topiaId) {
        return canDoOnEntity(transaction, topiaId, TaasUtil.UPDATE);
    }

    /**
     * Verifie que l'utilisateur a le droit demande sur l'entite ayant ce topiaId
     *
     * @param topiaId
     * @param actions
     * @return
     */
    protected boolean canDoOnEntity(TopiaContext transaction, String topiaId, int actions) {
        TaasService taas = null;
        try {
            taas = transaction.getService(TaasService.class);
        } catch (TopiaException te) {
            if (log.isErrorEnabled()) {
                log.error(te);
            }
        }
        boolean canDo = true;
        if (taas != null) {
            try {
                taas.check(topiaId, actions);
            } catch (SecurityException se) {
                if (log.isDebugEnabled()) {
                    log.debug("Permission check failed", se);
                }
                canDo = false;
            }
        }
        return canDo;
    }

    /**
     * Classe utilitaire pour retrouver le label corresondant
     */
    @Override
    public String getLabel(TopiaContext context, String category, String id) throws TopiaException {
        if (id != null) {
            Label lbl = SandraDAOHelper.getLabelDAO(context).findByNaturalId(id, category);
            if (lbl != null) {
                return lbl.getValeur();
            }
        }
        return null;
    }

    /**
     * Classe utilitaire pour détecter les modifications
     */
    protected String diff(Object compare, Object to) {
        if (compare == null && to == null) return null;

        if (to != null && (compare == null || !compare.equals(to))) {
            return to.toString();
        } else {
            return null;
        }
    }

    // Fait le diff et retourne le mix des 2 valeur sous la forme old&&new si il y a une modif, sinon, il renvoie un string vide
    protected String diffAndMix(String oldValue, String newValue) {

        boolean oldsNullOrEmpty = oldValue == null;
        if (!oldsNullOrEmpty){
            oldsNullOrEmpty = oldValue.trim().isEmpty();
        }
        boolean newNullOrEmpty = newValue == null;
        if (!newNullOrEmpty){
            newNullOrEmpty = newValue.trim().isEmpty();
        }

        if (oldsNullOrEmpty && newNullOrEmpty) {
            return null;
        }

        if (newValue == null){
            newValue = "";
        }
        if (oldValue == null){
            oldValue = "";
        }

        if (!oldValue.equals(newValue)) {
            return oldValue + "&&" + newValue;
        } else {
            return null;
        }
    }

    // Fait le diff et retourne le mix des 2 labels
    protected String diffAndMixOldAndNewLabelValue(TopiaContext context, String category, String oldValue, String newValue) throws TopiaException {
        boolean ndOld = (oldValue == null || oldValue.isEmpty() || oldValue.trim().equals("ND"));
        boolean ndNew = (newValue == null || newValue.isEmpty() || newValue.trim().equals("ND"));
        if (log.isDebugEnabled()){
            log.debug("Label value old : " + oldValue + " new : " + newValue);
        }
        if (ndOld && ndNew){
            return null;
        }
        return diffAndMix(getLabel(context, category, oldValue), getLabel(context, category, newValue));
    }

    // Fait le diff et retourne le mix des niveaux de securite
    protected String diffAndMixNiveauSecu(int oldSecurite, int newSecurite) {

        // Ne pas tenir envoyer le mail si cela passe de non renseigné a vert.
        boolean dontSendNotif = (oldSecurite == -1 && newSecurite <= 1);
        if (dontSendNotif){
            return null;
        }
        return diffAndMix(String.valueOf(oldSecurite), String.valueOf(newSecurite));
    }

    // Fait le diff et retourne le mix des list de risk
    protected String diffAndMixOldAndNewRiskValue(TopiaContext context, Collection<String> oldValues, Collection<String> newValues) throws TopiaException {
        String oldMix = "";
        if (oldValues != null) {
            for (String oldValue : oldValues) {
                oldMix += getLabel(context, "RSK", oldValue).trim() + "##";
            }
        }
        if (!oldMix.trim().isEmpty()) {
            oldMix = oldMix.substring(0, oldMix.length() - 2);
        }

        String newMix = "";
        if (newValues != null) {
            for (String newValue : newValues) {
                newMix += getLabel(context, "RSK", newValue).trim() + "##";
            }
        }
        if (!newMix.trim().isEmpty()) {
            newMix = newMix.substring(0, newMix.length() - 2);
        }

        return diffAndMix(oldMix.trim(), newMix.trim());
    }

    @Override
    public void notifyChangedBreeder(TopiaContext context, String notifierName, String userLogin, Eleveur eleveurOld, Eleveur eleveur) throws TopiaException {
        List<String> breederValues = getChangedDatasBreeder(context, eleveurOld, eleveur);

        // Ajout des variables d'environements et des entites
        List<String> valuesToSend = new ArrayList<String>();
        if (changed(breederValues)) {
            valuesToSend.add(ServiceHelper.getSandraName());
            valuesToSend.add(ServiceHelper.getSandraUrl());
            valuesToSend.add(userLogin);
            valuesToSend.add(eleveur.getTopiaId());
            valuesToSend.add(diffAndMix(eleveurOld.getRaisonSociale(), eleveur.getRaisonSociale()));
            valuesToSend.addAll(breederValues);
            notifyChanged(notifierName, valuesToSend);
        }
    }

    @Override
    public void notifyChangedBreeder(TopiaContext context, String notifierName, String userLogin, Eleveur eleveurOld, Eleveur eleveur, InfoAccess accesOld, InfoAccess acces) throws TopiaException {
        List<String> breederValues = getChangedDatasBreeder(context, eleveurOld, eleveur);
        List<String> infoAccessValues = getChangedDatasInfoAccess(context, accesOld, acces);

        // Ajout des variables d'environements et des entites
        List<String> valuesToSend = new ArrayList<String>();
        if (changed(breederValues) || changed(infoAccessValues)) {
            valuesToSend.add(ServiceHelper.getSandraName());
            valuesToSend.add(ServiceHelper.getSandraUrl());
            valuesToSend.add(userLogin);
            valuesToSend.add(eleveur.getTopiaId());
            valuesToSend.addAll(breederValues);
            valuesToSend.addAll(infoAccessValues);
            notifyChanged(notifierName, valuesToSend);
        }
    }

    @Override
    public void notifyChangedInfoAccess(TopiaContext context, String notifierName, String userLogin, String breederTopiaId, InfoAccess infoAccesOld, InfoAccess infoAcces) throws TopiaException {
        List<String> values = getChangedDatasInfoAccess(context, infoAccesOld, infoAcces);

        // Ajout des variables d'environements et des entites
        List<String> valuesToSend = new ArrayList<String>();
        if (changed(values)) {
            valuesToSend.add(ServiceHelper.getSandraName());
            valuesToSend.add(ServiceHelper.getSandraUrl());
            valuesToSend.add(userLogin);
            valuesToSend.add(infoAcces.getTopiaId());
            valuesToSend.add(breederTopiaId);

            valuesToSend.addAll(values);
            notifyChanged(notifierName, valuesToSend);
        }
    }

    @Override
    public void notifyChangedInfoAccess(TopiaContext context, String notifierName, String userLogin, String breederTopiaId, InfoAccess infoAccesOld, InfoAccess infoAcces, Double oldX, Double oldY) throws TopiaException {
        List<String> values = getChangedDatasInfoAccess(context, infoAccesOld, infoAcces);

        // Get new coord
        String newX = String.valueOf(infoAcces.getGps().getPoint().getX());
        String newY = String.valueOf(infoAcces.getGps().getPoint().getY());

        // Coords changed
        String x = diffAndMix(String.valueOf(oldX), newX);
        String y = diffAndMix(String.valueOf(oldY), newY);

        // Ajout des variables d'environements et des entites
        List<String> valuesToSend = new ArrayList<String>();
        if (changed(values) || x != null || y != null) {
            valuesToSend.add(ServiceHelper.getSandraName());
            valuesToSend.add(ServiceHelper.getSandraUrl());
            valuesToSend.add(userLogin);
            valuesToSend.add(infoAcces.getTopiaId());
            valuesToSend.add(breederTopiaId);
            valuesToSend.add(x);
            valuesToSend.add(y);
            valuesToSend.addAll(values);
            notifyChanged(notifierName, valuesToSend);
        }
    }

    protected List<String> getChangedDatasBreeder(TopiaContext context, Eleveur eleveurOld, Eleveur eleveur) throws TopiaException {
        // Modification demandant validation
        List<String> values = new ArrayList<String>();

        // Eleveur
//        values.add(diffAndMix(eleveurOld.getRaisonSociale(), eleveur.getRaisonSociale()));
        values.add(diffAndMix(eleveurOld.getTelephone(), eleveur.getTelephone()));
        values.add(diffAndMix(eleveurOld.getMobile(), eleveur.getMobile()));
        values.add(diffAndMix(eleveurOld.getAdresse(), eleveur.getAdresse()));
        values.add(diffAndMix(eleveurOld.getVille(), eleveur.getVille()));
        values.add(diffAndMix(eleveurOld.getCodePostal(), eleveur.getCodePostal()));
        values.add(diffAndMix(eleveurOld.getEmail(), eleveur.getEmail()));
//        values.add(diffAndMix(eleveurOld.getCodeINSEE(), eleveur.getCodeINSEE()));
        values.add(diffAndMixOldAndNewLabelValue(context, "ECH", eleveurOld.getContrainteHoraire(), eleveur.getContrainteHoraire()));
        values.add(diffAndMix(eleveurOld.getCommentaire(), eleveur.getCommentaire()));

        return values;
    }

    protected List<String> getChangedDatasInfoAccess(TopiaContext context, InfoAccess infoAcces, InfoAccess infoAccesUpdate) throws TopiaException {
        // Modification demandant validation
        List<String> values = new ArrayList<String>();

        // Eleveur
        if (infoAcces != null && infoAccesUpdate != null) {
            values.add(diffAndMix(infoAcces.getNomAcces(), infoAccesUpdate.getNomAcces()));
            values.add(diffAndMixOldAndNewLabelValue(context, "IAS", infoAcces.getAccesSilo(), infoAccesUpdate.getAccesSilo()));
            values.add(diffAndMixOldAndNewLabelValue(context, "IMC", infoAcces.getModeChargement(), infoAccesUpdate.getModeChargement()));
            values.add(diffAndMix(String.valueOf(infoAcces.getEtat()), String.valueOf(infoAccesUpdate.getEtat())));
            values.add(diffAndMixNiveauSecu(infoAcces.getNiveauSecurite(), infoAccesUpdate.getNiveauSecurite()));
            values.add(diffAndMixOldAndNewRiskValue(context, infoAcces.getRisques(), infoAccesUpdate.getRisques()));
            values.add(diffAndMix(infoAcces.getCommentaireSecurite(), infoAccesUpdate.getCommentaireSecurite()));
        }

        return values;
    }

    protected boolean changed(List<String> values) {
        // Verification de modification
        boolean sendNotif = false;
        for (String value : values) {
            sendNotif = (value != null && !value.trim().isEmpty()) || sendNotif;

            if (log.isDebugEnabled()){
                log.debug("Changed : " + sendNotif + " value : " + value);
            }
        }
        return sendNotif;
    }

    protected void notifyChanged(String notifierName, List<String> valuesToSend) throws TopiaException {
        if (log.isDebugEnabled()){
            log.debug("notifyChanged : " + notifierName);
        }
        // Envoie de la notification
        serviceNotifier.notifyEvent(notifierName, valuesToSend.toArray(new String[0]));
    }

    // Methods to copy if necessary field of bean
    @Override
    public <E extends TopiaEntity> E copyEntityIfNecessary(TopiaContext transaction, E beanFrom, E beanTo, String field) {
        try {
            String fromValue = BeanUtils.getSimpleProperty(beanFrom, field);
            String toValue = BeanUtils.getSimpleProperty(beanTo, field);

            if (fromValue == null || !fromValue.equals(toValue)) {
                BeanUtils.setProperty(beanTo, field, fromValue);
                incNbMaj(transaction);
            }
        } catch (Exception eee) {
            log.error("Failed to copy property " + field + " on bean " + beanFrom, eee);
        }
        return beanTo;
    }

    // Methods used to increment User indicators
    @Override
    public UserIndicateurs incSynch(TopiaContext transaction, TypeConnectionEnum typeConnection, boolean gprs, boolean update) throws TopiaException {
        UserIndicateurs result = null;
        switch (typeConnection) {
            case SYNCHRO:
                if (gprs) {
                    result = incNbSynchGprs(transaction, update);
                } else {
                    result = incNbSynchWifi(transaction, update);
                }
                result = updateLastSynchDate(transaction, result, update);
                break;
            case AUTO:
                result = incNbSynchAuto(transaction, update);
                result = updateLastSynchDate(transaction, result, update);
                break;
            case NOTIFICATION:
                result = incNbNotifs(transaction, update);
                break;
        }
        return result;
    }

    @Override
    @Transaction(errorMsg = "Failed to inc nb octets send gprs")
    public UserIndicateurs incNbOctetsReceivedGPRS(int inc) throws TopiaException {
        throw new RuntimeException("This method must be never call");
    }

    @Override
    public UserIndicateurs incNbOctetsReceivedGPRS(TopiaContext transaction, int inc) throws TopiaException {
        return incNbOctetsReceivedGPRS(transaction, inc, true);
    }

    @Override
    public UserIndicateurs incNbOctetsReceivedGPRS(TopiaContext transaction, int inc, boolean update) throws TopiaException {
        return inc(transaction, UserIndicateurs.PROPERTY_NB_OCTETS_RECEIVED_GPRS, inc, update);
    }

    @Override
    public UserIndicateurs incNbOctetsReceivedGPRS(UserIndicateurs userIndicateurs, int inc) throws TopiaException {
        return inc(userIndicateurs, UserIndicateurs.PROPERTY_NB_OCTETS_RECEIVED_GPRS, inc);
    }

    @Override
    @Transaction(errorMsg = "Failed to inc nb octets send gprs")
    public UserIndicateurs incNbOctetsSendGPRS(int inc) throws TopiaException {
        throw new RuntimeException("This method must be never call");
    }

    @Override
    public UserIndicateurs incNbOctetsSendGPRS(TopiaContext transaction, int inc) throws TopiaException {
        return incNbOctetsSendGPRS(transaction, inc, true);
    }

    @Override
    public UserIndicateurs incNbOctetsSendGPRS(TopiaContext transaction, int inc, boolean update) throws TopiaException {
        return inc(transaction, UserIndicateurs.PROPERTY_NB_OCTETS_SEND_GPRS, inc, update);
    }

    @Override
    public UserIndicateurs incNbOctetsSendGPRS(UserIndicateurs userIndicateurs, int inc) throws TopiaException {
        return inc(userIndicateurs, UserIndicateurs.PROPERTY_NB_OCTETS_SEND_GPRS, inc);
    }

    @Override
    @Transaction(errorMsg = "Failed to inc nb octets received wifi")
    public UserIndicateurs incNbOctetsReceivedWifi(int inc) throws TopiaException {
        throw new RuntimeException("This method must be never call");
    }

    @Override
    public UserIndicateurs incNbOctetsReceivedWifi(TopiaContext transaction, int inc) throws TopiaException {
        return incNbOctetsReceivedWifi(transaction, inc, true);
    }

    @Override
    public UserIndicateurs incNbOctetsReceivedWifi(TopiaContext transaction, int inc, boolean update) throws TopiaException {
        return inc(transaction, UserIndicateurs.PROPERTY_NB_OCTETS_RECEIVED_WIFI, inc, update);
    }

    @Override
    public UserIndicateurs incNbOctetsReceivedWifi(UserIndicateurs userIndicateurs, int inc) throws TopiaException {
        return inc(userIndicateurs, UserIndicateurs.PROPERTY_NB_OCTETS_RECEIVED_WIFI, inc);
    }

    @Override
    @Transaction(errorMsg = "Failed to inc nb octets send wifi")
    public UserIndicateurs incNbOctetsSendWifi(int inc) throws TopiaException {
        throw new RuntimeException("This method must be never call");
    }

    @Override
    public UserIndicateurs incNbOctetsSendWifi(TopiaContext transaction, int inc) throws TopiaException {
        return incNbOctetsSendWifi(transaction, inc, true);
    }

    @Override
    public UserIndicateurs incNbOctetsSendWifi(TopiaContext transaction, int inc, boolean update) throws TopiaException {
        return inc(transaction, UserIndicateurs.PROPERTY_NB_OCTETS_SEND_WIFI, inc, update);
    }

    @Override
    public UserIndicateurs incNbOctetsSendWifi(UserIndicateurs userIndicateurs, int inc) throws TopiaException {
        return inc(userIndicateurs, UserIndicateurs.PROPERTY_NB_OCTETS_SEND_WIFI, inc);
    }

    @Override
    @Transaction(errorMsg = "Failed to inc nb synch ko")
    public UserIndicateurs incNbSynchKo() throws TopiaException {
        throw new RuntimeException("This method must be never call");
    }

    public UserIndicateurs incNbSynchKo(TopiaContext transaction) throws TopiaException {
        return inc(transaction, UserIndicateurs.PROPERTY_NB_SYNCH_KO);
    }

    @Override
    public UserIndicateurs incNbSynchKo(TopiaContext transaction, int inc) throws TopiaException {
        return incNbSynchKo(transaction, inc, true);
    }

    @Override
    public UserIndicateurs incNbSynchKo(TopiaContext transaction, int inc, boolean update) throws TopiaException {
        return inc(transaction, UserIndicateurs.PROPERTY_NB_SYNCH_KO, inc, update);
    }

    @Override
    public UserIndicateurs incNbSynchKo(UserIndicateurs userIndicateurs, int inc) throws TopiaException {
        return inc(userIndicateurs, UserIndicateurs.PROPERTY_NB_SYNCH_KO, inc);
    }

    @Override
    @Transaction(errorMsg = "Failed to inc nb notifs")
    public UserIndicateurs incNbNotifs() throws TopiaException {
        throw new RuntimeException("This method must be never call");
    }

    @Override
    public UserIndicateurs incNbNotifs(TopiaContext transaction) throws TopiaException {
        return incNbNotifs(transaction, true);
    }

    @Override
    public UserIndicateurs incNbNotifs(TopiaContext transaction, boolean update) throws TopiaException {
        return inc(transaction, UserIndicateurs.PROPERTY_NB_NOTIFS, update);
    }

    @Override
    public UserIndicateurs incNbNotifs(UserIndicateurs userIndicateurs) throws TopiaException {
        return inc(userIndicateurs, UserIndicateurs.PROPERTY_NB_NOTIFS);
    }

    @Override
    @Transaction(errorMsg = "Failed to inc nb synch auto")
    public UserIndicateurs incNbSynchAuto() throws TopiaException {
        throw new RuntimeException("This method must be never call");
    }

    @Override
    public UserIndicateurs incNbSynchAuto(TopiaContext transaction) throws TopiaException {
        return incNbSynchAuto(transaction, true);
    }

    @Override
    public UserIndicateurs incNbSynchAuto(TopiaContext transaction, boolean update) throws TopiaException {
        return inc(transaction, UserIndicateurs.PROPERTY_NB_SYNCH_AUTO, update);
    }

    @Override
    public UserIndicateurs incNbSynchAuto(UserIndicateurs userIndicateurs) throws TopiaException {
        return inc(userIndicateurs, UserIndicateurs.PROPERTY_NB_SYNCH_AUTO);
    }

    @Override
    @Transaction(errorMsg = "Failed to inc nb gps maj")
    public UserIndicateurs incNbGpsMaj() throws TopiaException {
        throw new RuntimeException("This method must be never call");
    }

    @Override
    public UserIndicateurs incNbGpsMaj(TopiaContext transaction) throws TopiaException {
        return incNbGpsMaj(transaction, true);
    }

    @Override
    public UserIndicateurs incNbGpsMaj(TopiaContext transaction, boolean update) throws TopiaException {
        return inc(transaction, UserIndicateurs.PROPERTY_NB_GPS_MAJ, update);
    }

    @Override
    public UserIndicateurs incNbGpsMaj(UserIndicateurs userIndicateurs) throws TopiaException {
        return inc(userIndicateurs, UserIndicateurs.PROPERTY_NB_GPS_MAJ);
    }

    @Override
    @Transaction(errorMsg = "Failed to inc nb secu maj")
    public UserIndicateurs incNbSecuMaj() throws TopiaException {
        throw new RuntimeException("This method must be never call");
    }

    @Override
    public UserIndicateurs incNbSecuMaj(TopiaContext transaction) throws TopiaException {
        return incNbSecuMaj(transaction, true);
    }

    @Override
    public UserIndicateurs incNbSecuMaj(TopiaContext transaction, boolean update) throws TopiaException {
        return inc(transaction, UserIndicateurs.PROPERTY_NB_SECU_MAJ, update);
    }

    @Override
    public UserIndicateurs incNbSecuMaj(UserIndicateurs userIndicateurs) throws TopiaException {
        return inc(userIndicateurs, UserIndicateurs.PROPERTY_NB_SECU_MAJ);
    }

    @Override
    @Transaction(errorMsg = "Failed to inc nb maj")
    public UserIndicateurs incNbMaj() throws TopiaException {
        throw new RuntimeException("This method must be never call");
    }

    @Override
    public UserIndicateurs incNbMaj(TopiaContext transaction) throws TopiaException {
        UserIndicateurs result = incNbMaj(transaction, true);
        return result;
    }

    @Override
    public UserIndicateurs incNbMaj(TopiaContext transaction, boolean update) throws TopiaException {
        UserIndicateurs result = inc(transaction, UserIndicateurs.PROPERTY_NB_MAJ, update);
        if (result != null) {
            result = updateLastModif(transaction, result, update);
        }
        return result;
    }

    @Override
    @Transaction(errorMsg = "Failed to inc nb maj")
    public UserIndicateurs incNbMaj(UserIndicateurs userIndicateurs) throws TopiaException {
        throw new RuntimeException("This method must be never call");
    }

    public UserIndicateurs incNbMaj(TopiaContext transaction, UserIndicateurs userIndicateurs) throws TopiaException {
        UserIndicateurs result = inc(userIndicateurs, UserIndicateurs.PROPERTY_NB_MAJ);
        result = updateLastModif(transaction, result);
        return result;
    }

    @Override
    @Transaction(errorMsg = "Failed to inc nb synch wifi")
    public UserIndicateurs incNbSynchWifi() throws TopiaException {
        throw new RuntimeException("This method must be never call");
    }

    @Override
    public UserIndicateurs incNbSynchWifi(TopiaContext transaction) throws TopiaException {
        return incNbSynchWifi(transaction, true);
    }

    @Override
    public UserIndicateurs incNbSynchWifi(TopiaContext transaction, boolean update) throws TopiaException {
        return inc(transaction, UserIndicateurs.PROPERTY_NB_SYNCH_WIFI, update);
    }

    @Override
    public UserIndicateurs incNbSynchWifi(UserIndicateurs userIndicateurs) throws TopiaException {
        return inc(userIndicateurs, UserIndicateurs.PROPERTY_NB_SYNCH_WIFI);
    }

    @Override
    @Transaction(errorMsg = "Failed to inc nb synch gprs")
    public UserIndicateurs incNbSynchGprs() throws TopiaException {
        throw new RuntimeException("This method must be never call");
    }

    @Override
    public UserIndicateurs incNbSynchGprs(TopiaContext transaction) throws TopiaException {
        return incNbSynchGprs(transaction, true);
    }

    @Override
    public UserIndicateurs incNbSynchGprs(TopiaContext transaction, boolean update) throws TopiaException {
        return inc(transaction, UserIndicateurs.PROPERTY_NB_SYNCH_GPRS, update);
    }

    @Override
    public UserIndicateurs incNbSynchGprs(UserIndicateurs userIndicateurs) throws TopiaException {
        return inc(userIndicateurs, UserIndicateurs.PROPERTY_NB_SYNCH_GPRS);
    }

    protected UserIndicateurs updateLastSynchDate(TopiaContext transaction, UserIndicateurs userIndicateurs, boolean update)  throws TopiaException {
        UserIndicateursDAO userIndicateursDAO = SandraDAOHelper.getUserIndicateursDAO(transaction);
        userIndicateurs.setLastSynch(new Date());
        if (update) {
            userIndicateurs = userIndicateursDAO.update(userIndicateurs);
        }
        return userIndicateurs;
    }

    protected UserIndicateurs updateLastModif(TopiaContext transaction, UserIndicateurs userIndicateurs) throws TopiaException {
        return updateLastModif(transaction, userIndicateurs, true);
    }

    protected UserIndicateurs updateLastModif(TopiaContext transaction, UserIndicateurs userIndicateurs, boolean update)  throws TopiaException {
        UserIndicateursDAO userIndicateursDAO = SandraDAOHelper.getUserIndicateursDAO(transaction);
        userIndicateurs.setLastModif(new Date());
        if (update) {
            userIndicateurs = userIndicateursDAO.update(userIndicateurs);
        }
        return userIndicateurs;
    }

    protected UserIndicateurs inc(TopiaContext transaction, String methodName) throws TopiaException {
        return inc(transaction, methodName, 1);
    }

    protected UserIndicateurs inc(TopiaContext transaction, String methodName, int inc) throws TopiaException {

        return inc(transaction, methodName, inc, true);
    }

    protected UserIndicateurs inc(TopiaContext transaction, String methodName, boolean update) throws TopiaException {
        return inc(transaction, methodName, 1, update);
    }

    protected UserIndicateurs inc(TopiaContext transaction, String methodName, int inc, boolean update) throws TopiaException {
        TaasUser user = SecurityHelper.getUser();
        UserIndicateursDAO userIndicateursDAO = SandraDAOHelper.getUserIndicateursDAO(transaction);

        UserIndicateurs userIndicateurs = userIndicateursDAO.findByTaasUser(user);
        if (userIndicateurs != null) {
            userIndicateurs = inc(userIndicateurs, methodName, inc);
            if (update) {
                userIndicateurs = userIndicateursDAO.update(userIndicateurs);
            }
        }
        return userIndicateurs;
    }

    protected UserIndicateurs inc(UserIndicateurs userIndicateurs, String methodName) {
        return inc(userIndicateurs, methodName, 1);
    }

    protected UserIndicateurs inc(UserIndicateurs userIndicateurs, String methodName, int inc) {
        try {
            String value = BeanUtils.getSimpleProperty(userIndicateurs, methodName);
            int i = Integer.parseInt(value);
            i = i + inc;
            BeanUtils.setProperty(userIndicateurs, methodName, i);
        } catch (Exception eee) {
            log.error("Failled to inc : " + methodName, eee);
        }
        return userIndicateurs;
    }
}
