package com.cybelia.sandra.services;

import com.cybelia.sandra.entities.ActionSecurite;
import com.cybelia.sandra.entities.Camion;
import com.cybelia.sandra.entities.ChargementUsineConfig;
import com.cybelia.sandra.entities.Chauffeur;
import com.cybelia.sandra.entities.Eleveur;
import com.cybelia.sandra.entities.Etape;
import com.cybelia.sandra.entities.InfoAccess;
import com.cybelia.sandra.entities.Label;
import com.cybelia.sandra.entities.LigneProduit;
import com.cybelia.sandra.entities.Note;
import com.cybelia.sandra.entities.Societe;
import com.cybelia.sandra.entities.Tour;
import com.cybelia.sandra.entities.Transporteur;
import com.cybelia.sandra.entities.UserIndicateurs;
import com.cybelia.sandra.entities.Usine;
import com.cybelia.sandra.entities.trace.SuiviEtape;
import com.cybelia.sandra.entities.trace.SuiviLigneProduit;
import com.cybelia.sandra.services.ejb3.Transaction;
import com.vividsolutions.jts.geom.Point;
import java.io.InputStream;
import java.util.Date;
import java.util.List;
import javax.ejb.Remote;
import org.apache.commons.lang3.tuple.Pair;
import org.jboss.ejb3.annotation.RemoteBinding;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.taas.entities.TaasUser;

/**
 * Services pour la gestion de la partie web
 *
 * @author julien
 */
@Remote
@RemoteBinding(jndiBinding="ServiceWebImpl/remote")
public interface ServiceWeb {

    void createMissingPrincipalsForCamions(TopiaContext transaction) throws TopiaException;

    void deleteDuplicatedEtapesAndCreateUniqueIndex(TopiaContext transaction) throws TopiaException;

    void deleteDuplicatedToursAndCreateUniqueIndex(TopiaContext transaction) throws TopiaException;

    /**
     * Create all action security for dangerous breeder
     *
     * @param transaction
     * @throws TopiaException
     */
    void createAllActionSecurityForEleveurs(TopiaContext transaction) throws TopiaException;

    /**
    * Cree tous les accessEleveur manquant et remplis le commentaire de l'infoAccess par ceux de l'elveur
    *
    * @param transaction
    * @throws TopiaException
    */
    void createAndFillAccessEleveur(TopiaContext transaction) throws TopiaException;

    /**
     * Cree tous les indicateurs pour les utilisateurs
     */
    void resetAllUserIndicateurs();

    /**
     * Send users stats notification
     */
    void sendStatNotification();

    /**
     * Send dangerous breeder notification
     * @param transaction
     */
    void sendFindDangerousBreeder(TopiaContext transaction);

    /**
     * Calcule tous les niveaux de securite des eleveurs
     *
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     * @param transaction
     */
    void updateAllWorstSecurityLevel(TopiaContext transaction) throws TopiaException, LogicException;

    /**
     * Recupere le nombre de tours par rapport a un ensemble de criteres
     *
     *
     * @param transaction
     * @param dateDebut           la date de debut du filtre
     * @param dateFin             la date de fin du filtre
     * @param planifie            drapeau planifie du filtre
     * @param termine             drapeau termine du filtre
     * @param topiaIdSociete      l'id de la societe du filtre
     * @param topiaIdTransporteur l'id du transporteur du filtre
     * @param topiaIdCamion       l'id du camion du filtre
     * @param topiaIdChauffeur    l'id du chauffeur du filtre
     * @return le nombre de tours (filtres ou non)
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     */
    int getToursCount(TopiaContext transaction, Date dateDebut, Date dateFin, boolean planifie,
                      boolean termine, String topiaIdSociete, String topiaIdTransporteur,
                      String topiaIdCamion, String topiaIdChauffeur) throws TopiaException, LogicException;

    /**
     * Recupere les tours par rapport a un ensemble de criteres
     *
     *
     * @param transaction
     * @param dateDebut           la date de debut du filtre
     * @param dateFin             la date de fin du filtre
     * @param planifie            drapeau planifie du filtre
     * @param termine             drapeau termine du filtre
     * @param topiaIdSociete      l'id de la societe du filtre
     * @param topiaIdTransporteur l'id du transporteur du filtre
     * @param topiaIdCamion       l'id du camion du filtre
     * @param topiaIdChauffeur    l'id du chauffeur du filtre
     * @return la liste des tours (filtres ou non)
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     */
    List<Tour> getTours(TopiaContext transaction, Date dateDebut, Date dateFin, boolean planifie,
                        boolean termine, String topiaIdSociete, String topiaIdTransporteur,
                        String topiaIdCamion, String topiaIdChauffeur) throws TopiaException, LogicException;

    /**
     * Recupere les tours par rapport a un ensemble de criteres, paginee et triee
     *
     *
     * @param transaction
     * @param dateDebut           la date de debut du filtre
     * @param dateFin             la date de fin du filtre
     * @param planifie            drapeau planifie du filtre
     * @param termine             drapeau termine du filtre
     * @param topiaIdSociete      l'id de la societe du filtre
     * @param topiaIdTransporteur l'id du transporteur du filtre
     * @param topiaIdCamion       l'id du camion du filtre
     * @param topiaIdChauffeur    l'id du chauffeur du filtre
     * @param startIndex          la position de la premiere societe a remonter
     * @param endIndex            la position de la derniere societe a remonter
     * @param sortCriterion       la propriete a trier
     * @param sortDirection       l'ordre du tri
     * @return la liste des tours (filtres ou non)
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     */
    List<Tour> getTours(TopiaContext transaction, Date dateDebut, Date dateFin, boolean planifie,
                        boolean termine, String topiaIdSociete, String topiaIdTransporteur,
                        String topiaIdCamion, String topiaIdChauffeur,
                        int startIndex, int endIndex, String sortCriterion, int sortDirection) throws TopiaException, LogicException;

    /**
     * Recupere une étape particulier a partir de son identifiant technique
     *
     *
     * @param transaction
     * @param topiaId l'id technique et unique du tour a recuperer
     * @return l etape en question
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique si le tour n'existe pas
     */
    Pair<Etape, SuiviEtape> getEtape(TopiaContext transaction, String topiaId) throws LogicException, TopiaException;

    /**
     * Recupere un tour particulier a partir de son identifiant technique
     *
     *
     * @param transaction
     * @param topiaId l'id technique et unique du tour a recuperer
     * @return le tour en question
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique si le tour n'existe pas
     */
    Tour getTour(TopiaContext transaction, String topiaId) throws TopiaException, LogicException;

    /**
     * Recupere tous les tours d'un camion
     *
     *
     * @param transaction
     * @param camionTopiaId TopiaId du camion concerne
     * @return dateFin      Date de de fin de l'intervale de navigation
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique si le tour n'existe pas
     */
    List<Tour> getToursByCamionAndDates(TopiaContext transaction, String camionTopiaId, Date dateDebut, Date dateFin) throws TopiaException;

    /**
     * Recupere les tours en chargement entre deux dates pour une usine.
     *
     *
     * @param transaction
     * @param dateDebut    la date de debut du filtre
     * @param dateFin      la date de fin du filtre
     * @param topiaIdUsine l'id de l'usine du filtre
     * @return la liste des tours en chargement
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     */
    List<Tour> getLoading(TopiaContext transaction, Date dateDebut, Date dateFin, String topiaIdUsine) throws TopiaException, LogicException;

    /**
     * Sauvegarde la disponibilite d'une ligne produit
     *
     *
     * @param transaction
     * @param topiaIdProduit l'id de la ligne produit
     * @param disponible     la disponibilite
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur la ligne produit n'existe pas
     */
    void saveProduitDisponible(TopiaContext transaction, String topiaIdProduit, boolean disponible) throws TopiaException, LogicException;

    /**
     * Recupere les produits d'un tour charge dans une usine
     *
     *
     * @param transaction
     * @param topiaIdUsine l'id de l'usine
     * @param topiaIdTour  l'id du tour
     * @return les produits
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     */
    List<LigneProduit> getProduitsLoading(TopiaContext transaction, String topiaIdUsine, String topiaIdTour) throws TopiaException, LogicException;

    /**
     * Recupere les societes
     *
     * @return la liste des societes
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     * @param transaction
     */
    List<Societe> getSocietes(TopiaContext transaction) throws TopiaException, LogicException;

    /**
     *
     * @param transaction
     * @param startIndex    la position de la premiere societe a remonter
     * @param endIndex      la position de la derniere societe a remonter
     * @param sortCriterion la propriete a trier
     * @param sortDirection l'ordre du tri
     * @return la liste des societes paginee et triee
     * @throws LogicException si erreur technique avec la base
     * @throws TopiaException si erreur logique TODO
     */
    List<Societe> getSocietes(TopiaContext transaction, int startIndex, int endIndex, String sortCriterion, int sortDirection) throws TopiaException, LogicException;


    /**
     * Recupere une societe
     *
     *
     * @param transaction
     * @param topiaId l'id technique unique de la societe recherchee
     * @return la societe demendee
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si la societe n'existe pas
     */
    Societe getSociete(TopiaContext transaction, String topiaId) throws TopiaException, LogicException;

    /**
     * Creer ou sauvegarder une societe
     *
     *
     * @param transaction
     * @param societe la societe a sauvegarder
     * @return la societe sauvegardee
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si la societe existe deja (code deja existant)
     */
    Societe saveSociete(TopiaContext transaction, Societe societe) throws TopiaException, LogicException;

    /**
     * Recupere les transporteurs
     *
     * @return la liste des transporteurs
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     * @param transaction
     */
    List<Transporteur> getTransporteurs(TopiaContext transaction) throws TopiaException, LogicException;

    /**
     *
     * @param transaction
     * @param startIndex    la position du premier transporteur a remonter
     * @param endIndex      la position du dernier transporteur a remonter
     * @param sortCriterion la propriete a trier
     * @param sortDirection l'ordre du tri
     * @return la liste des transporteurs paginee et triee
     * @throws LogicException si erreur technique avec la base
     * @throws TopiaException si erreur logique TODO
     */
    List<Transporteur> getTransporteurs(TopiaContext transaction, int startIndex, int endIndex, String sortCriterion, int sortDirection) throws TopiaException, LogicException;

    /**
     * Recupere un transporteur
     *
     *
     * @param transaction
     * @param topiaId l'id technique unique du transporteur recherche
     * @return le transporteur demande
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si le transporteur n'existe pas
     */
    Transporteur getTransporteur(TopiaContext transaction, String topiaId) throws TopiaException, LogicException;

    /**
     * Creer ou sauvegarder un transporteur.
     *
     *
     * @param transaction
     * @param transporteur le transporteur a sauvegarder
     * @return la transporteur sauvegarde
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si le transporteur existe deja (code deja existant)
     */
    Transporteur saveTransporteur(TopiaContext transaction, Transporteur transporteur) throws TopiaException, LogicException;

    /**
     * Recupere les usines
     *
     * @return la liste des usines
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     * @param transaction
     */
    List<Usine> getUsines(TopiaContext transaction) throws TopiaException, LogicException;

    /**
     *
     * @param transaction
     * @param startIndex    la position de la premiere usine a remonter
     * @param endIndex      la position de la derniere usine a remonter
     * @param sortCriterion la propriete a trier
     * @param sortDirection l'ordre du tri
     * @return la liste des usines paginee et triee
     * @throws LogicException si erreur technique avec la base
     * @throws TopiaException si erreur logique TODO
     */
    List<Usine> getUsines(TopiaContext transaction, int startIndex, int endIndex, String sortCriterion, int sortDirection) throws TopiaException, LogicException;

    /**
     * Recupere une usine
     *
     *
     * @param transaction
     * @param topiaId l'id technique unique de l'usine recherche
     * @return l'usine demende
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si l'usine n'existe pas
     */
    Usine getUsine(TopiaContext transaction, String topiaId) throws TopiaException, LogicException;

    /**
     * Creer ou sauvegarder une usine
     *
     *
     *
     * @param transaction
     * @param usine l'usine a sauvegarde
     * @return l'usine sauvegarde
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si l'usine existe deja (code deja existant)
     */
    Usine saveUsine(TopiaContext transaction, Usine usine) throws TopiaException, LogicException;

    /**
     * Recupere les chauffeurs
     *
     * @return la liste des chauffeurs
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     * @param transaction
     */
    List<Chauffeur> getChauffeurs(TopiaContext transaction) throws TopiaException, LogicException;

    /**
     *
     * @param transaction
     * @param startIndex    la position du premier chauffeur a remonter
     * @param endIndex      la position du dernier chauffeur a remonter
     * @param sortCriterion la propriete a trier
     * @param sortDirection l'ordre du tri
     * @return la liste des chauffeurs  paginee et triee
     * @throws LogicException si erreur technique avec la base
     * @throws TopiaException si erreur logique TODO
     */
    List<Chauffeur> getChauffeurs(TopiaContext transaction, int startIndex, int endIndex, String sortCriterion, int sortDirection) throws TopiaException, LogicException;

    /**
     * Recupere une chauffeur
     *
     *
     * @param transaction
     * @param topiaId l'id technique unique du chauffeur recherchee
     * @return le chauffeur demendee
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si le chauffeur n'existe pas
     */
    Chauffeur getChauffeur(TopiaContext transaction, String topiaId) throws TopiaException, LogicException;

    /**
     * Creer ou sauvegarder une usine
     *
     *
     * @param transaction
     * @param chauffeur le chauffeur a sauvegarder
     * @return le chauffeur sauvegarde
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si le chauffeur existe deja (code deja existant)
     */
    Chauffeur saveChauffeur(TopiaContext transaction, Chauffeur chauffeur) throws TopiaException, LogicException;

    /**
     * Ajout d'un camion
     *
     *
     * @param transaction
     * @param topiaIdChauffeur l'id du chauffeur
     * @param topiaIdCamion    l'id du camion
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     */
    void addCamionsToChauffeur(TopiaContext transaction, String topiaIdChauffeur, String... topiaIdCamion) throws TopiaException, LogicException;

    /**
     * Suppression d'un camion
     *
     *
     * @param transaction
     * @param topiaIdChauffeur l'id du chauffeur
     * @param topiaIdCamion    l'id du camion
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     */
    void deleteCamionsToChauffeur(TopiaContext transaction, String topiaIdChauffeur, String... topiaIdCamion) throws TopiaException, LogicException;

    /**
     * Marque le camion comme actif
     *
     *
     * @param transaction
     * @param topiaId of camion
     * @param actif if actif
     * @throws TopiaException si erreur thechnique avec la base
     * @return Camion modified
     */
    Camion setActifCamion(TopiaContext transaction, String topiaId, boolean actif) throws TopiaException;

    /**
     * Recupere les camions actifs
     *
     * @return la liste des camions
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     * @param transaction
     */
    List<Camion> getCamions(TopiaContext transaction) throws TopiaException, LogicException;

    /**
     * Recupere les camions
     *
     *
     * @param transaction
     * @param societe       societe concerne
     * @param actif if camion to display is actif
     * @return la liste des camions
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     */
    List<Camion> getCamions(TopiaContext transaction, String societe, boolean actif) throws TopiaException, LogicException;

    /**
     *
     * @param transaction
     * @param societe       societe concerne
     * @param actif         si les camions sont actifs
     * @param startIndex    la position du premier camion a remonter
     * @param endIndex      la position du dernier camion a remonter
     * @param sortCriterion la propriete a trier
     * @param sortDirection l'ordre du tri
     * @return la liste des camions paginee et triee
     * @throws LogicException si erreur technique avec la base
     * @throws TopiaException si erreur logique TODO
     */
    List<Camion> getCamions(TopiaContext transaction, String societe, boolean actif, int startIndex, int endIndex, String sortCriterion, int sortDirection) throws TopiaException, LogicException;

    /**
     * Recupere un camion a partir de son topiaId
     *
     *
     * @param transaction
     * @param topiaId l'id technique unique du camion recherchee
     * @return le camion demendee
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si le camion n'existe pas
     */
    Camion getCamion(TopiaContext transaction, String topiaId) throws TopiaException, LogicException;

    /**
     * Creer ou sauvegarder un camion
     *
     *
     * @param transaction
     * @param camion le camion a sauvegarder
     * @return le camion sauvegarde
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si le camion existe deja (code deja existant)
     */
    Camion saveCamion(TopiaContext transaction, Camion camion) throws TopiaException, LogicException;

    /**
     * Recuperer tous les camions princpaux
     *
     * @return la liste de tous les camions ptrinciapux
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si le camion existe deja (code deja existant)
     * @param transaction
     */
    List<Camion> getCamionsPrincipaux(TopiaContext transaction) throws TopiaException, LogicException;

    /**
     * Recuperer le camion principal d'un camion
     *
     *
     * @param transaction
     * @param camionId l'id du camion
     * @return le camion principal ou <code>null</code> si le camion est principal
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si le camion n'existe pas
     */
    Camion getCamionPrincipal(TopiaContext transaction, String camionId)
            throws TopiaException, LogicException;

    /**
     * Rattache un camion a son camion principal
     *
     *
     * @param transaction
     * @param camionId          l'id du camion
     * @param camionPrincipalId l'id du camion principal
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si les camions n'existent pas
     */
    void addCamionToCamionPrincipal(TopiaContext transaction, String camionId, String camionPrincipalId)
            throws LogicException, TopiaException;

    /**
     * Recupere les eleveurs
     *
     * @return la liste des eleveurs
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     * @param transaction
     */
    List<Eleveur> getEleveurs(TopiaContext transaction) throws TopiaException, LogicException;

    /**
     *
     * @param transaction
     * @param startIndex    la position du premier eleveur a remonter
     * @param endIndex      la position du dernier eleveur a remonter
     * @param sortCriterion la propriete a trier
     * @param sortDirection l'ordre du tri
     * @return la liste des eleveurs paginee et triee
     * @throws LogicException si erreur technique avec la base
     * @throws TopiaException si erreur logique TODO
     */
    List<Eleveur> getEleveurs(TopiaContext transaction, int startIndex, int endIndex, String sortCriterion, int sortDirection)
            throws TopiaException, LogicException;

    /**
     * Find tour with breeder criterias
     *
     *
     * @param transaction
     * @param dateLivraisonStart search with livraison date
     * @param dateLivraisonEnd search with livraison date
     * @param breederTopiaId id of eleveur
     * @param startIndex    la position du premier eleveur a remonter
     * @param endIndex      la position du dernier eleveur a remonter
     * @param sortCriterion la propriete a trier
     * @param sortDirection l'ordre du tri
     * @return la liste des eleveurs paginee et triee
     * @throws LogicException si erreur technique avec la base
     * @throws TopiaException si erreur logique TODO
     */
    List<Pair<Etape, SuiviEtape>> getEtapesByEleveurs(TopiaContext transaction,
                                  Date livraisonDateStart,
                                  Date livraisonDateEnd,
                                  String breederTopiaId,
                                  int startIndex,
                                  int endIndex,
                                  String sortCriterion,
                                  int sortDirection) throws TopiaException, LogicException;

    /**
     * Find tours with breeder criterias
     *
     *
     * @param transaction
     * @param dateLivraisonStart search with livraison date
     * @param dateLivraisonEnd search with livraison date
     * @param breederTopiaId id of eleveur
     * @return la liste des eleveurs paginee et triee
     * @throws LogicException si erreur technique avec la base
     * @throws TopiaException si erreur logique TODO
     */
    int getEtapesByEleveursCount(TopiaContext transaction,
                               Date livraisonDateStart,
                               Date livraisonDateEnd,
                               String breederTopiaId) throws TopiaException, LogicException;

    /**
     * Find breeder with criterias
     *
     *
     * @param transaction
     * @param query search on name and location
     * @param gpsKey search on gps state
     * @param nivSecu search on security level
     * @param camion search on camion
     * @param societe search on societe
     * @param dateLivraison search with livraison date
     * @param operator AND or OR for search beteen gps and nivSecu
     * @param startIndex    la position du premier eleveur a remonter
     * @param endIndex      la position du dernier eleveur a remonter
     * @param sortCriterion la propriete a trier
     * @param sortDirection l'ordre du tri
     * @return la liste des eleveurs paginee et triee
     * @throws LogicException si erreur technique avec la base
     * @throws TopiaException si erreur logique TODO
     */
    List<Eleveur> getEleveurs(TopiaContext transaction, String query, String gpsKey, String nivSecu, String camion, String societe,
                              Date dateLivraison, String operator, int startIndex, int endIndex, String sortCriterion, int sortDirection)
                              throws TopiaException, LogicException;

    /**
     * Find breeder with criterias
     *
     *
     * @param transaction
     * @param query search on name and location
     * @param gpsKey search on gps state
     * @param nivSecu search on security level
     * @param camion search on camion
     * @param societe search on societe
     * @param livraisonDate search with livraison date
     * @param operator AND or OR for search beteen gps and nivSecu
     * @return la liste des eleveurs paginee et triee
     * @throws LogicException si erreur technique avec la base
     * @throws TopiaException si erreur logique TODO
     */
    int getEleveursCount(TopiaContext transaction, String query, String gpsKey, String nivSecu, String camion, String societe, Date livraisonDate, String operator)
                              throws TopiaException, LogicException;

    /**
     * Creer ou sauvegarder les coordonnées GPS d'un eleveur
     *
     *
     * @param transaction
     * @param topiaId le topiaId de l'eleveur
     * @param point   le point GPS
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si l'eleveur n'existe pas
     */
    void saveEleveurGPS(TopiaContext transaction, String topiaId, Point point) throws TopiaException, LogicException;

    /**
     * Sauvegarder les modifications d'un eleveur
     *
     *
     * @param transaction
     * @param breeder concerne
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si l'eleveur n'existe pas
     */
    Eleveur saveEleveur(TopiaContext transaction, Eleveur breeder) throws TopiaException, LogicException;

    /**
     * Verouille ou deverouille le gps d'un eleveur
     *
     *
     * @param transaction
     * @param topiaId le topiaId de l'eleveur
     * @param lock    precise si le gps est verouillé
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si l'eleveur n'existe pas
     */
    void lockEleveurGPS(TopiaContext transaction, String topiaId, boolean lock) throws TopiaException, LogicException;

    /**
     * Recupere l'eleveur a partir du topia id
     *
     *
     * @param transaction
     * @param topiaId l'id de l'eleveur
     * @return l'eleveur
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     */
    Eleveur getEleveur(TopiaContext transaction, String topiaId) throws TopiaException, LogicException;

    /**
     * Recupere le message de bienvenue
     *
     * @return le message de bienvenue
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     * @param transaction
     */
    String getMessageBienvenue(TopiaContext transaction) throws TopiaException, LogicException;

    /**
     * Recupere le copyright
     *
     * @param transaction
     * @return le copyright
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     */
    String getCopyright(TopiaContext transaction) throws TopiaException, LogicException;

    /**
     * Creation ou modification d'un utilisateur
     *
     *
     *
     * @param transaction
     * @param user    l'utilisateur a sauvegarder
     * @param isWeb   le drapeau pour indiquer que l'utilisateur peut se connecter au web
     * @param isAdmin le drapeau pour indiquer que l'utilisatreur est administrateur du profil
     * @param profil  type du profil utilisateur
     * @return l'utilisateur sauvegarde
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     */
    TaasUser saveUser(TopiaContext transaction, TaasUser user, boolean isWeb, boolean isAdmin, String profil)
            throws TopiaException, LogicException;

    /**
     * Ajout d'un profil
     *
     *
     * @param transaction
     * @param topiaIdUser   l'id de l'utilisateur
     * @param topiaIdProfil l'id du profil
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     */
    void addProfilUser(TopiaContext transaction, String topiaIdUser, String topiaIdProfil) throws TopiaException, LogicException;

    /**
     * Suppression d'un profil
     *
     *
     * @param transaction
     * @param topiaIdUser   l'id de l'utilisateur
     * @param topiaIdProfil l'id du profil
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     */
    void deleteProfilUser(TopiaContext transaction, String topiaIdUser, String topiaIdProfil) throws TopiaException, LogicException;

    /**
     * Recupere le nombre d'utilisateurs
     *
     *
     * @param transaction
     * @param viewCamion visualisation des utilisateurs camions
     * @return le nombre d'utilisateurs
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     */
    int getUsersCount(TopiaContext transaction, boolean viewCamion) throws TopiaException, LogicException;

    /**
     * Recupere le nombre d'useur indicateurs utilisateurs
     *
     *
     * @param transaction
     * @param viewCamion visualisation des utilisateurs camions
     * @return le nombre d'utilisateurs
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     */
    @Transaction
    int getUserIndicateursCount(TopiaContext transaction, boolean viewCamion) throws TopiaException, LogicException;

    /**
     * Retourne la liste des utilisateurs
     *
     * @return la liste des utilisateurs
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     * @param transaction
     */
    List<TaasUser> getAllUsers(TopiaContext transaction) throws TopiaException, LogicException;

    /**
     * Retourne la liste des utilisateurs
     *
     * @param transactrion
     * @return la liste des indicateurs d'utilisateurs
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     */
    List<UserIndicateurs> getAllUserIndicateurs(TopiaContext transactrion) throws TopiaException, LogicException;

    /**
     * Retourne la liste des utilisateurs
     *
     *
     * @param transaction
     * @param isCamion visualisation exclusive des utilisateurs camions
     * @return la liste des utilisateurs
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     */
    List<TaasUser> getAllUsers(TopiaContext transaction, boolean isCamion) throws TopiaException, LogicException;

    /**
     * Retourne la liste des utilisateurs camions
     *
     * @return la liste des utilisateurs
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     * @param transaction
     */
    List<TaasUser> getAllCamionUser(TopiaContext transaction) throws TopiaException, LogicException;

    /**
     *
     * @param transaction
     * @param startIndex    la position du premier utilisateur a remonter
     * @param endIndex      la position du dernier utilisateur a remonter
     * @param sortCriterion la propriete a trier
     * @param sortDirection l'ordre du tri
     * @return la liste des utilisateurs paginee et triee
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     */
    List<TaasUser> getUsers(TopiaContext transaction, boolean viewCamion, int startIndex, int endIndex, String sortCriterion, int sortDirection) throws TopiaException, LogicException;

    /**
     *
     * @param transaction
     * @param startIndex    la position du premier utilisateur a remonter
     * @param endIndex      la position du dernier utilisateur a remonter
     * @param sortCriterion la propriete a trier
     * @param sortDirection l'ordre du tri
     * @return la liste des indicateur des utilisateurs paginee et triee
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     */
    List<UserIndicateurs> getUserIndicateurs(TopiaContext transaction, boolean viewCamion, int startIndex, int endIndex, String sortCriterion, int sortDirection) throws TopiaException, LogicException;

    /**
     * Recupere un utilisateur a partir de son identifiant technique
     *
     *
     * @param transaction
     * @param topiaId l'identifiant technique unique de l'utilisateur a recuperer
     * @return la liste des utilisateurs paginee et triee
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique si l'utilisateur n'existe pas
     */
    TaasUser getUser(TopiaContext transaction, String topiaId) throws TopiaException, LogicException;

    /**
     * Recupere un utilisateur a partir de son identifiant ou de celui de ses indicateurs
     *
     *
     * @param transaction
     * @param topiaId l'identifiant technique unique de l'utilisateur ou de l'indicateur a recuperer
     * @return la liste des utilisateurs paginee et triee
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique si l'utilisateur n'existe pas
     */
    TaasUser getSafeUser(TopiaContext transaction, String topiaId) throws TopiaException, LogicException;

    /**
     * Recupere les indicatuers de l'utilisateur a partir de son identifiant technique
     *
     *
     * @param transaction
     * @param topiaId l'identifiant technique unique de l'indicateurs de l'utilisateur a recuperer
     * @return la liste des utilisateurs paginee et triee
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique si l'utilisateur n'existe pas
     */
    UserIndicateurs getUserIndicateurs(TopiaContext transaction, String topiaId) throws TopiaException, LogicException;

    /**
     * Recupere l'utilisateur connecte
     *
     * @return l'utilisateur connnecte
     * @throws TopiaException si erreur technique avec la base
     */
    TaasUser getUser() throws TopiaException;

    /**
     * Modifi le mot de passe d'un utilisateur
     *
     *
     *
     * @param transaction
     * @param passwdBefore l'ancien password
     * @param passwdNew    le nouveau password
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     */
    void passwdUser(TopiaContext transaction, String passwdBefore, String passwdNew) throws TopiaException, LogicException;

    /**
     * Envoi par mail un nouveau mot de passe
     *
     *
     * @param email l'email du user
     * @param transaction
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     */
    void sendPasswdUser(TopiaContext transaction, String email) throws TopiaException, LogicException;

    /**
     * Retourne la liste des utilisateurs pour un principal donne
     *
     *
     * @param transaction
     * @param name nom du principal
     * @return la liste des utilisateurs pour le principal
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     */
    List<TaasUser> getUsersPrincipal(TopiaContext transaction, String name) throws TopiaException, LogicException;

    /**
     * //fixme On ne peut pas utiliser des InputStream sur des appel EJB
     * Recupere le fichier associe à la note
     *
     *
     * @param transaction
     * @param topiaId identifiant de la note
     * @return le fichier de la note
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si la note n'existe pas
     */
    InputStream getFileNote(TopiaContext transaction, String topiaId) throws TopiaException, LogicException;

    /**
     * //fixme On ne peut pas utiliser des InputStream sur des appel EJB
     * Sauvegarde ou cree une note
     *
     *
     * @param transaction
     * @param note        note a sauvegarder
     * @param inputStream fichier associe
     * @return note sauvegarde
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur de creation du fichier
     */
    Note saveNote(TopiaContext transaction, Note note, InputStream inputStream) throws TopiaException, LogicException;

    /**
     * Ajoute une note a un eleveur
     *
     *
     *
     * @param transaction
     * @param eleveurTopiaId identifiant de l'eleveur
     * @param noteTopiaId    identifiant de la note
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si l'eleveur ou la note n'existe pas
     */
    void addNote(TopiaContext transaction, String eleveurTopiaId, String noteTopiaId) throws TopiaException, LogicException;

    /**
     * Supprime la note et le fichier de la note
     *
     *
     * @param transaction
     * @param topiaId identifiant de la note
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si la note n'existe pas
     */
    void deleteNote(TopiaContext transaction, String topiaId) throws TopiaException, LogicException;

    /**
     * Recupere une note
     *
     *
     * @param transaction
     * @param topiaId identifiant de la note
     * @return note sauvegarde
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si la note n'existe pas
     */
    Note getNote(TopiaContext transaction, String topiaId) throws TopiaException, LogicException;

    /**
     * Recupere les labels
     *
     * @return la liste des labels
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si erreur logique TODO
     * @param transaction
     */
    List<Label> getLabels(TopiaContext transaction) throws TopiaException, LogicException;


    /**
     *
     * @param transaction
     * @param startIndex    la position du premier label a remonter
     * @param endIndex      la position du dernier label a remonter
     * @param sortCriterion la propriete a trier
     * @param sortDirection l'ordre du tri
     * @return la liste des labels paginee et triee
     * @throws LogicException si erreur technique avec la base
     * @throws TopiaException si erreur logique TODO
     */
    List<Label> getLabels(TopiaContext transaction, int startIndex, int endIndex, String sortCriterion, int sortDirection) throws TopiaException, LogicException;

    /**
     * Recupere une societe
     *
     *
     * @param transaction
     * @param topiaId l'id technique unique du label recherchee
     * @return le label demande
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si la label n'existe pas
     */
    Label getLabel(TopiaContext transaction, String topiaId) throws TopiaException, LogicException;

    /**
     * Creer ou sauvegarder un label
     *
     *
     * @param transaction
     * @param label la societe a sauvegarder
     * @return le label sauvegardee
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si la label existe deja (categorie et id deja existant en mode update)
     */
    Label saveLabel(TopiaContext transaction, Label label) throws TopiaException, LogicException;

    /**
     * Supprime un label de la base
     *
     *
     * @param transaction
     * @param topiaId l'id technique unique du label a supprimer
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si la label n'existe pas
     */
    void deleteLabel(TopiaContext transaction, String topiaId) throws TopiaException, LogicException;

    /**
     * Recupere une configuration de chargmenent usine a partir de son usine.
     * <p/>
     * Si la configuration n'existe pas, on en cree une nouvelle avec
     * le parametrage par defaut (nombreHeure=25,positionHeureCourant=40)
     *
     *
     * @param transaction
     * @param topiaId l'id technique unique de la configuration de chargmenent usine recherchee
     * @return la  configuration de chargmenent usine demande
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException ???
     */
    ChargementUsineConfig getChargementUsineConfigForUsine(TopiaContext transaction, String topiaId) throws TopiaException, LogicException;

    /**
     * Recupere une configuration de chargmenent usine
     *
     *
     * @param transaction
     * @param topiaId l'id technique unique de la configuration de chargmenent usine recherchee
     * @return la  configuration de chargmenent usine demande
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException ???
     */
    ChargementUsineConfig getChargementUsineConfig(TopiaContext transaction, String topiaId) throws TopiaException, LogicException;

    /**
     * Creer ou sauvegarder un configuration de chargmenent usine
     *
     *
     * @param transaction
     * @param chargementUsineConfig la configuration de chargmenent usine a sauvegarder
     * @return la configuration de chargmenent usine sauvegardee
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException ???
     */
    ChargementUsineConfig saveChargementUsineConfig(TopiaContext transaction, ChargementUsineConfig chargementUsineConfig) throws TopiaException, LogicException;

    /**
     * Recupere un eleveur
     *
     *
     * @param transaction
     * @param infoAccessTopiaId de l'infoAccess concerne
     * @return eleveur associe
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException ???
     */
    Eleveur findEleveurByAccesSilo(TopiaContext transaction, String infoAccessTopiaId) throws TopiaException, LogicException;

    /**
     * Recupere un InfoAccess
     *
     *
     * @param transaction
     * @param topiaId
     * @return info access cree
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException ???
     */
    InfoAccess getInfoAccess(TopiaContext transaction, String topiaId) throws TopiaException, LogicException;

    /**
     * Sauvegarde ou cree l'access silo
     *
     *
     * @param transaction
     * @param infoAccess concerne
     * @return info access cree
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException ???
     */
    InfoAccess saveInfoAccess(TopiaContext transaction, InfoAccess infoAccess, String breederTopiaID) throws TopiaException, LogicException;

    /**
     * Suppression d'un access silo
     *
     *
     * @param transaction
     * @param infoAccessTopiaId de l'acces a supprime
     * @return topiaId de l'info access cree
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException ???
     */
    void deleteAccesSilo(TopiaContext transaction, String infoAccessTopiaId) throws TopiaException, LogicException;

    /**
     * Lock ou unlock l'access silo concerne, si il est locker on le delock et vice versa
     *
     *
     * @param transaction
     * @param infoAccessTopiaId a modifier
     * @return topiaId de l'info access cree
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException ???
     */
    void lockOrUnlockInfoAccess(TopiaContext transaction, String infoAccessTopiaId) throws TopiaException, LogicException;

    /**
     * Find all duplicate infoAcces and delete them arbitrarily
     *
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException ???
     * @param transaction
     */
    void findAndDeleteDuplicateInfoAccess(TopiaContext transaction) throws TopiaException, LogicException;

    /**
     * Retourn le suivi ligne produit du produit en parametre
     *
     * @param transaction
     * @param product concerne
     * @return SuiviLigneProduit associé
     * @throws TopiaException si erreur technique avec la base
     */
    SuiviLigneProduit getSuiviProduit(TopiaContext transaction, LigneProduit product) throws TopiaException;

    /**
     * Sauvegarde l'action en parametre
     *
     *
     * @param transaction
     * @param action de securite to save
     * @param eleveurId associe
     * @param siloId associe
     * @throws TopiaException si erreur technique avec la base
     */
    ActionSecurite saveActionSecurite(TopiaContext transaction, ActionSecurite action, String eleveurId, String siloId) throws TopiaException;

    /**
     * Supprime l'action de securite en parametre
     *
     *
     * @param transaction
     * @param topiaId de l'action a supprimmer
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si l'action relative au topiaId n'existe pas
     */
    void deleteActionSecurite(TopiaContext transaction, String topiaId) throws TopiaException, LogicException;

    /**
     * Retourne l'action de securite correspondant a l'id mis en parametre
     *
     *
     * @param transaction
     * @param topiaId de l'action
     * @return l'action de securite
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si l'action relative au topiaId n'existe pas
     */
    ActionSecurite getActionSecurite(TopiaContext transaction, String topiaId) throws TopiaException, LogicException;

    /**
     * Retourne la societe de l'utilisateur passe en parametre
     *
     *
     * @param transaction
     * @param user concrened
     * @return societe of user
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si l'action relative au topiaId n'existe pas
     */
    Societe getUserSociete(TopiaContext transaction, TaasUser user) throws TopiaException, LogicException;

    /**
     * Envoie une notification et garde les utilisateurs qui ont fait la demande
     * pour les notifier par la suite lors du deverrouillage
     *
     *
     * @param transaction
     * @param topiaId de l'eleveur
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si l'action relative au topiaId n'existe pas
     */
    void requestToUnlockBreeder(TopiaContext transaction, String topiaId) throws TopiaException, LogicException;

    /**
     * Envoie une notification et garde les utilisateurs qui ont fait la demande
     * pour les notifier par la suite lors du deverrouillage
     *
     * @param transaction
     * @param topiaId de l'info access
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si l'action relative au topiaId n'existe pas
     */
    void requestToUnlockInfoAcces(TopiaContext transaction, Eleveur eleveur, String topiaId) throws TopiaException, LogicException;

    /**
     * Enregistre l'eleveur comme etant dupliquer pour IBU et merge les 2 eleveurs
     *
     * @param transaction
     * @param eleveurId eleveur to merge
     * @param eleveurDuplicatedId eleveur to delete
     * @return eleveur eleveur merged updated
     * @throws TopiaException si erreur technique avec la base
     * @throws LogicException si l'action relative au topiaId n'existe pas
     */
    public Eleveur registerBreederAsDuplicated(TopiaContext transaction, String eleveurId, String eleveurDuplicatedId) throws TopiaException, LogicException;
} //ServiceWeb
